   ; GAUSSIAN_NOISE_STD: estimate the standard deviation of the noise in
   ; a given image, assuming a gaussian probability distribution.
   ; Normally distributed noise includes for instance read-out-noise,
   ; but also the photon (Poisson) noise associated to nearly spatially
   ; uniform signals contributing to the image (e.g. dark current, smooth
   ; nebulosities in the sky, etc.). A Poisson-distributed 2-D signal may
   ; be considered as distributed according to a unique gaussian
   ; probability law if:
   ; 1) the signal is not too faint
   ; 2) the signal is approximately spatially uniform
   ; If the image contains both gaussian distributed and photon noise due
   ; to non-smooth sources (e.g. stars), the noise standard deviation is
   ; of course over-estimated.

   ; ALGORITHM DESCRIPTION: remove median smoothed image, to subtract
   ; the contribution of stars and background nebulosities. The residual
   ; image should contain just noise; it is clipped, considering only
   ; those pixels within a suitable range from their median value. Then
   ; the histogram is computed and fit with a gaussian, whose standard
   ; deviation approximates the gaussian noise standard deviation.
   ; Notice that the median subtraction may leave a non-negligible
   ; amount of pixels with a very small positive value: the 0-bin of
   ; the histogram may not be reliable, and thus it is excluded from the
   ; gaussian fit.

   ; INPUT
   ;	data: image
   ;	[PATCH = ]: patch size to compute median smoothed image
   ;		(default = 3)
   ;	[/NO_SUB]: avoid median subtraction; the histogram may be
   ;		asymmetric, due to the contribution of sources
   ;	[N_STD = ]: relative threshold to clip residual image, expressed
   ;		in units of standard deviation from the median value of the
   ;		residual
   ; OUTPUT
   ;	i: intensity values in the median-removed image
   ;	h: histogram of median-removed image
   ;	hfit: gaussian fit to the histogram
   ; NOTE 1: the fitted value corresponding to the 0-bin is a replica
   ;	of the left closest value (see ALGORITHM DESCRIPTION)
   ; NOTE 2: display the histogram and the best fit gaussian as follows:
   ;	IDL> plot, i, h, PSYM = 10  &  oplot, i, hfit, LINESTYLE = 1




   ; LEFT_HWHM: find left-tail HFHM of an histogram.

   FUNCTION left_hfhm, h

	s = reverse( sort( h ) )  &  threshold = half * h[s[1]]
	l = s[1]  &  while  h[l] gt threshold and l gt 0  do  l = l - 1
	l = l > 0
	return, s[1] - l + 1
   end

   ; OPT_HISTO: auxiliary procedure. Compute the optimal histogram by
   ; adjusting iteratively the bin size so that the (left) HWHM of the
   ; histogram itself contains at least a minimum number of bins.

   PRO opt_histo, data, x, h

	half = 0.5		; relative cut to find the histogram HWHM (obviously 0.5)
	min_n_half = 5	; min no. of bins in histogram HWHM
	max_nbin = 1000	; max total no. of histogram bins (error condition)
	range = float( max( [ abs(min(data)), abs(max(data)) ] ) )
	dmin = -range  &  dmax = +range  &  range = 2*range  &  bin = range
	; compute the histogram, adjusting iteratively the bin size
	repeat begin
	   ; reduce bin size and range
	   range = range / 2
	   dmin = -range  &  dmax = +range
	   bin = bin / 2  &  nbin = range / bin
	   ; compute histogram
	   histo, data, dmin, dmax, bin, x, h
	   ; find left HWHM of histogram (starting from 2nd most populated bin...)
	   s = reverse( sort( h ) )  &  threshold = half * h[s[1]]
	   l = s[1]  &  while  h[l] gt threshold and l gt 0  do  l = l - 1
	   l = l > 0  &  n_half = s[1] - l + 1
	endrep until  n_half gt min_n_half or nbin ge max_nbin
;	if  nbin ge max_nbin  then  h = -1	; error
	return
   end




   PRO gaussian_noise_std, data, PATCH = patch, $
   						   N_STD = n_std, mode, std, i, h, hfit

	; remove median
	if  n_elements( patch ) eq 0  then  patch = 3
	residual = data - median_filter( data, patch )
	; extract a sample of uniformly distributed points
	npoints = 2048 < n_elements( data )
	s = float( mysize( data, /DIM ) )
	nx = round( sqrt( s[0]/s[1] * npoints ) )
	ny = round( sqrt( s[1]/s[0] * npoints ) )
	x = round( sampling_grid( nx, s[0] / nx ) )
	y = round( sampling_grid( ny, s[1] / ny ) )
	x = x # ( lonarr( ny ) + 1 )  &  y = ( lonarr( nx ) + 1 ) #  y
	p = residual[ x, y ]
	; compute histogram with suitable bin size
	opt_histo, p, i, h
	; iterative gaussian fit
;	w0 = ( where( i ge 0 ) )[0]  &  i = i[0:w0-1]  &  h = h[0:w0-1]
	maxit = 50  &  tol = 0.01
	it = 0  &  converging = 0B
	m = max( h, mode )
	std = max( [ i[mode]-min(i), max(i)-i[mode] ] )
	while  not converging and it lt maxit  do begin
	   w0 = ( where( i ge 0 ) )[0]  &  i0 = i[w0]
	   w = where( i le mode + std and i ne i0 )
	   hfit = gaussfit( i[w], h[w], NTERMS = 3, c )
	   std0 = std  &  mode = c[1]  &  std = abs( c[2] )
	   converging = abs( ( std - std0 ) / std0 ) lt tol  &  it = it + 1
	endwhile
	if  it eq maxit and not converging  then  std = -1
	return
   end