   ; GAUSSIAN_NOISE_STD: estimate the standard deviation of the noise in
   ; a given image, assuming a gaussian probability distribution.
   ; Normally distributed noise includes for instance read-out-noise,
   ; but also the photon (Poisson) noise associated to nearly spatially
   ; uniform signals contributing to the image (e.g. dark current, smooth
   ; nebulosities in the sky, etc.). A Poisson-distributed 2-D signal may
   ; be considered as distributed according to a unique gaussian
   ; probability law if:
   ; 1) the signal is not too faint
   ; 2) the signal is approximately spatially uniform
   ; If the image contains both gaussian distributed and photon noise due
   ; to non-smooth sources (e.g. stars), the noise standard deviation is
   ; of course over-estimated.

   ; ALGORITHM DESCRIPTION: remove median smoothed image, to subtract
   ; the contribution of stars and background nebulosities. The residual
   ; image should contain just noise; it is clipped, considering only
   ; those pixels within a suitable range from their median value. Then
   ; the histogram is computed and fit with a gaussian, whose standard
   ; deviation approximates the gaussian noise standard deviation.
   ; Notice that the median subtraction may leave a non-negligible
   ; amount of pixels with a very small positive value: the 0-bin of
   ; the histogram may not be reliable, and thus it is excluded from the
   ; gaussian fit.

   ; INPUT
   ;	data: image
   ;	[PATCH = ]: patch size to compute median smoothed image
   ;		(default = 3)
   ;	[/NO_SUB]: avoid median subtraction; the histogram may be
   ;		asymmetric, due to the contribution of sources
   ;	[N_STD = ]: relative threshold to clip residual image, expressed
   ;		in units of standard deviation from the median value of the
   ;		residual
   ; OUTPUT
   ;	i: intensity values in the median-removed image
   ;	h: histogram of median-removed image
   ;	hfit: gaussian fit to the histogram
   ; NOTE 1: the fitted value corresponding to the 0-bin is a replica
   ;	of the left closest value (see ALGORITHM DESCRIPTION)
   ; NOTE 2: display the histogram and the best fit gaussian as follows:
   ;	IDL> plot, i, h, PSYM = 10  &  oplot, i, hfit, LINESTYLE = 1




   ; OPT_HISTO: auxiliary procedure. Compute the optimal histogram by
   ; adjusting iteratively the bin size so that the (left) HWHM of the
   ; histogram itself contains at least a minimum number of bins.

   PRO opt_histo, data, x, h

	half = 0.5		; relative cut to find the histogram HWHM (obviously 0.5)
	min_n_half = 5	; min no. of bins in histogram HWHM
	max_nbin = 1000	; max total no. of histogram bins (error condition)
	range = float( max( [ abs(min(data)), abs(max(data)) ] ) )
	dmin = -range  &  dmax = +range  &  range = 2*range  &  bin = range
	; compute the histogram, adjusting iteratively the bin size
	repeat begin
	   ; reduce bin size
	   bin = bin / 2  &  nbin = range / bin
	   ; compute histogram
	   histo, data, dmin, dmax, bin, x, h
	   ; find left HWHM of histogram (starting from 2nd most populated bin...)
	   s = reverse( sort( h ) )  &  threshold = half * h[s[1]]
	   l = s[1]  &  while  h[l] gt threshold and l gt 0  do  l = l - 1
	   l = l > 0  &  n_half = s[1] - l + 1
	endrep until  n_half gt min_n_half or nbin ge max_nbin
;	if  nbin ge max_nbin  then  h = -1	; error
	return
   end



   FUNCTION gaussian_noise_std, data, PATCH = patch, NO_SUB = no_sub, $
   								N_STD = n_std, i, h, hfit

	; remove median
	if  n_elements( patch ) eq 0  then  patch = 3
	if  not keyword_set( no_sub )  then $
	   residual = data - median_filter( data, patch )  else $
	   residual = data
	; clip data
	if  n_elements( n_std ) eq 0  then  n_std = 3
	med = median( residual )
	range = n_std * sqrt( ( moment( residual - med ) )[1] )
	w = where( residual gt med - range and residual lt med + range, count )
	if  count eq 0  then  return, -1
	residual = residual[w]
	; compute histogram with suitable bin size
	opt_histo, residual, i, h
;	if  h[0] lt 0  then  return, -1
	; gaussian fit to histogram, excluding 0-bin
	if  not keyword_set( no_sub )  then begin
	   w0 = where( i ge 0 )  &  w0 = w0[0]
	   w = where( lindgen( n_elements( i ) ) ne w0 )
	endif else  w = lindgen( n_elements( h ) )
	hfit = gaussfit( i[w], h[w], NTERMS = 3, c )
	; insert dummy value in hfit for visualization purposes
	if  not keyword_set( no_sub )  then begin
	   aux = hfit
	   hfit = hfit[0:w0-1]  &  hfit = add_elements( hfit, hfit[w0-1] )
	   hfit = add_elements( hfit, aux[w0:n_elements(aux)-1] )
	endif
	return, c[2]
   end