   ; GINV: Compute the generalized inverse (in the Moore - Penrose sense)
   ;	of a 2-D array by Gram-Schmidt orthogonalization of its columns.
   ;	The computation of the inverse is performed in double precision;
   ;	if the input array is not in double precision, the result is in
   ;	single precision.
   ;	Adapted in IDL from the FORTRAN version written by S. Lorenzutta.

   ; INPUT
   ;	a: 2-D array to be inverted
   ; OUTPUT
   ;	Return generalized inverse of a
   ;	[RANK = ]: rank of a (no. of linearly independent columns)

   ; NOTE: The columns of a are tested for linear dependence.
   ;	The tolerance for this test is  2 * zero( /DOUBLE_P ).





   ; GS_ORTH: recursive Gram-Schmidt orthogonalization of a set of n
   ;	m-components vectors. The vectors are stored as a (m*n) array,
   ;	to speed up the computation if virtual memory should be used.

   PRO gs_orth, v, u, lin_indep, n, m, tol

	this = n - 1
	if  this gt 0  then begin
	   ; Induction case: apply the G-S orthogonalization to this vector.
	   ; Orthogonalize the first (n - 1) vectors before
	   gs_orth, v, u, lin_indep, n - 1, m, tol
	   norm = total( v[*,this] * v[*,this] )
	   prod = ( v[*,this] # v )[0:this-1]
	   v[*,this] = v[*,this] - ( prod * lin_indep[0:this-1] ) # $
	   						   transpose( v[*,0:this-1] )
	   u[*,this] = u[*,this] - prod # transpose( u[*,0:this-1] )
	   new_norm = total( v[*,this] * v[*,this] )
	   is_lin_indep = norm ne 0
	   if  is_lin_indep  then  is_lin_indep = new_norm / norm gt tol
	   if  is_lin_indep  then begin
	      ; this vector is linearly independent
	      lin_indep[this] = 1  &  norm = 1 / sqrt( new_norm )
	   endif else begin
		  ; this vector is linearly dependent
	      prod = u[*,this] # u[*,0:this-1]
		  v[*,this] = - ( prod * lin_indep[0:this-1] ) # $
		  				transpose( v[*,0:this-1] )
	      lin_indep[this] = 0
	      norm = 1 / sqrt( total( u[*,this] * u[*,this] ) )
	   endelse
	endif else begin
	   ; Base case: one vector to be normalized
	   norm = sqrt( total( v[*,this] * v[*,this] ) )
	   if  norm ne 0  then begin
	      norm = 1 / norm  &  lin_indep[this] = 1
	   endif else  lin_indep[this] = 0
	endelse
	v[*,this] = v[*,this] * norm  &  u[*,this] = u[*,this] * norm
	return
   end


   FUNCTION ginv, a, RANK = rank

	if  mysize( a, /N_DIM ) ne 2  then  return, 0
	s = mysize( a, /DIM )  &  n = s[0]  &  m = s[1]
	type_a = mysize( a, /TYPE )
	a_inv = transpose( double( a ) )	; transpose the set of column
						; vectors to speed up the computation if virtual
						; memory should be used
	u = dblarr( n, n )  &  diag = lindgen( n )  &  u[diag,diag] = 1
	lin_indep = lonarr( n )  &  tol = 2 * zero( /DOUBLE_P )
	gs_orth, a_inv, u, lin_indep, n, m, tol
	a_inv = temporary( a_inv ) # transpose( u )
	if  type_a lt 5  then  a_inv = float( a_inv )
	rank = total( lin_indep )
	return, a_inv
   end
