   ; IMAGE_BACKGROUND: estimate the background emission in a 2-D image.

   ; ALGORITHM DESCRIPTION
   ; The image is partitioned into sub-frames defined by a rectangular
   ; grid. The mesh points coincide with the centers of the corresponding
   ; regions. For each sub-image a background estimate is obtained as the
   ; local minimum (there is no reason to think of something smaller than
   ; the background ..). This background estimate is associated to the
   ; mesh point corresponding to sub-image under examination. The sampled
   ; version of the background is smoothed by median filtering. Finally a
   ; global estimate is obtained by means of spline interpolation of the
   ; measured values on a grid having the same size as the image. Bicubic
   ; splines are used by default: to use bilinear splines, set DEGREE = 1.

   ; INPUT
   ;	image
   ;	dx, dy: integer x- and y- grid spacing for background sampling
   ;		(in pixel units)
   ;	[MED_BOX = ]: size of box for median smoothing of sampled points
   ;		(default = ( no. of grid points ) / 10).
   ;		Minimum box size allowed = 3
   ;	[DEGREE = ]: spline degree (default = 3)

   ; OUTPUT
   ;	Return a background estimate, having the same size as the image
   ;	If an error occurs during spline interpolation, return an array
   ;	of 0s.

   ; NOTE: the local estimate of the sky is estimated with SKY, not with
   ;	   the local minimum


   FUNCTION image_background, image, dx, dy, MED_BOX = med_box, $
   							  _EXTRA = extra

	if  mysize( image, /N_DIM ) ne 2  then  return, 0
	s = mysize( image, /DIM )  &  sx = s[0]  &  sy = s[1]
	if  n_params() eq 2  then  dy = dx
	idx = round( dx )  &  idy = round( dy )
	; Pad the image with 0s
	sx = ( sx / idx + 1 ) * idx  &  sy = ( sy / idy + 1 ) * idy
	ima = extend_array( image, sx, sy )
	; Define the sampling grid
	nx = sx / idx  &  ny = sy / idy
	x = sampling_grid( nx, idx, lx0, ux0 )  &  ux0 = ux0 < sx
	y = sampling_grid( ny, idy, ly0, uy0 )  &  uy0 = uy0 < sy
	lx = round( x - idx / 2. ) > 0  &  ux = round( x + idx / 2. ) < ( sx - 1 )
	ly = round( y - idy / 2. ) > 0  &  uy = round( y + idy / 2. ) < ( sy - 1 )
	; Background sampling
	b = fltarr( nx, ny )
	for  i = 0, ny - 1  do  for  j = 0, nx - 1  do begin
	   sky, ima[ lx[j]:ux[j],ly[i]:uy[i] ], lb, /SILENT
	   b[j,i] = lb
	endfor
;	for  i = 0, ny - 1  do  for  j = 0, nx - 1  do $
;	   b[j,i] = min( ima[ lx[j]:ux[j],ly[i]:uy[i] ] )
	; Sampled background smoothing
	if  n_elements( med_box ) eq 0  then $
	   box = fix( min( [nx, ny] ) / 10. )  else  box = med_box
	box = box > 3
	b = median_filter( b, box )
	; Background interpolation
	b = spline_magnify( b, idx, idy, error )
	if  error  then  b = ima - ima
	return, b[0:s[0]-1,0:s[1]-1]
   end
