   ; IMAGE_SHIFT: fractional shift of a 2-D image by interpolation.

   ; INPUT
   ;	image
   ;	x_shift, y_shift: fractional shifts (<= 0.5 pixel) along the
   ;		coordinate axes
   ;	[INTERP_TYPE = ]: type of interpolating technique for fractional
   ;		image shift. Possible choices are: 'FT' (shift by Fourier
   ;		Transform, default) and 'SPLINE*', for * = 1, 3, 5 (shift by
   ;		spline interpolation)
   ;	[/NO_EXTEND]: the image is extended by 2 pixels in each direction
   ;		to avoid overlapping errors in the shift. Extending the array
   ;		may be useful when the image support extension concides with
   ;		the array size (i.e. there are no 0s at the image border).
   ;		Set /NO_EXTEND to prevent extending the data array
   ;	[/ANY_SHIFT]: if the fractional shift exceeds 1/2 pixel, the input
   ;		image is returned. /ANY_SHIFT is to avoid this restriction, at
   ;		the expense of some edge effects in the shifted image ...

   ; INPUT / OUTPUT
   ;	[AUX_DATA = ]: auxiliary data for FT or spline interpolation, to
   ;		be used as input/output keyword when IMAGE_SHIFT is iteratively
   ;		called by some other IDL routine (e.g. CENTROIDER)

   ; OUTPUT
   ;	Return shifted image


   FUNCTION image_shift, image, x_shift, y_shift, INTERP_TYPE = interp_type, $
   						 NO_EXTEND = no_extend, ANY_SHIFT = any_shift,		 $
   						 AUX_DATA = aux_data

	if  mysize( image, /N_DIM ) ne 2  then  return, image
	if  x_shift eq 0 and y_shift eq 0  then  return, image
	if  ( abs( x_shift ) gt 0.5 or abs( y_shift ) gt 0.5 ) and $
		not keyword_set( any_shift )  then  return, image
	if  n_elements( interp_type ) eq 0  then  interp_type = 'FT'
	interp_type = strupcase( interp_type )
	if  strmid( interp_type, 0, 1 ) eq 'S'  then  interp = 'SPLINE'	else $
	if  strmid( interp_type, 0, 1 ) eq 'M'  then  interp = 'MIXED'  else $
												  interp = 'FT'
	if  interp eq 'SPLINE' and strlen( interp_type ) eq 7  then $
	   degree = fix( strmid( interp_type, 6, 1 ) )  else  degree = 3
	i = image  &  s = mysize( i, /DIM )  &  sx = s[0]  &  sy = s[1]
	no_data = n_elements( aux_data ) eq 0
	if  n_elements( aux_data ) ne 0  then  $
	   no_data = n_tags( aux_data ) eq 0
	if  not keyword_set( no_extend )  then begin
	   xo = 1  &  yo = 1
	   sx = sx + 2 * xo  &  sy = sy + 2 * yo
	   if  no_data  then $
	      i = extend_array( image, sx, sy, X_OFF = xo, Y_OFF = xo )
	endif

	case  interp  of

	   'SPLINE': begin
	   			 if  no_data  then begin
	   			    spline_coeff, i, DEGREE = degree, $
	   			    			  coefficients, x_knots, y_knots, x, y
	   			    aux_data = { coefficients: coefficients, $
	   			    			 x_knots: x_knots, y_knots: y_knots, $
	   			    			 x: x, y: y, degree: degree }
	   			 endif else begin
	   			    coefficients = aux_data.coefficients
	   				x_knots = aux_data.x_knots  &  y_knots = aux_data.y_knots
	   				x = aux_data.x 				&  y = aux_data.y
	   				degree = aux_data.degree
	   			 endelse
	   			 xs = x - x_shift  &  ys = y - y_shift
	   			 shifted_image = spline_interp( $
	   			 					coefficients, x_knots, y_knots, $
	   			 					DEGREE = degree, xs, ys )
	   			 end

	   'FT':	 begin
	   			 if  no_data  then begin
	   				image_ft = fft( i )  &  aux_data = { image_ft: image_ft }
	   			 endif else  image_ft = aux_data.image_ft
	   			 phi = phase( x_shift, y_shift, sx, sy )
	   			 shifted_image = float( fft( image_ft * phi, /INVERSE ) )
	   			 end

	   'MIXED':  begin
	   			 if  no_data  then begin
	   			    model = gauss2dfit( i, mod_coeff )
	   			    model = model - mod_coeff[0]  &  residual = i - model
	   			    spline_coeff, residual, DEGREE = degree, $
	   			    			  res_coeff, x_knots, y_knots, x, y
	   			    aux_data = { mod_coeff: mod_coeff, res_coeff: res_coeff, $
	   			    			 x_knots: x_knots, y_knots: y_knots, $
	   			    			 x: x, y: y, degree: degree }
	   			 endif else begin
	   			    mod_coeff = aux_data.mod_coeff
	   			    res_coeff = aux_data.res_coeff
	   				x_knots = aux_data.x_knots  &  y_knots = aux_data.y_knots
	   				x = aux_data.x 				&  y = aux_data.y
	   				degree = aux_data.degree
	   			 endelse
	   			 xs = x - x_shift  &  ys = y - y_shift
	   			 shifted_model = mod_coeff[1] * ellipt_gaussian( sx, sy, $
	   			 				 mod_coeff[4] + x_shift, mod_coeff[5] + y_shift, 0, $
	   			 				 mod_coeff[2], mod_coeff[3] )
	   			 shifted_residual = spline_interp( $
	   			 					res_coeff, x_knots, y_knots, $
	   			 					DEGREE = degree, xs, ys )
	   			 shifted_image = shifted_model + shifted_residual
				 end

	endcase

	if  not keyword_set( no_extend )  then begin
	   sx = sx - 2 * xo  &  sy = sy - 2 * yo
	   shifted_image = shifted_image[xo:xo+sx-1,yo:yo+sy-1]
	endif
	return, shifted_image
   end
