   ; IPP: estimate isoplanatic patch parameters (IPP's).

   ; ALGORITHM DESCRIPTION: the goal is to fit a set of differently
   ; elongated stellar images with a model given by the convolution of
   ; the guide star (GS) and an elliptical gaussian pointing at the GS
   ; itself. The standard deviations of the gaussian are the IPP's.
   ; The guide star and the stars to be fit are selected by the user
   ; with a mouse click.
   ; If the guide star is not available (e.g. it is outside the frame or
   ; it is saturated (sigh!)), it is necessary to approximate it by
   ; combining the images of some stars around it (and as close as possible).
   ; The combination procedure is very similar to the PSF extraction strategy.
   ; In this case, the reference position is that of the guide star, as
   ; if it were available.
   ; Before fitting off-axis stars, a background estimate is subtracted from
   ; the stellar field image.
   ; The measured IPP's, as a function of the radial distance r, are then
   ; fit by means of two distinct polynomial laws (with no additive constant,
   ; since elongation and width at the guide star location must be = 0 by
   ; definition). Before computing the best fit polynomial, bad measurements
   ; are rejected (see ACCEPT_IPP).

   ; NOTE on the GUIDE STAR: the user is asked to click on the GS if its
   ; coordinates have not been passed with the input keywords X_GS, Y_GS.
   ; If the user selects more than 1 star, the program assumes that the true
   ; GS is not availble and must be replaced by a combination of the stars
   ; just selected by the user for the purpose. But the position of the true
   ; GS is required anyway: it must be passed in this case with the input
   ; keywords X_POS_GS, Y_POS_GS. If more than 1 GS star have been selected or
   ; passed with X_GS, Y_GS and at the same time the keywords X_POS_GS, Y_POS_GS
   ; are not used, the program will stop.

   ; INPUT
   ;	image: stellar field
   ;	[DISPLAY =]: properly scaled version of image for visualization
   ;		(default = image)
   ;	[WSIZE = ]: size of window to display image (default = image size)
   ;	[WNUM = ]: window number for image display (default = 0)
   ;	[NFWHM_BKG = ]: box size for background estimation in units of
   ;		GS FWHM (default = 5)
   ;	[NFWHM_FIT]: size of fitting box for IPP's estimation, expressed in
   ;		units of FWHM of the stellar image being fit
   ;	[_EXTRA = ]: optional input parameters of CLICK_ON_MAX,
   ;		ESTIMATE_BACKGROUND, SUPERPOSE_STARS, FIT_KERNEL
   ;	[MAX_LW_RATIO = ]: maximum acceptable ratio  l(r)/w(r), to reject
   ;		bad measurements (default = 2*sqrt(3))
   ;	[POLY_DEG = ]: degree of polynomial law to fit (r, l) and (r, w)
   ;		(default = 2). NOTE: the costant term is always excluded
   ;	[FILENAME = ]: name of file to print results
   ;	[/GRAPH]: display measured IPP's and polynomial fit

   ; INPUT/OUTPUT
   ;	[X_GS = , Y_GS = ]: position of the GS (or stars around it to form
   ;		an approximation of the GS)
   ;	[X_POS_GS = , Y_POS_GS = ]: position of true GS. It MUST be passed when
   ;		more than 1 star is either selected or passed with X_GS, Y_GS
   ;	[X_STAR = , Y_STAR = ]: coordinates of IPP's stars

   ; OUTPUT
   ;	x, y: coordinates of fitted stellar images
   ;	r: radial distances from GS
   ;	a: position angle (rad.) of fitted stars
   ;	l, w: measured elongation and width
   ;	s: scaling factors for best fit models (see FIT_KERNEL)
   ;	e: normalized least square errors (see FIT_KERNEL)
   ;	guide_star: image of guide star used in the fit
   ;	accepted_stars: stack of successfully fitted stars
   ;	best_fit_models: stack of best fit models
   ;	l_coeff, w_coeff: coefficients of l(r) and w(r) polynomials
   ;	l_fit, w_fit: fitted values for l(r) and w(r)



   ; IPP_PRINT: print results to ASCII file.

   PRO ipp_print, filename, x0, y0, gs_fwhm, x, y, r, a, fw_hm, l, w, s, e

	get_lun, u
	openw, u, filename
	printf, u, 'guide star at  (', x0, ',', y0, ' )', FORMAT = '(A, I5, A, I5, A)'
	printf, u, 'guide star approximate FWHM = ', gs_fwhm, FORMAT = '(A, F5.2)'
	printf, u, ''  &  printf, u, ''
	printf, u, 'star ID   FWHM        x     y       r       a       l       w       s       e'
	printf, u, ''
	nstar = n_elements(e)
	for  n = 0, nstar - 1 do $
	printf, u, n+1, fw_hm[n], x[n], y[n], r[n], 180./!pi*a[n], l[n], w[n], s[n], e[n], $
	FORMAT = '(I4, 5X, F5.2, 6X, I4, 2X, I4, 3X, F6.1, 2X, F6.1, 2X, F6.2, 2X, F6.2, 2X, F6.3, 2X, F6.3)'
	printf, u, ''  &  printf, u, ''
	printf, u, 'Output description:'
	printf, u, ''
	printf, u, 'FWHM: star FWHM'
	printf, u, 'x, y: coordinates (origin at (0,0))'
	printf, u, 'r: radial distance from the guide star'
	printf, u, 'a: position angle around guide star, counter-clockwise from x-axis'
	printf, u, 'l, w: standard deviations of the best fit elliptical gaussian'
	printf, u, 's: scaling factor of best fit model (for flux normalization)'
	printf, u, 'e: square root of relative least square fitting error'
	printf, u, '   e >= 0: normal exit'
	printf, u, '   e = -1: the fitting procedure has not converged'
	printf, u, '   e = -2: the fit has converged, but the parameters are not acceptable'
	printf, u, '   e = -3: the star being fitted is too near a brighter one'
	printf, u, ''
	printf, u, 'NOTE: all coordinates, distances, etc. are in pixel units'
	printf, u, '      angles are in degrees'
	close, u
	return
   end

   ; IPP_GRAPH: graphic output of measured IPP's and polynomial fit.

   PRO ipp_graph, r, l, l_fit, w, w_fit

	!P.multi = [0, 1, 2]
	lo_x = 0.  &  up_x = max(r)*1.1  &  xrange = [lo_x, up_x]
	lo_y = 0.  &  up_y = max([max(l), max(l_fit)]) * 1.1  &  yrange = [lo_y, up_y]
	plot, PSYM = 1, XRANGE = xrange, YRANGE = yrange, XMIN = 2, YMIN = 2,	  $
		  CHARSIZE = 1.5, XTITLE = 'radial distance from guide star [pixel]', $
		  YTITLE = 'l [pixel]', r, l
	oplot, r, l_fit
	plot, PSYM = 1, XRANGE = xrange, YRANGE = yrange, XMIN = 2, YMIN = 2,	  $
		  CHARSIZE = 1.5, XTITLE = 'radial distance from guide star [pixel]', $
		  YTITLE = 'w [pixel]', r, w
	oplot, r, w_fit
	!P.multi = 0
	return
   end

   ; ACCEPT_IPP: check if estimated IPP's are acceptable.

   FUNCTION accept_ipp, l, w, max_lw_ratio

	accept = l ge 0 and w ge 0 and l ge w
	if  accept and l*w gt 0  then  accept = l/w lt max_lw_ratio
	return, accept
   end


   PRO ipp, image, DISPLAY = im_to_display, WSIZE = wsize, WNUM = wnum,   $
   			X_GS = x_gs, Y_GS = y_gs, X_POS_GS = x_gs0, Y_POS_GS = y_gs0, $
   			X_STAR = xs, Y_STAR = ys, _EXTRA = extra,				$
   			NFWHM_BKG = n_fwhm_bkg,	NFWHM_FIT = n_fwhm_fit,			$
   			MAX_LW_RATIO = max_ratio, POLY_DEG = poly_deg,			$
   			x, y, r, a, l, w, s, e, guide_star, accepted_stars,		$
   			best_fit_models, l_coeff, l_fit, w_coeff, w_fit,		$
   			FILENAME = filename, GRAPH = graph

;	on_error, 1
	; Display image
	select = n_elements( x_gs ) * n_elements( y_gs ) eq 0 or $
			 n_elements( xs ) * n_elements( ys ) eq 0
	if  select  then  begin
	   if  n_elements( im_to_display ) eq 0  then  im_to_display = image
	   if  n_elements( wsize ) eq 0  then  ws = mysize( image, /DIM )  else $
	   if  n_elements( wsize ) eq 1  then  ws = [wsize, wsize]
	   ws = ws[0:1]
	   if  n_elements( wnum ) eq 0  then  wnum = 0
	   window, wnum, XSIZE = ws[0], YSIZE = ws[1]
	   tvscl, congrid( im_to_display, ws[0], ws[1] )
	endif
	; Select guide star(s)
	if  n_elements( x_gs ) * n_elements( y_gs ) eq 0  then begin
	   print, 'SELECT GUIDE STAR(S)'
	   click_on_max, image, _EXTRA = extra, /MARK, PSYM = 1, x_gs, y_gs
	   if  n_elements( x_gs ) * n_elements( y_gs ) eq 0  then  return
	endif
	; Select candidate stars
	if  n_elements( xs ) * n_elements( ys ) eq 0  then begin
	   print, 'SELECT STARS FOR IPPs ESTIMATION'
	   click_on_max, image, _EXTRA = extra, /MARK, PSYM = 7, xs, ys
	   if  n_elements( xs ) * n_elements( ys ) eq 0  then  return
	endif
	nstars = n_elements( xs )  &  x = xs  &  y = ys

	; Compute distances, etc.
	if  n_elements( x_gs ) gt 1  then begin
	   if  n_elements( x_gs0 ) * n_elements( y_gs0 ) eq 0  then  return
	   x0 = x_gs0  &  y0 = y_gs0
	endif else begin
	   x0 = x_gs  &  y0 = y_gs
	endelse
	r = distance( x, y, x0, y0 )
	w = sort( r )  &  r = r[w]  &  x = x[w]  &  y = y[w]
	a = angle( x0, y0, x, y )
	e = fltarr( nstars )  &  w = e  &  s = e  &  l = e  &  fw_hm = e
	accepted_stars  = ptrarr( nstars, /ALLOCATE_HEAP )
	best_fit_models = ptrarr( nstars, /ALLOCATE_HEAP )
	; Estimate and subtract background
	gs_fwhm = fwhm( image, X = x_gs[0], Y = y_gs[0] )
	if  n_elements( n_fwhm_bkg ) eq 0  then  n_fwhm_bkg = 5
	boxsize = round( n_fwhm_bkg * gs_fwhm )
	background = estimate_background( image, boxsize, _EXTRA = extra )
	i = image - background  &  background = 0
	; Estimate IPP's for each selected star
	if  n_elements( n_fwhm_fit ) eq 0  then  n_fwhm_fit = 5
	if  n_elements( max_ratio ) eq 0  then  max_ratio = 2*sqrt(3)
	edge = 2  &  guide_star = 0
	for  n = 0, nstars - 1  do begin
	   ; Extract guide star and stellar image
	   fw_hm[n] = fwhm( i, X = x[n], Y = y[n] )
	   boxsize = round( n_fwhm_fit * fw_hm[n] )
	   gs = superpose_stars( i, x_gs, y_gs, boxsize + 2*edge, $
	   						 _EXTRA = extra, /SUB_PIX )
	   star = extract( i, boxsize +2*edge, X = x[n], Y = y[n], $
	   				   LO_X = lx, LO_Y = ly )
	   m = get_max( star )
	   if  m[0] eq x[n]-lx  and  m[1] eq y[n]-ly  then begin
	   ; Determine best fit gaussian
	   fit_kernel, star, gs, a[n], _EXTRA = extra, EDGE = edge, $
				   ln, wn, sn, en, model, centered_data
if en NE -1 then begin
print,'ID   FWHM   x   y     r       a       l       w       s       e'
print,strtrim(string(n),2),' ',strtrim(string(fw_hm[n]),2),' ',strtrim(string(x[n]),2),' ',strtrim(string(y[n]),2),' ',strtrim(string(r[n]),2),' ',$
	strtrim(string(a[n]),2),' ',strtrim(string(ln),2),' ',strtrim(string(wn),2),' ',strtrim(string(sn),2),' ',strtrim(string(en),2)
endif
	   ; Store results
	   gs = extract( gs, boxsize, boxsize )
	   if  n_elements(gs) gt n_elements(guide_star)  then  guide_star = gs
	   e[n] = en  &  accept = en ge 0	; e = -1 ==> not converged
	   if  accept  then  accept = accept_ipp( ln, wn, max_ratio )
	   if  accept  then begin
	      l[n] = ln  &  w[n] = wn  &  s[n] = sn  &  e[n] = en
	      *accepted_stars[n] = centered_data
	      *best_fit_models[n] = model
	   endif  else  e[n] = -2	; e = -2 ==> not acceptable
	   endif  else  e[n] = -3	; e = -3 ==> not isolated
	endfor

	; Print results to file
	gs_fwhm = fwhm( guide_star )
	if  n_elements(filename) ne 0  then $
	   ipp_print, filename, x0, y0, gs_fwhm, x, y, r, a, fw_hm, l, w, s, e
	; Check and save acceptable results
	w0 = where( e ge 0 and l GT 0.04249, nstars )
	if  nstars eq 0  then begin
	   print, 'no successful fit'  &  return
	endif
	x = x[w0]  &  y = y[w0]  &  r = r[w0]  &  a = a[w0]
	l = l[w0]  &  w = w[w0]  &  s = s[w0]  &  e = e[w0]
	accepted_stars = accepted_stars[w0]
	accepted_stars = ptr_to_cube( same_size( accepted_stars ), /FREE )
	best_fit_models = best_fit_models[w0]
	best_fit_models = ptr_to_cube( same_size( best_fit_models ), /FREE )
	; Polynomial fit
	if  n_elements( poly_deg ) eq 0  then  poly_deg = 2
	if  n_elements( r ) lt poly_deg + 1  then begin
	   print, 'too few points for polynomial fit'  &  return
	endif
	polyfit, r, l, /NO_CONST, poly_deg, l_coeff, l_fit
	polyfit, r, w, /NO_CONST, poly_deg, w_coeff, w_fit
	; Graphic output
	if  keyword_set(graph)  then  ipp_graph, r, l, l_fit, w, w_fit
	return
   end
