;+
;				gaussint_fit
;
; Computes least squares Gaussian integrated to the pixel size.  Useful
; to fitting Gaussians to undersampled data.
; 
; CALLING SEQUENCE:
;	fit = gaussint_fit(x,y,a,sigma)
;
; INPUTS:
;	x - positions of the data points)
;	y - flux values
; OPTIONAL KEYWORD INPUTS:
;	pixel_size - size of the pixels (default = 1.0) I.E. the extent
;		of the input region integrated for each pixel.
;	estimates - intial guess for the coefficients with NTERMS elements in
;		same order as output A vector.  If not supplied the program
;		will generate a guess.
;	weights - vector of weights for the input data.
;	nterms - number of terms to fit between 3 and 6 (default = 6)
;		nterms = 3 - no background
;		nterms = 4 - constant background
;		nterms = 5 - linear background
;		nterms = 6 - quadratic background
;
; OUTPUTS:
;	a - coefficients of the fit
;		a(0) = peak of the Gaussian
;		a(1) = center
;		a(2) = sigma
;		a(3) = constant term of linear background
;		a(4) = linear term of the background
;		a(5) = quadratic term of the background
;	sigma - sigmas of the coefficients generated by CURVEFIT
;	fit - fitted data values
;
; NOTES:
;	If !debug is non-zero then the routine will generate a plot with
;	the original data plotted as a solid histogram plot, The fitted 
;	integrated Gaussian as a dashed histogram plot, the Gaussian computed
;	using the intial guess of the coeficients as a dotted line, the
;	computed background as a dashed line, and the unintegrated Gaussian
;	as a solid line.
;
;	If an initial guess is not supplied, then the guess is determined by:
;
;	   Background guess is a polynomial fit to the first 20% and
;	   last 20% of the data points.
;
;	   The peak of the Gaussian is the maximum value after subtracting
;	   the background guess.
;
;	   The center of the Gaussian is the position of the peak data point
;	   after subtraction of the background guess.
;
;	   The width of the Gaussian is set to 0.5 times the range covered by
;	   The data points with a data value greater than 1/2 the peak value
;	   (after subtracting the estimated background)
;
; EXAMPLE:
;	; Create some test data, rebin it, add noise and fit it.
;
;	xorig = findgen(100)/10.0
;	gauss, xorig, 5.2, 0.4, 100.0, yorig    ;center=5.2, sigma=0.4, peak=100
;	yorig = yorig + 20.0 + 1.2*xorig	;linear baseline
;	x = rebin(xorig,10)			;rebin to pixel size
;	y = rebin(yorig,10)
;	y = y + randomn(12345,10)		;add random noise
;
;	fit = gaussint_fit(x,y,a)
;	print,a
;
; HISTORY:
;	version 1  D. Lindler Sept. 1998
;-
; ============================================================================
;
function gaussian_integral,xcenter,sigma,peak,x1,x2
;
;			gaussian_integral
;
; Integrate gaussian function
;
; CALLING SEQENCE:
;	result = gaussian_integral(xcenter,sigma,peak,x1,x2)
;
; INPUTS:
;	xcenter = center of Gaussian
;	sigma = sigma of Gaussian
;	peak = peak of Gaussian
;	x1 = minimum(s) limit of integration
;	x2 = maximum(s) limit of integration
;
; OUTPUTS:
;	integrals from x1 to x2 are returned.
;
; HISTORY
;	version 1  D. Lindler  Sept. 1998
;
; ---------------------------------------------------------------------------
	return,(errorf((x2-xcenter)/sqrt(2)/sigma) - $
		errorf((x1-xcenter)/sqrt(2)/sigma)) * $
		sqrt(!pi)/sqrt(2)*sigma*peak
end
;
; =============================================================================
;
pro gaussint_funct,x,a,f
;
; 
; Routine to compute integrated gaussian function for curvefit
;	a(0) = peak of gaussian
;	a(1) = center
;	a(2) = sigma
;	a(3) = constant term of background
;	a(4) = linear term of background
;	a(5) = quadratic term of the background
;
	common gaussian_common,pwidth		;pixel width

	nterms = n_elements(a)
	f = gaussian_integral(a(1),a(2),a(0),x-pwidth/2,x+pwidth/2)/pwidth
	if nterms gt 3 then f = f + a(3)
	if nterms gt 4 then f = f + a(4)*x
	if nterms gt 5 then f = f + a(5)*x*x
	return
	end
;
; ============================================================================
;

function gaussint_fit,x,y,a,sigma,pixel_size=pixel_size,estimates=estimates, $
	weights=weights,nterms=nterms
;
; Integrated Gaussian Fit Main routine
;
	common gaussian_common,pwidth		;pixel width

	if n_params(0) lt 1 then begin
	   print,'CALLING SEQUENCE: fit = gaussint_fit(x,y,a,sigma)
	   print,'OPTIONAL KEYWORD INPUTS: pixel_size, estimates, weight, nterms
	   return,0
	end

	if n_elements(pixel_size) eq 0 then pwidth=1.0 $
				       else pwidth=float(pixel_size)
	if n_elements(weights) eq 0 then weights = x*0+1.0
	if n_elements(nterms) eq 0 then nterms = 6 else nterms = nterms>3<6

	if n_elements(estimates) eq 0 then begin
		a = dblarr(nterms)
;
; find polynomial baseline using 20 percent of points on each end
;
		if nterms gt 3 then begin
		   ns = n_elements(x)
	   	   n20 = (ns/5)>2
		   if nterms eq 4 then begin
		   	a(3) = total([y(0:n20-1),y(ns-n20:ns-1)])/(2*n20)
			ymod = y - a(3)
		     end else begin
			c = poly_fit([x(0:n20-1),x(ns-n20:ns-1)], $
				     [y(0:n20-1),y(ns-n20:ns-1)],1)
			ymod = y - c(0) - c(1)*x
			a(3) = c(0)
			a(4) = c(1)
		   end
		end else ymod = y
;
; find peak and center guess
;
		a(0) = max(ymod,position)
		a(1) = x(position)
;
; crude guess for sigma
;
		n = total(ymod gt (a(0)/2))>1	;number of points > hmax
		a(2) = n*((max(x)-min(x))/n_elements(x))/2
;
	end else a = estimates
;
; Plot Guess
;
	if !debug then begin
		plot,x,y,psym=10,xstyle=1
		gaussint_funct,x,a,f
		oplot,x,f,line=1
	endif
;
; Fit results
;
	fit = curvefit(double(x),double(y),weights,a,sigma, $
			function_name='gaussint_funct',/noderivative)
	if !debug then begin
		oplot,x,fit,psym=10,line=2,thick=2
		back = x*0
		if nterms gt 3 then back = back+a(3)
		if nterms gt 4 then back = back+x*a(4)
		if nterms gt 5 then back = back+x*x*a(5)
		oplot,x,back,line=2
		xx = congrid(x,200,/interp)
		gauss,xx,a(1),a(2),a(0),yy
		if nterms gt 3 then yy = yy + a(3)
		if nterms gt 4 then yy = yy + a(4)*xx
		if nterms gt 5 then yy = yy + a(5)*xx*xx
		oplot,xx,yy,thick=2
	end
return,fit
end
