;-----------------------------------------------------------------------------
;+
;
; NAME:
;   makepupil.pro
;
; IDENT:
;
;   Version 1.0. Francoise Delplancke (ESO). February 1998.
;
; PURPOSE:
;
;   This module creates a matrix representing the pupil of the telescope.
;   Module of the "WFS Geometry" library.
;
; DESCRIPTION:
;
;   It makes an annular pupil of the right size with values of 1 inside
;   and 0 outside of the pupil.
;
; CALL:
;
;   pup = makepupil (dim, pupdiam, cobs, xc=xc, yc=yc)
;
; INPUTS:
;
;   dim     = dimension of the simulation matrix to generate
;             [integer, in sampling points]
;
;   pupdiam = actual linear dimension of the pupil in the simulation
;             [integer, in sampling points]
;
;   cobs    = relative diameter of the central obsturation
;             [float, <1, fraction of pupdiam]
;
;   xc, yc  = keywords giving the position of the center of the pupil
;             [integer, <dim, default value = dim/2]
;
; OUTPUT:
;
;   pup = the matrix containing the pupil
;         [floats, dim x dim]
;
; COMMON:
;
;   It is called by including the file param.dat in this program.
;   See param.dat for the definition of each common variable.
;
; HISTORY:
;
;   Created on December 8, 1998, by F. Delplancke (ESO)
;   Developed from Francois Rigaut's program simul.pro v.3.0.
;   Last modification on April 23, 1998, by F. Delplancke.
;
;-
;-----------------------------------------------------------------------------


FUNCTION makepupil, dim, pupdiam, cobs, xc=xc, yc=yc

IF NOT keyword_set(xc) THEN xc =  dim/2-0.5
IF NOT keyword_set(yc) THEN yc =  dim/2-0.5

x  = findgen(dim)#replicate(1.,dim)
y  = transpose(x)
x  = x - xc
y  = y - yc
tmp   = sqrt(x^2.+y^2.)/(pupdiam/2.)

pup   =  (tmp LE 1.) AND (tmp GE cobs)         ; 1 under the pupil, 0 elsewhere
tpup  =  total(pup)

return, pup                                    ; returns the matrix

END                                            ; end of function MAKEPUPIL.PRO

