   ; MAX_CORRELATION: given an image of an object presumably located at
   ;	(x0,y0), find a better estimate of its position by maximizing the
   ;	correlation coefficient with a template image.

   ; ALGORITHM DESCRIPTION
   ;	The algorithm computes the correlation coefficient corresponding
   ;	to all the possible object locations within a box centered on the
   ;	initial estimate supplied on input: the position yielding the largest
   ;	correlation is taken as the best estimate of the object location.
   ;	It is then possible to estimate the object true location with
   ;	sub-pixel accuracy: this refinement is accomplished by centering the
   ;	image by means of CENTROIDER. In this case the input template is
   ;	assumed to be already centered. The sub-pixel positioning of the
   ;	object should be skipped with noisy images: the centering algorithm
   ;	acts essentially on the image maximum, which may not coincide with
   ;	the true 'center' of the object if the image is too noisy.

   ; INPUT
   ;	image, template: 2-D image and template
   ;	x0, y0: presumed position of object to be correlated
   ;	x_size, y_size: sizes of correlation box (>= 3 pixels)
   ;	search_box: size of 2-D box around (x0,y0) to search for the
   ;		object true location. If search_box = 1, simply compute the
   ;		correlation of the template and the object, assumed to be at
   ;		(x0,y0)
   ;	[X_BAD = , Y_BAD = ]: x- and y- coordinates of image pixels to be
   ;		excluded. These coordinates must be referred to the reference
   ;		frame of the input image
   ;	[/SUB_PIX]: determine the object position with sub-pixel accuracy
   ;		This step is always skipped if the image contains known bad
   ;		pixels: sub-pixel positioning involves interpolation, which may
   ;		be strongly unstable with bad data
   ;	[_EXTRA = ]: optional input parameters of CENTROIDER

   ; OUTPUT
   ;	x, y: estimate of object position yielding the largest correlation
   ;	correl: maximum value of correlation coefficient

   ; NOTE: the correlation coefficient is defined as in CORRELATION_COEFF


   PRO max_correlation, image, template, x0, y0, x_size, y_size, search_box, $
   						X_BAD = xb, Y_BAD = yb, SUB_PIX = sub_pix,	 		 $
   						_EXTRA = extra, x, y, correl

	s = mysize( image, /DIM )  &  min_size = 3
	h = round( search_box > 1 ) / 2
	correl = fltarr( ( 2*h + 1 )^2 )  &  x = correl  &  y = correl
	template1 = extract( template, x_size, y_size )
	n = 0
	for  j = -h, +h  do  for  i = -h, +h  do begin
	   x[n] = ( round( x0 ) + j ) > 0 < ( s[0] - 1 )
	   y[n] = ( round( y0 ) + i ) > 0 < ( s[1] - 1 )
	   ima_box = extract( image, x_size, y_size, X = x[n], Y = y[n], $
	   					  LO_X = lx, UP_X = ux, LO_Y = ly, UP_Y = uy )
	   sx = ux - lx + 1  &  sy = uy - ly + 1
	   if  min( [sx, sy] ) ge min_size  then begin
	      tem_box = template1
	      if  sx ne x_size or sy ne y_size  then begin
	         tem_box = center_image( tem_box, x[n] - lx, y[n] - ly )
	         tem_box = tem_box[0:sx-1,0:sy-1]
	      endif
	      if  n_elements( xb ) * n_elements( yb ) ne 0  then begin
	         wb = where( xb ge lx and xb le ux and yb ge ly and yb le uy, nb )
	         if  nb ne 0  then $
	            wb = coord_to_subs( xb[wb] - lx, yb[wb] - ly, sx )
	      endif
	      correl[n] = correlation_coeff( ima_box, tem_box, EXCLUDE = wb )
	   endif
	   n = n + 1
	endfor
	correl = max( correl, w )  &  x = x[w]  &  y = y[w]
	if  keyword_set( sub_pix ) and $
		n_elements( xb ) * n_elements( yb ) eq 0  then begin
	   ima_box = centroider( image, XC = x, YC = y, _EXTRA = extra, $
	   						 XSHIFT = xs, YSHIFT = ys )
	   ima_box = extract( ima_box, x_size, y_size, X = x, Y = y, $
	   					  LO_X = lx, UP_X = ux, LO_Y = ly, UP_Y = uy )
	   sx = ux - lx + 1  &  sy = uy - ly + 1
	   if  min( [sx, sy] ) ge min_size  then begin
	      tem_box = template1
	      if  sx ne x_size or sy ne y_size  then begin
	         tem_box = center_image( tem_box, x - lx, y - ly )
	         tem_box = tem_box[0:sx-1,0:sy-1]
	      endif
	      correl1 = correlation_coeff( ima_box, tem_box )
	      x1 = x - xs  &  y1 = y - ys
	      if  correl1 gt correl  then begin
	         correl = correl1  &  x = x1  &  y = y1
	      endif
	   endif
	endif
	return
   end
