   ; NEWTON_GAUSS: given a parametric model and a set of experimental data
   ;	to be approximated, estimate the optimal set of parameters.

   ; ALGORITHM DESCRIPTION: The problem may be expressed as a set of
   ;	non-linear algebraic equations, in the form
   ;	model(x) - data = 0,
   ;	where model is a vector function of a scalar or vector variable x
   ;	and data is the vector of observations to be approximated with the
   ;	model. The algebraic system is solved by means of the iterative
   ;	Newton-Gauss scheme
   ;	M dx = data - model(x)  and   x' = x + dx
   ;	where M is the Iacobi matrix of the model with respect to the
   ;	parameters vector and x' is a new approximation of the parameters.
   ;	At each iteration, the correction vector dx is found as the minimum
   ;	norm (in the Moore-Penrose sense) solution of the linear system
   ;	M dx = data - model
   ;	The algorithm requires an initial estimate of the solution and a
   ;	stopping criterion (e.g. variation of parameters smaller than a
   ;	pre-fixed tolerance).
   ;	Bad data masking may be performed by:
   ;	1) supplying on input an array of subscripts for the points to be
   ;	   masked (see keyword BAD_DATA)
   ;	2) supplying on input an estimate of the background noise standard
   ;	   deviation, used by the algorithm to identify the points where
   ;	   the error between the data and the model (computed at a pre-fixed
   ;	   iteration of the Newton scheme) is too large
   ;	   (see below for details)
   ;	The two data masking mode may be used together.

   ; INPUT
   ;	model_fun: name of a function to compute the model, defined as
   ;		FUNCTION model_fun, x
   ;	iacobi_fun: name of a function to compute the Iacobi matrix of the
   ;		model respect to the variable x, defined as
   ;		FUNCTION iacobi_fun, x
   ;	error_fun: name of a function to compute the error between the model
   ;		and the data, defined as
   ;		FUNCTION error_fun, model, data
   ;	converg_fun: name of a function to check convergence between
   ;		successive approximations x0 and x of the solution, defined as
   ;		FUNCTION converg_fun, x0, x
   ;	ini_x: initial estimate of the solution
   ;	data: data to be approximated by the model
   ;	[BAD_DATA = ]: array of subscripts for the data points to be masked
   ;	[/MASK]: if set, bad data masking is performed as described above
   ;		at point 2). Using this option, the following parameters may be
   ;		supplied by the user:
   ;		SIGMA_NOISE = sigma, NOISE_TOL = tol, WHEN = mask_now (see below)
   ;	[SIGMA_NOISE = ]: standard deviation of noise;
   ;		to be supplied only if /MASK is set (default = estimate of the
   ;		background noise standard deviation, derived by BACKGROUND_NOISE)
   ;	[WHEN = ]: iteration no. to search for bad data to be masked
   ;		(i.e. bad data are identified after (mask_now-1) iterations);
   ;		to be supplied only if /MASK is set
   ;		(default = at the 2nd iteration)
   ;	[NOISE_TOL = ]: bad data masking is active on the data points
   ;		where the absolute error between the data and the the model at
   ;		iteration "mask_now" is > tol * sigma;
   ;		to be supplied only if /MASK is set
   ;		(default = 5)
   ;	[/NO_SCALE]: avoid scaling variables to improve eigenvalue ratio of
   ;		the Hessian matrix
   ;	[MAXIT = ]: maximum number of iterations (default = 20)
   ;	[VERBOSE = ]: name of a procedure to print partial results,
   ;		defined as
   ;		PRO out_pro, iteration no., error, current solution
   ;	[_EXTRA = ]: BACKGROUND_NOISE optional input parameters
   ;		(if /MASK is set)
   ;	NOTE: the auxiliary data required by model_fun, iacobi_fun,
   ;		error_fun, converg_fun and out_pro may be passed to these
   ;		functions by means of COMMON blocks defined in the procedure
   ;		calling NEWTON_GAUSS

   ; OUTPUT
   ;	x: solution of the set of equations
   ;	error: error between model(x) and input data
   ;	model: model computed at the point x
   ;	[IT = ]: actual number of iterations
   ;	[W_BAD = ]: array of subscripts for masked data. If bad data
   ;		masking is performed with MODE 1 (see above), the value of
   ;		this output keyword coincides with the input value provided
   ;		with the keyword BAD_DATA

   ; REQUIRED PROCEDURES / FUNCTIONS:
   ;	LS_SYS, MIN_NORM_INVERSION, GINV





   ; SCALE_TRANS: compute scaling array for the Hessian matrix h.

   FUNCTION scale_trans, h

	s = mysize( h, /DIM )  &  s = s[0]  &  d = lindgen( s )
	scaling = identity( s )
	w = where( h[d,d] ne 0, n )
	if  n ne 0  then  scaling[d[w],d[w]] = 1 / sqrt( h[d[w],d[w]] )
	return, scaling
   end

   ; IDENTITY: return a n*n identity matrix.

   FUNCTION identity, n

	d = lindgen( n )
	i = fltarr( n, n )  &  i[d,d] = 1
	return, i
   end

   PRO newton_gauss, model_fun, iacobi_fun, error_fun, converg_fun, $
   					 ini_x,	data, BAD_DATA = bad_data, MASK = mask, $
   					 SIGMA_NOISE = sigma, NOISE_TOL = tol, WHEN =	$
   					 mask_now, NO_SCALE = no_scale, MAXIT = maxit, 	$
   					 VERBOSE = out_pro, _EXTRA = extra,				$
   					 x, error, model, IT = it, W_BAD = w_bad

	on_error, 1
	if  n_elements( bad_data ) ne 0  then  w_bad = bad_data
	if  keyword_set( mask )  then begin
	  if  n_elements( mask_now ) eq 0  then  mask_now = 2
	  if  n_elements( sigma )    eq 0  then $
	     sigma = background_noise( data, _EXTRA = extra, /EXCLUDE )
	  if  n_elements( tol )	     eq 0  then  tol = 5
	endif
	if  n_elements( maxit ) eq 0  then  maxit = 20
	verbose = n_elements( out_pro ) ne 0
	n_data = n_elements( data )
	it = 0  &  x = ini_x
	model = call_function( model_fun, x )
	if  n_elements( model ) ne n_data  then  message, 'input data error'
	error = call_function( error_fun, mask_pixels( model, w_bad ), $
									  mask_pixels( data, w_bad ) )
	if  verbose  then  call_procedure, out_pro, it, error, x
	converging = 0B

	while  it lt maxit and not converging  do begin

	   it = it + 1
	   if  keyword_set( mask )  then  if  it eq mask_now  then $
	      w_bad = add_elements( w_bad, $
	      						where( abs( data - model ) gt  tol*sigma ) )
	   iacobi = call_function( iacobi_fun, x )
	   ls_sys, iacobi, reform( data - model, n_data ), MASK = w_bad, $
	   		   lin_hessian, grad
	   if  it eq 1  then  if  not keyword_set( no_scale )  then $
	      s = scale_trans( lin_hessian )
	   if  not keyword_set( no_scale )  then begin
	     lin_hessian = s # lin_hessian # transpose( s )
	     grad = grad # s
	   endif
	   dx = min_norm_inversion( lin_hessian, grad )
	   if  not keyword_set( no_scale )  then  dx = dx # s
	   x0 = x  &  error0 = error  &  x = x + dx
	   model = call_function( model_fun, x )
	   error = call_function( error_fun, mask_pixels( model, w_bad ), $
							  mask_pixels( data, w_bad ) )
	   if  verbose  then  call_procedure, out_pro, it, error, x
	   converging = call_function( converg_fun, x0, x )

	endwhile

	if  not converging  then begin
	   if  verbose  then  print, 'newton-gauss search not converged'
	   error = -1
	endif else  if  verbose  then $
	   print, 'newton-gauss search converged'

	return
   end
