   ; NEWTON_GAUSS: given a parametric model and a set of experimental data to be
   ;	approximated, estimate the optimal set of parameters.

   ; ALGORITHM DESCRIPTION: The problem may be expressed as a set of non-linear
   ;	algebraic equations, in the form
   ;	model(x) - data = 0,
   ;	where model is a vector function of a scalar or vector variable x and
   ;	data is the vector of observations to be approximated with the model.
   ;	The algebraic system is solved with the iterative Newton-Gauss scheme
   ;	M dx = data - model(x)  and   x' = x + dx
   ;	where M is the Iacobi matrix of the model with respect to the parameters
   ;	vector and x' is a new approximation of the parameters.
   ;	At each iteration, the correction vector  dx  is found as the minimum
   ;	norm (in the Moore-Penrose sense) solution of the linear system
   ;	M dx = data - model
   ;	In case of jamming (i.e. when x oscillates around the true solution
   ;	without either converging or diverging), an approximation of the
   ;	solution is found by a 1-dimensional search along the correction vector
   ;	found with the Newton-Gauss scheme.
   ;	The algorithm requires an initial estimate of the solution and a
   ;	stopping criterion (e.g. variation of parameters smaller than a
   ;	pre-fixed tolerance).
   ;	Bad data masking may be performed by:
   ;	1) supplying on input an array of subscripts for the points to be
   ;	   masked (see keyword BAD_DATA)
   ;	2) supplying on input an estimate of the background noise standard
   ;	   deviation, used by the algorithm to identify the points where the
   ;	   error between the data and the model (computed at a pre-fixed
   ;	   iteration of the Newton scheme) is too large (see below for details)

   ; INPUT
   ;	model_fun: name of a function to compute the model, defined as
   ;		FUNCTION model_fun, x, [KEYWORD_1 = , ..., KEYWORD_N = ]
   ;		where the keyword parameters are passed through NEWTON_GAUSS by
   ;		keyword inheritance
   ;	iacobi_fun: name of a function to compute the Iacobi matrix of the
   ;		model respect to the variable x, defined as
   ;		FUNCTION iacobi_fun, x, [KEYWORD_1 = , ..., KEYWORD_N = ]
   ;		where the keyword parameters are passed through NEWTON_GAUSS by
   ;		keyword inheritance
   ;	error_fun: name of a function to compute the error between the model
   ;		and the data, defined as
   ;		FUNCTION error_fun, model, EXCLUDE = exclude, $
   ;							 [KEYWORD_1 = , ..., KEYWORD_N = ]
   ;		where exclude is an optional input/output parameter, e.g. a vector
   ;		of subscripts identifying the data to be excluded, and the other
   ;		keyword parameters are passed through NEWTON_GAUSS by keyword
   ;		inheritance
   ;	converg_fun: name of a function to check convergence between successive
   ;		approximation x0 and x of the solution, defined as
   ;		FUNCTION converg_fun, x0, x, [KEYWORD_1 = , ..., KEYWORD_N = ]
   ;		where the keyword parameters are passed through NEWTON_GAUSS by
   ;		keyword inheritance
   ;	ini_x: initial estimate of the solution
   ;	data: data to be approximated by the model
   ;	[BAD_DATA = bad_data]: array of subscripts for the data points to be
   ;		masked (data masking MODE 1, described above)
   ;	[/MASK]: if set, bad data masking is performed as described above at
   ;		point 2) (data masking MODE 2). Using this data masking mode, the
   ;		following parameters may be supplied by the user:
   ;		SIGMA_NOISE = sigma, NOISE_TOL = tol, WHEN = mask_now (see below)
   ;	[SIGMA_NOISE = sigma]: standard deviation of noise;
   ;		to be supplied only if /MASK is set (default = estimate of the
   ;		background noise standard deviation, derived by BACKGROUND_NOISE)
   ;	[WHEN = mask_now]: iteration no. to search for bad data to be masked
   ;		(i.e. bad data are identified after (mask_now-1) iterations);
   ;		to be supplied only if /MASK is set (default = at the 2nd iteration)
   ;	[NOISE_TOL = tol]: bad data masking is active on the data points
   ;		where the absolute error between the data and the the model at
   ;		iteration "mask_now" is > tol * sigma;
   ;		to be supplied only if /MASK is set (default = 5)
   ;	[/VERBOSE]: print partial results
   ;	[OUT_PRO = out_pro]: name of a procedure to print partial results,
   ;		defined as
   ;		PRO out_pro, iteration no., error, current solution, $
   ;					 [KEYWORD_1 = , ..., KEYWORD_N = ]
   ;		where the keyword parameters are passed through NEWTON_GAUSS by
   ;		keyword inheritance
   ;		(default name = 'output_pro')
   ;	[MAXIT = maxit]: maximum number of iterations (default = 20)
   ;	[_REF_EXTRA = extra]: optional input parameters required by model_fun,
   ;		iacobi_fun, error_fun, converg_fun, out_pro and, if /MASK is set,
   ;		by BACKGROUND_NOISE
   ;	NOTE: the auxiliary data required by model_fun, iacobi_fun, error_fun,
   ;		converg_fun and out_pro may be passed to these functions in one of
   ;		the following ways:
   ;		1) by means of COMMON blocks defined in the procedure calling
   ;		   NEWTON_GAUSS
   ;		2) by means of keyword parameters, included in the definition of
   ;		   these functions, and passed on by NEWTON_GAUSS through the
   ;		   extra keyword _REF_EXTRA (see below)

   ; INPUT/OUTPUT
   ;	[_REF_EXTRA = extra]: it may contain keyword parameters modified by
   ;		other procedures/functions called by NEWTON_GAUSS.
   ;		Suppose for instance that the computation of the model requires
   ;		the computation of auxiliary data and that NEWTON_GAUSS is used
   ;		many times by some calling routine: if the auxiliary data can be
   ;		used again, it is convenient to save them into a set of input/
   ;		output keywords handled by NEWTON_GAUSS as a '_REF_EXTRA' keyword

   ; OUTPUT
   ;	x: solution of the set of equations
   ;	error: error between model(x) and input data
   ;	model: model computed at the point x
   ;	[IT = it]: actual number of iterations
   ;	[W_BAD = w_bad]: array of subscripts for masked data. If bad data
   ;		masking is performed with MODE 1 (see above), the value of this
   ;		output keyword coincides with the input value provided with the
   ;		keyword BAD_DATA

   ; REQUIRED PROCEDURES / FUNCTIONS:
   ;	LS_SYS, MIN_NORM_INVERSION, GINV, GOLDEN_SECTION



   PRO newton_gauss, model_fun, iacobi_fun, error_fun, converg_fun, ini_x,	 $
		     data, BAD_DATA = bad_data, MASK = mask, SIGMA_NOISE = sigma,	 $
		     NOISE_TOL = tol, WHEN = mask_now, VERBOSE = verbose, 			 $
		     OUT_PRO = out_pro, MAXIT = maxit, _REF_EXTRA = extra, 			 $
		     x, error, model, IT = it, W_BAD = w_bad

common test, converging, jamming
;	on_error, 1
	if  n_elements( bad_data ) ne 0  then begin		; masking with MODE 1
	   w_bad = bad_data  &  mask = 0
	endif
	if  keyword_set( mask )  then begin
	  if  n_elements( mask_now ) eq 0  then  mask_now = 2
	  if  n_elements( sigma )    eq 0  then $
	     sigma = background_noise( data, _EXTRA = extra, /EXCLUDE )
	  if  n_elements( tol )	     eq 0  then  tol = 5
	endif
	if  n_elements( maxit ) eq 0  then  maxit = 20
	if  keyword_set( verbose )  then $
	   if  n_elements( out_pro ) eq 0  then  out_pro = 'output_pro'
	n_data = n_elements( data )
	it = 0  &  x = ini_x
	model = call_function( model_fun, x, _EXTRA = extra )
	if  n_elements( model ) ne n_data  then  message, 'input data error'
	error = call_function( error_fun, model, EXCLUDE = w_bad, _EXTRA = extra )
	if  keyword_set( verbose )  then $
	   call_procedure, out_pro, it, error, x, _EXTRA = extra
	converging = 0B
	jamming = 0B  &  jam_up = 0B  &  jam_down = 0B

	while  it lt maxit and not converging and not jamming  do begin

	   it = it + 1
	   if  keyword_set( mask )  then  if  it eq mask_now  then $
	      w_bad = where( abs( data - model ) gt  tol*sigma )
	   iacobi = call_function( iacobi_fun, x, _EXTRA = extra )
	   ls_sys, iacobi, reform( data - model, n_data ), MASK = w_bad, $
	   		   lin_hessian, grad
	   dx = min_norm_inversion( lin_hessian, grad )
	   x0 = x  &  error0 = error  &  x = x + dx
	   model = call_function( model_fun, x, _EXTRA = extra )
	   error = call_function( error_fun, model, EXCLUDE = w_bad, _EXTRA = extra )
	   if  keyword_set( verbose ) $
	      then  call_procedure, out_pro, it, error, x, _EXTRA = extra
	   converging = call_function( converg_fun, x0, x, _EXTRA = extra )
	   if  not jam_up    then  jam_up = error gt error0    else $
	   if  not jam_down  then  jam_down = error le error0  else $
	   begin
	      jamming = error gt error0
	      jam_up = jamming  &  jam_down = jamming
	   endelse

	endwhile

	if  not converging and jamming  then begin
	   golden_section, x0, x, error0, error, error_fun, X_FUN = model_fun, $
	   				   EXCLUDE = w_bad, CONVERG_FUN = converg_fun,		   $
					   _EXTRA = extra, x, error, MIN_X_FUN = model
	   converging = not converging
	endif
	if  not converging  then begin
	   if  keyword_set( verbose )  then $
	      print, 'newton-gauss search not converged'
	   error = -1
	endif else $
	   if  keyword_set( verbose )  then begin
	      if  jamming  then $
	         call_procedure, out_pro, it, error, x, _EXTRA = extra
	      print, 'newton-gauss search converged'
	   endif

	return

   end
