FUNCTION OFFSETS ,im1,im2,DISP1=disp1,DISP2=disp2,WSIZE=wsize,WNUM=wnum, $
		ROT=rot,XYTOL=xytol,PSCALE=pscale

;NAME:
;	OFFSETS

;CALLING SEQUENCE:
;	output=OFFSETS(im1,im2,DISP1=disp1,DISP2=disp2,WSIZE=wsize,WNUM=wnum, $
;		ROT=rot,XYTOL=xytol,PSCALE=pscale)

;PURPOSE: 
;	Display two images, ask to select stars by clicking on it, compute
;	an average of X and Y offsets between frames. If there's a rotation
;	between frames, the program looks for a rotation angle and a translation
;	offset. It minimizesim the position difference between:
;		(x2,y2) and  translated(rotated (x1,y1))
; 	which means, rotate stars selected in im1, then translate them and look
;	if it matches with the corresponding stars in im2.       
;	The rotation center is the pixel (0,0) of im1.
;	Possibility to cope with differents pixels scales between images

;INPUT:
;	IM1,IM2 - image to considered.

;OUTPUT:     
;	[XOFFSET,YOFFSET] - scalar of value giving X and Y offsets.
;	or [ANGLE,XOFFSET,YOFFSET] if ROT keyword is set.
;	Output angle is in radian.	

;OPTIONAL INPUT:
;	DISP1,DISP2 - images to be display. For example you can display a
;			log plot: 
;		offsets,im1,im2,xoffset,yoffset,disp1=alog10(im1), $
;			disp2=alog10(im2) 
;	WNUM - 	A vector of window numbers to be displayed. 
;		Default is WNUM=[2,3]
;	WSIZE - A vector of image size to be display.
;		Default is WSIZE=[size of image1,size of image2]
;	ROT - 	Set this keyword in case of rotation between images. It's a 
;		3 elements vector of [angle(deg),xoffset,yoffset] which is a $
;		 starting guess for the Newton algorithm.
; 	PSCALE - Scalar elements of (IM1 Pixel scale)/(IM2 Pixel scale) if
;			im1 and im2 have differents pixel scale.
;	XYTOL - Tolerance (scalar) for X and Y error. If the difference between
;		X2 and Y2 and the translated (rotated) X1 Y1 position is more $
;		than XYTOL then there's a warning printed on screen.  

;EXAMPLE:
;	test=offsets(im1,im2,disp1=alog10(im1>600),disp2=alog10(im2>600),$
;		xytol=4.,Rot=[180,2000,2000],wsize=[512,512])
;	Will display successively two 512*512 images with a log10 scale and
;	search for (after stars selections) the rotation angle, the X Y offsets. 
;	Warning will be printed if for X2-rotated(X1)-xoffset > 4. pixels
;				 or if Y2-rotated(Y1)-yoffset > 4. pixels

;RESTRICTION:
;	This program use findstars procedures. You must have findstars.


device, CURSOR_STANDARD=35
if n_params() LT 2 then begin
	print,'SYNTAX: '
	print,'output=OFFSETS(im1,im2,DISP1=disp1,DISP2=disp2,WSIZE=wsize,'
	print,'		WNUM=wnum,ROT=rot,XYTOL=xytol,PSCALE=pscale)'
	return,0
endif

if not keyword_set(disp1) then disp1=im1
if not keyword_set(disp2) then disp2=im2
if not keyword_set(wsize) then wsize=[n_elements(im1(0,*)), $
	n_elements(im1(*,0)),n_elements(im2(0,*)),n_elements(im2(*,0))]
if not keyword_set(wnum) then wnum=[2,3]
if not keyword_set(XYTOL) then XYTOL=3 
if n_elements(wsize) EQ 2 then wsize=[wsize,wsize]

;STARS SELECTION
print,'Select star(s) in the first image to compute approximate offset:'
a=select_stars(im1,10,x1,y1,DISPLAY=disp1,WNUM=wnum(0),WSIZE=wsize(0:1),$
	/NO_REMOVE,/NO_CENTER,/NO_NORM,/NO_VERBOSE,NO_EXTRACT,/NO_PROCESS,$
	/_EXTRA,/MARK)

print,'Select star(s) (same number, same order) in the 2nd image:'
b=select_stars(im2,10,x2,y2,DISPLAY=disp2,WNUM=wnum(1),WSIZE=wsize(2:3),$
	/NO_REMOVE,/NO_CENTER,/NO_NORM,/NO_VERBOSE,NO_EXTRACT,/NO_PROCESS,$
	/_EXTRA,/MARK)
if n_elements(x1) NE n_elements(x2) then begin
	print,'You must choose the same number of stars'
	return,0
endif

if keyword_set(PSCALE) then begin
	x2=x2/float(pscale)
	y2=y2/float(pscale)
endif

;ROTATION ANGLE AND TRANSLATION COMPUTATION
if keyword_set(ROT) then begin
	rotate1,x2,y2,x1,y1,[rot(0)*3.1415/180.,rot(1),rot(2)],res
	for i=0,n_elements(x1)-1 do begin 
		x1(i)=x1(i)*cos(res(0))-y1(i)*sin(res(0))+res(1)
		y1(i)=x1(i)*sin(res(0))+y1(i)*cos(res(0))+res(2)
	endfor
	xerr=x2-x1 & yerr=y2-y1
	print,'Mean residual error after rotation and translation:'
	print,'		X error = ',strtrim(mean(xerr),1)
	print,'		Y error = ',strtrim(mean(yerr),1) & print,''
 	id=where((abs(xerr) GT xytol) OR (abs(yerr) GT xytol))
	if id(0) NE -1 then begin
		print,'WARNING: Error above Tolerance for following stars:'
		print,'(x2,y2,Xerr,Yerr):'
		for i=0,n_elements(id)-1 do print,x2(id(i)),y2(id(i)),$
			xerr(id(i)),yerr(id(i))
	endif
	outres=res
endif else begin
	xoffset=mean(x2-x1) & yoffset=mean(y2-y1)
	print,'' & print,'approximate X offset: ',strtrim(xoffset,1)
	print,'approximate Y offset: ',strtrim(yoffset,1) & print,''
	xerr=x2-x1-xoffset & yerr=y2-y1-yoffset
	print,'Mean residual error after translation:'
	print,'	X error = ',strtrim(mean(xerr),1)
	print,'	Y error = ',strtrim(mean(yerr),1) & print,''
	id=where((abs(xerr) GT xytol) OR (abs(yerr) GT xytol))
	if id(0) NE -1 then begin
		print,'WARNING: Error above Tolerance for following stars'
		print,'(x2,y2,Xerr,Yerr):'
		for i=0,n_elements(id)-1 do print,x2(id(i)),y2(id(i)),$
			xerr(id(i)),yerr(id(i))
	endif
	outres=[xoffset,yoffset]
endelse

return,outres
end
