   ; PSF_EXTRACT: iterative estimation of the PSF in a stellar field.

   ; ALGORITHM DESCRIPTION
   ;	The user selects a set of suitable stars by mouse clicking
   ;	(trying to avoid close binaries!). The PSF is essentially computed
   ;	as the median of these stars. The ouskirts smoothing procedure (see
   ;	SKIRT_SMOOTH) is finally applied.
   ;	In practice an accurate estimation requires subtraction of the
   ;	background and of secondary sources: this task is accomplished
   ;	iteratively, by means of FINDSTARS. Preliminary estimates of
   ;	background and PSF are derived and used to run FINDSTARS, in order
   ;	to
   ;	- subtract the brightest stars, thus improving the background
   ;	  estimate
   ;	- subtract secondary sources around previously selected stars
   ;	  to refine the PSF estimate
   ;	The whole process is repeated iteratively. The total number of
   ;	iterations is equal the number of detection levels passed to
   ;	FINDSTARS (see parameter min_intensity)

   ; INPUT
   ;	image: stellar field
   ;	psfsize: size of PSF to be extracted (upper limit)
   ;	[N_FWHM = ]: the background may be estimated either by median
   ;		filtering (see MEDIAN_FILTER) or by sampling + spline
   ;		interpolation (see IMAGE_BACKGROUND). In any case, the box
   ;		width for median smoothing or background sampling is set to
   ;		N_FWHM times an estimate of the PSF FWHM
   ;		(default N_FWHM = 5)
   ;	min_intensity: 1-D array, containing the intensity levels used by
   ;		FINDSTARS as lower thresholds for detection
   ;	[MIN_CORRELATION = ]: correlation threshold (default = 0.85)
   ;	[/SKY_INTERP]: estimate background by means of IMAGE_BACKGROUND
   ;	[_EXTRA = ]: optional parameters of SELECT_STARS, FINDSTARS,
   ;		IMAGE_BACKGROUND, SKIRT_SMOOTH

   ; OUTPUT
   ;	psf: psf estimate
   ;	background: image background estimate




   ; UPDATE_PSF: refine the PSF estimate, using a pre-fixed set of stars.

   FUNCTION update_psf, image, psf, image_model, background, $
   						x0, y0, x, y, f, _EXTRA = extra

	imsize = mysize( image, /DIM )
	psfsize = ( mysize( psf, /DIM ) )[0]
	if  n_elements( f ) ne 0  then begin
	   n0 = n_elements( x0 )  &  w = lonarr( n0 )
	   for  n = 0, n0 - 1  do begin
	      m = min( distance( x, y, x0[n], y0[n] ), here )
	      w[n] = here
	   endfor
	   selected_stars = stars( psf, x[w], y[w], f[w], _EXTRA = extra, $
	   						   imsize[0], imsize[1] )
	endif else  selected_stars = fltarr( imsize[0], imsize[1] )
	residual = image - ( image_model - selected_stars ) - background
	stack = select_stars( residual, psfsize, /NO_REMOVE, /NO_VERBOSE, $
						  _EXTRA = extra, x0, y0 )
	psf = stack_median( stack );, WEIGHTS = sqrt( residual[x0,y0] > 0 ) )
	psf = skirt_smooth( psf, /PAD_0, _EXTRA = extra )
	return, normalize( psf )
   end


   ; MAX_SIZE: determine maximum allowable size of PSF, given a set of
   ;	selected stars centered at (x,y).

   FUNCTION max_size, x, y, s

	l = min( [ min( x ), min( y ) ] )
	r = min( [ min( s[0] - 1 - x ), $
			   min( s[1 < ( n_elements(s) - 1 )] - 1 - y ) ] )
    return, 2 * min( [l, r] )
   end


   PRO psf_extract, image, psfsize, N_FWHM = n_fwhm, _EXTRA = extra, $
   		    min_intensity, min_correlation, psf, background

	; Select stars
	stack = select_stars( image, psfsize, _EXTRA = extra, /NO_EXTRACT, $
						  x_selected, y_selected )
	if  n_elements( x_selected ) * n_elements( y_selected ) eq 0  then $
	   return
	image_size = mysize( image, /DIM )
	s = min( [ max_size( x_selected, y_selected, image_size ), psfsize ] )
	; Preliminary estimate of background
	if  n_elements( n_fwhm ) eq 0  then  n_fwhm = 5
	core = 15	; max. core size of a typical PSF
	box = round( n_fwhm * image_width( extract( image, core, core, $
				 X = x_selected[0], Y = y_selected[0] ) ) )
	box = box + 1 - box mod 2
	background = estimate_background( image, box, _EXTRA = extra )
	; Preliminary psf estimate
	psf = update_psf( image, fltarr( s, s ), $
					  fltarr( image_size[0], image_size[1] ), $
					  background, x_selected, y_selected,	  $
					  _EXTRA = extra, INNER_RAD = box / 2 )
   	; Iterative PSF refinement
   	n_levels = n_elements( min_intensity )
   	for  n = 0, n_levels - 1  do begin
   	   ; Analyze the image with the current estimates of psf and background.
   	   ; Let FINDSTARS refine the background estimate
	   findstars, image, psf, BACKGROUND = background, _EXTRA = extra, $
	   			  N_FWHM = n_fwhm, min_intensity[n], min_correlation,  $
	   			  N_ITER = 1, x, y, f, image_model, background
	   ; Refine psf estimate, using new background
	   psf = update_psf( image, psf, image_model, background, $
   					     x_selected, y_selected, x, y, f,	  $
   					     _EXTRA = extra, INNER_RAD = box / 2 )
   	endfor
   	return
   end
