   ; SEARCH_MAX: find the local maxima within a 2-D image above a
   ;	threshold, by means of ALL_MAX.

   ; INPUT
   ;	image
   ;	[MINIFICATION = ]: image minification factor (integer, >= 1).
   ;		With very noisy data the search for local maxima is more robust
   ;		if the image is previously downsampled (by neighbor averaging).
   ;		Default is  MINIFICATION = 1 (i.e. no minification)
   ;	threshold: lower value for detection
   ;	[X0 = , Y0 = ]: reference position (see keywords below)
   ;		(default = central pixel of image)
   ;	[MAX_DIST = ]: maximum distance (in pixels) of detected
   ;		maxima from (x0,y0)
   ;	[/NEAREST]: among the detected maxima, select the nearest to (x0,y0)
   ;	[/MAXIMUM]: among the detected maxima, select the brightest one
   ;	NOTE. The 3 checks above are performed in the same order as they are
   ;		presented; of course it is not possible to select the nearest
   ;		maximum to (x0,y0) and at the same time the brightest one ...
   ; OUTPUT
   ;	n: number of detected maxima
   ;	x, y, i: position and central intensity of local maxima
   ; NOTE: if image minification is applied, output intensities of detected
   ;	maxima are referred to the minified image, whereas positions are
   ;	transformed back to the reference frame of the input image

   PRO search_max, image, MINIFICATION = mf, threshold,	  $
   				   X0 = x0, Y0 = y0, MAX_DIST = max_dist, $
   				   NEAREST = nearest, MAXIMUM = maximum, n, x, y, i

	if  n_elements( mf )  eq 0  then  mf = 1  &  mf = round( mf ) > 1
	if  n_elements( x0 ) * n_elements( y0 ) ne 0  then begin
	   x_ref = float( x0 ) / mf  &  y_ref = float( y0 ) / mf
	endif
	check_dist = n_elements( max_dist ) ne 0
	select_nearest  = keyword_set( nearest )
	select_maximum  = keyword_set( maximum )
	all_max, minify( image, mf, mf ), SX = sx, SY = sy, threshold, x, y, i, n
	found = n ne 0
	if  found  then begin
	   if  check_dist or select_nearest then begin
	      if  n_elements( x_ref ) * n_elements( y_ref ) eq 0  then begin
	         x_ref = sx / 2  &  y_ref = sy / 2
	      endif
	      d = distance( x, y, x_ref, y_ref )
	      if  check_dist  then begin
	         w = where( d le max_dist, n )  &  found = n ne 0
	         if  found  then begin
	            d = d[w]  &  x = x[w]  &  y = y[w]  &  i = i[w]
	         endif
	      endif
	      if  found and select_nearest  then begin
	         m = min( d, w )  &  x = x[w]  &  y = y[w]  &  i = i[w]
	      endif
	   endif
	   if  found and select_maximum  then begin
	      m = max( i, w )  &  x = x[w]  &  y = y[w]  &  i = i[w]
	   endif
	endif
	if  found  then begin
	   n = n_elements( i )
	   x = mf * x + ( mf - 1 ) / 2.  &  y = mf * y + ( mf - 1 ) / 2.
	endif
	return
   end
