   ; SELECT STARS: extraction and pre-processing of stellar images.
   ;	A stack of stellar images, selected interactively by the user, is
   ;	created. Each image is processed by median removal (to subtract any
   ;	smooth background contribution), centering with sub-pixel accuracy
   ;	and normalization. Any of the previous steps may be skipped. If a
   ;	background estimate better than the one obtained by local median
   ;	smoothing is available, it is better to extract candidate stars
   ;	directly on the background-removed image and avoid median subtraction.
   ;	An estimate of the PSF for the given stellar field may be derived by
   ;	computing the median of the stack with STACK_MEDIAN. The outer part
   ;	of the resulting image may then be smoothed by means of SKIRT_SMOOTH.

   ; INPUT
   ;	image: stellar field image
   ;	boxsize: size of retrieved PSF
   ;	[MED_BOXSIZE = ]: width of median filtering box for local median
   ;		removal (default = boxsize / 2). It must be < image size
   ;	[DISPLAY = ]: image to be displayed for stars selection
   ;		(e.g. the original image in a suitable scale, as SQRT, LOG, etc.)
   ;		(default = input image)
   ;	[WNUM = ]: window number to display the stellar field (default = 0)
   ;	[WSIZE = ]: scalar or 2-elements array defining the window size
   ;		(default = image size)
   ;	[/NO_REMOVE]: avoid median removal
   ;	[/NO_CENTER]: avoid sub-pixel centering
   ;	[/NO_NORM]: avoid normalization
   ;	[/NORM_MAX]: each stellar image may be normalized to a total
   ;		volume = 1 (default) or to a maximum = 1 if /NORM_MAX is set
   ;	[_EXTRA = ]: optional input parameters of CLICK_ON_MAX and CENTROIDER
   ;	[/NO_VERBOSE]: avoid text output
   ;	[/NO_EXTRACT]: just select stars, without extracting sub-images
   ;	[/NO_PROCESS]: avoid processing of selected stars (extraction only)

   ; INPUT/OUTPUT
   ;	x_selected, y_selected: x- and y- coordinates of selected stars
   ;		If coordinates are supplied on input, the selection step is
   ;		skipped. This may be useful to re-process previously selected
   ;		stars with different parameters

   ; OUTPUT
   ;	Return stack of processed stellar images
   ;	scale: central intensities of selected stars before normalization


   FUNCTION select_stars, image, boxsize, $
   					DISPLAY = im_to_display, WNUM = wnum, WSIZE = wsize,  $
   					MED_BOXSIZE = med_boxsize, NO_REMOVE = no_remove,	  $
   					NO_CENTER = no_center, _EXTRA = extra, NORM_MAX =	  $
   					norm_max, NO_NORM = no_norm, NO_VERBOSE = no_verbose, $
   					NO_EXTRACT = no_extract, NO_PROCESS = no_process,	  $
   					x_selected, y_selected, scale

	no_remove = keyword_set( no_remove ) or keyword_set( no_process )
	no_center = keyword_set( no_center ) or keyword_set( no_process )
	no_norm = keyword_set( no_norm ) or keyword_set( no_process )
	verbose = not ( keyword_set( no_verbose ) or keyword_set( no_process ) )
	if  n_elements( med_boxsize ) eq 0  then  med_boxsize = boxsize / 2
	med_boxsize = med_boxsize < ( min( mysize( image, /DIM ) ) - 1 )
	med_boxsize = med_boxsize + ( 1 - med_boxsize mod 2 )
	; Display image
	if  n_elements( im_to_display ) eq 0  then  im_to_display = image
	if  n_elements( wnum )  eq 0  then  wnum  = 0
	if  n_elements( wsize ) eq 0  then  ws = mysize( image, /DIM )  else $
	if  n_elements( wsize ) eq 1  then  ws = [wsize, wsize]		    else $
	   ws = wsize
	select = n_elements( x_selected ) * n_elements( y_selected ) eq 0
	if  select  then begin
	   window, wnum, XSIZE = ws[0], YSIZE = ws[1]
	   tvscl, congrid( im_to_display, ws[0], ws[1] )
	   ; Select candidate stars by mouse click
	   if  verbose  then $
	      print, 'Select stars by mouse click. Press right button to exit'
	   click_on_max, image, _EXTRA = extra, x, y
	endif else begin
	   x = x_selected  &  y = y_selected
	endelse
	if  keyword_set( no_extract )  then begin
	   if  n_elements( x ) * n_elements( y ) ne 0  then begin
	      x_selected = x  &  y_selected = y
	   endif
	   return, 0
	endif
	; Extraction and processing of selected stellar images
	nstars = n_elements( x )
	if  verbose  then  print, nstars, ' selected stars'
	if  nstars eq 0  then  return, -1
	if  not keyword_set( no_remove )  then $
	   aug_size = boxsize + ( med_boxsize / 2 ) * 2  else $
	   aug_size = boxsize
	stack = fltarr( boxsize, boxsize, nstars )
	if  verbose  then  print, 'Processing of selected sub-images ...'
	cbox = 3  &  n_accept = 0
	for  n = 0, nstars - 1  do begin
	   box = extract( image, aug_size, aug_size, X = x[n], Y = y[n], $
	   				  LO_X = lx, UP_X = ux, LO_Y = ly, UP_Y = uy )
	   if  ux - lx + 1 eq aug_size and uy - ly + 1 eq aug_size  then begin
	   	  ; Median removal
		  if  not keyword_set( no_remove )  then $
		     box = box - median( box, med_boxsize )
		  ; Size reduction
		  box = extract( box, boxsize, boxsize, $
		     			 X = aug_size / 2, Y = aug_size / 2 )
		  ; Centering
		  if  not keyword_set( no_center )  then begin
		     box = centroider( box, _EXTRA = extra, $
		     				   XC = boxsize / 2, YC = boxsize / 2 )
		     m = get_max( extract( box, cbox, cbox, $
		     			  X = boxsize / 2, Y = boxsize / 2 ) )
		     dx = cbox / 2 - m[0]  &  dy = cbox / 2 - m[1]
		     box = shift( box, dx, dy )
		  endif
		  star_intensity = box[boxsize / 2, boxsize / 2]
		  ; Normalization
		  if  not keyword_set( no_norm )  then $
		     if  keyword_set( norm_max )  then $
		        box = box / max( box )	  else $
		        box = normalize( box )
		  ; Stacking
		  stack[*,*,n_accept] = box  &  n_accept = n_accept + 1
		  ; Save coordinates of selected star
		  if  select  then begin
		     x_selected = add_elements( x_selected, x[n] )
		     y_selected = add_elements( y_selected, y[n] )
		     scale = add_elements( scale, star_intensity )
		  endif
	   endif
	endfor
	if  verbose  then  print, n_accept, ' accepted stars'
	if  n_accept eq 0  then  return, -1
	stack = stack[*,*,0:n_accept-1]
	return, stack
   end
