   ; SKIRT_SMOOTH: median smoothing of the outskirts of a square image.

   ;	The value of each pixel outside a circle centered on the image
   ;	maximum and having a pre-fixed radius is replaced by the median
   ;	computed over a suitable neighborhood, bounded in lower and
   ;	upper radial distance from the center and in azimuthal angle
   ;	around its position. The size of the neighborhood increases with
   ;	the distance from the center and is computed by the auxiliary
   ;	functions RAD_WIDTH and AZI_WIDTH. The radial dependence of the
   ;	box size is modeled by a power law.

   ; INPUT
   ;	image: square image to be smoothed
   ;	[PATCH_SIZE = ]: 'true' size of the image being smoothed, if it
   ;		has been extended for some reason; this optional parameter
   ;		is important for the computation of the default value of
   ;		inner_rad (see below) when the input image has been extended
   ;		before the call to SKIRT_SMOOTH
   ;		(default = image size)
   ;	[INNER_RAD = ]: smoothing is not performed in the inner part of
   ;		the image, of radius inner_rad (default = patch_size / 4)
   ;	[R_WIDTH = ]: radial width (in pixels) of the smoothing box at a
   ;		distance 2*inner_rad from the center (default = inner_rad / 2)
   ;	[A_WIDTH = ]: azimuthal width (in radians) of the smoothing box at a
   ;		distance 2*inner_rad from the center (default = pi / 8)
   ;	[R_EXP = , A_EXP = ]: exponents of the power laws expressing the
   ;		radial dependence of the box radial and azimuthal widths
   ;		(default = 2, 3)
   ;	[/PAD_0]: The outskirts smoothing procedure may produce artifacts
   ;		at the image edge and thus needs an outer 'working space' in
   ;		the form of an additional border, which should then be cut away.
   ;		If this 'working space' is missing, it may be created by padding
   ;		with 0s the image being input to the ouskirts smoothing procedure,
   ;		thus 'forcing' the final image to approach 0 at its border.
   ;		For this purpose, set /PAD_0
   ;	[/SHOW]: display binary image representing the smoothing box as
   ;		the radial distance from the center increases

   ; OUTPUT
   ;	Return smoothed image
   ; NOTE: to display the smoothing neighborhood of each pixel, cancel the
   ;	   semicolon ';' at the beginning of line 119 and compile the file.





   ; RAD_WIDTH, AZI_WIDTH: auxiliary functions to compute radial and
   ;	azimuthal width of the smoothing neighborhood for a given pixel.
   ; INPUT
   ;	r: radial distance of the pixel from the center
   ;	inner_rad: inner radius for median filtering
   ;		(see the keyword INNER_RAD in SKIRT_SMOOTH)
   ;	rad_coeff, azi_coeff: coefficients computed by SKIRT_SMOOTH as
   ;		a function of the optional parameters R_WIDTH and A_WIDTH
   ;	rad_exp, azi_exp: power law exponents
   ; OUTPUT
   ;	Return radial and azimuthal widths in pixel units.
   ;	Divide the azimuthal width by r to obtain the corresponding
   ;	value in angular units (radians).

   FUNCTION rad_width, r, inner_rad, rad_coeff, rad_exp

	return, rad_coeff * ( r / (2.* inner_rad) ) ^rad_exp
   end

   FUNCTION azi_width, r, inner_rad, azi_coeff, azi_exp

	return, azi_coeff * ( r / (2.* inner_rad) ) ^azi_exp
   end





   FUNCTION skirt_smooth, image, $
   					PATCH_SIZE = patch_size, INNER_RAD = inner_rad,		$
   					R_WIDTH = rw, A_WIDTH = aw, R_EXP = re, A_EXP = ae, $
   					PAD_0 = pad_0, SHOW = show

	if  mysize( image, /N_DIM ) ne 2  then  return, image
	s = mysize( image, /DIM )
	if  s[0] ne s[1]  then  return, image
	s = s[0]
	; Define inner and outer radial bounds of the region to be smoothed
	if  n_elements( patch_size ) eq 0  then  patch_size = s
	if  n_elements( inner_rad ) eq 0   then  $
	   inner_rad = round( patch_size / 4. )
	outer_rad = round( patch_size / sqrt(2) )
	; Define default width of filtering box at 2*inner_rad from the center
	if  n_elements( rw ) eq 0  then  rw = inner_rad / 2.
	if  n_elements( aw ) eq 0  then  aw = !pi / 8
	if  n_elements( re ) eq 0  then  re = 2
	if  n_elements( ae ) eq 0  then  ae = 3
	rad_coeff = rw  &  azi_coeff = 2 * inner_rad * aw
	; Initialization of smoothed image
	smooth_image = center_image( image, s / 2, s / 2 )
	if  keyword_set( pad_0 )  then begin
	   s = 2 * s
	   smooth_image = extend_array( smooth_image, s, s, $
	   								X_OFF = s / 4, Y_OFF = s / 4 )
	endif
	if  keyword_set( show )  then  window, XSIZE = 256, YSIZE = 256, /FREE
	saved_image = smooth_image
	; For each pixel, compute distance from the center and azimuthal angle
	x = lindgen( s*s ) mod s  &  y = lindgen( s*s ) / s
	r_distance = round( distance( x, y, s / 2, s / 2 ) )
	azi = angle( s / 2, s / 2, x, y )
	; Iterate on annular regions of increasing radius, from inner to outer
	for  r = inner_rad, outer_rad  do begin
	   ; Identify pixels at this distance from the center
	   wr = where( r_distance eq r, nr )
	   if  nr ne 0  then begin
	      ; Compute radial and azimuthal width of smoothing box
		  w = rad_width( r, inner_rad, rad_coeff, re )
		  l = azi_width( r, inner_rad, azi_coeff, ae ) / r
		  for  n = 0, nr - 1  do begin
		  	 ; Define the smoothing 'box' of each pixel
			 rad_range = abs( r_distance - r ) le w / 2
			 delta_azi = abs( azi - azi[wr[n]] )
			 azi_range = delta_azi le l / 2  or  2*!pi - delta_azi le l / 2
		     neigh = where( ( rad_range and azi_range ) and 1B )
		     if  keyword_set( show ) and $
		     	 x[wr[n]] eq y[wr[n]] and x[wr[n]] gt s / 2  then begin
			    b = bytarr( s, s )  &  b[neigh] = 1
			    tvscl, congrid( b, 256, 256 )
		     endif
			 ; Compute median of box
			 smooth_image[ x[wr[n]], y[wr[n]] ] = $
			    median( saved_image[neigh], /EVEN )
		  endfor
	   endif
	endfor
	if  keyword_set( pad_0 )  then $
	   smooth_image = extract( smooth_image, s / 2, s / 2 )
	return, smooth_image
   end