   ; SPLINE_COEFF: compute the coefficients of a 2-D interpolating spline
   ;	defined over the rectangular domain [lo_x,up_x] x [lo_y,up_y],
   ;	given a set of observations on a rectangular grid of points (x,y).

   ; INPUT
   ;	data: input data to be interpolated
   ;	[DEGREE = ]: spline odd degree (default = 3)
   ; OUTPUT
   ;	coefficients: sx * sy array of spline coefficients
   ;	x_knots, y_knots: x- and y- knots of interpolating spline
   ; INPUT/OUTPUT
   ;	[x, y]: optional vectors of abscissae and ordinates.
   ;		If not set, a default rectangular grid is defined, with size
   ;		sx * sy, where sx and sy are the x- (row) and y- (column) size
   ;		of the data array
   ; 		NOTE: x and y must be in strictly increasing order
   ;	[lo_x, up_x, lo_y, up_y]: bounds of rectangular domain





   ; ASCENDING: given the vector x of size n, check the condition
   ;	x[i] < x[i+1], for i = 0, n - 2.

   FUNCTION ascending, x

	if  mysize( x, /N_DIM ) gt 1  then  return, 0B
	n = n_elements( x )
	check = ( x lt shift( x, -1 ) )[0:n-2] and 1B
	return, min( check )
   end



   ; CHECK_DATA: check input coordinates. If not set, define a default grid.

   PRO check_data, data, x, y, lo_x, up_x, lo_y, up_y, error

	error = mysize( data, /N_DIM ) ne 2  &  if  error  then  return
	sx = n_elements( x )  &  sy = n_elements( y )
	if  sx * sy * $
	    n_elements( lo_x ) * n_elements( up_x ) * $
	    n_elements( lo_y ) * n_elements( up_y ) eq 0  then begin
	   s = mysize( data, /DIM )  &  sx = s[0]  &  sy = s[1]
	   x = sampling_grid( sx, 1, lo_x, up_x )
	   y = sampling_grid( sy, 1, lo_y, up_y )
	endif else $
	   error = min( x ) lt lo_x  or  max( x ) gt up_x  or $
	   		   min( y ) lt lo_y  or  max( y ) gt up_y  or $
	   		   not ascending( x ) or not ascending( y )
	return
   end



   ; COMPUTE_KNOTS: 1-D set of knots for spline interpolation of odd degree.

   FUNCTION compute_knots, x, a, b, degree, error

	n = n_elements( x )
	error = n le degree or degree mod 2 eq 0  &  if  error  then  return, 0
	k = fltarr( n + degree + 1 )
	k[0:degree] = a  &  k[n:n+degree] = b	; additional knots
	if  n gt degree + 1  then $				; interior knots
	   k[degree + 1:n - 1] = x[degree/2 + 1:degree/2 + n - degree - 1]
	return, k
   end



   ; GIVPAR: compute parameters for a Givens transformation.

   PRO givpar, piv, r, c, s

	abs_piv = abs( piv )
	if  abs_piv ge r  then $
	   aux_r = abs_piv * sqrt( 1 + (r/piv)^2 )  else $
	   aux_r = r * sqrt( 1 + (piv/r)^2 )
	c = r / aux_r  &  s = piv / aux_r  &  r = aux_r
	return
   end



   ; G_ROTATE: rotate vectors v1 and v2 applying a Givens rotation (c,s).

   PRO g_rotate, c, s, v1, v2

	u1 = v1  &  u2 = v2
	v1 = c * u1 - s * u2
	v2 = s * u1 + c * u2
	return
   end



   ; GIV_TRANSFORM: rotate observation array a and data array d
   ;	applying Givens rotations.

   PRO giv_transform, a, npt, d, degree

	n_points = n_elements( npt )
	aux_a = a  &  a = a - a
	aux_d = transpose( d )  &  d = aux_d - aux_d
	for  n = 0L, n_points - 1  do begin
	   this = npt[n]  &  va = aux_a[*,n]  &  vd = aux_d[*,n]
	   for  i = 0, degree  do begin
	      piv = va[i]
	      if  piv ne 0  then begin
	         temp = a[0,this]  &  givpar, piv, temp, c, s   &  a[0,this] = temp
	         temp = d[*,this]  &  g_rotate, c, s, vd, temp  &  d[*,this] = temp
	         if  i lt degree  then begin
	            temp = va[i+1:degree]  &  temp1 = a[1:degree-i,this]
	            g_rotate, c, s, temp, temp1
	            va[i+1:degree] = temp  &  a[1:degree-i,this] = temp1
	         endif
	      endif
	      this = this + 1
	   endfor
	endfor
	return
   end



   ; BACK_SUB: solve the set of m linear algebraic systems  a x = b,
   ;	where a is an n*n upper triangular matrix of bandwidth w,
   ;	x and b are m*n arrays.

   FUNCTION back_sub, a, b, w

	s = mysize( b, /DIM )  &  m = s[0]  &  n = s[1]
	x = fltarr( m, n + w - 1 )
	for  i = n - 1, 0, -1  do $
	   x[*,i] = ( b[*,i] -  x[*,i+1:i+w-1] # a[1:w-1,i] ) / a[0,i]
	return, x[*,0:n-1]
   end



   ; SOLVE_SYS: compute the coefficients of the interpolating spline
   ;	as the solution of the linear system  (ay) c (ax)' = data

   FUNCTION solve_sys, ax, nptx, ay, npty, data, degree

	d = data
	; Reduce (ax) to upper triangular form. Apply the same transformation
	; to the data.
	giv_transform, ax, nptx, d, degree
	; Reduce (ay) to upper triangular form. Apply the same transformation
	; to the data.
	giv_transform, ay, npty, d, degree
	; Solve the linear system  (ay) c (ax)' = d
	temp = back_sub( ay, d, degree + 1 )
	c = back_sub( ax, transpose( temp ), degree + 1 )
	return, transpose( c )	; c is (nx) * (ny)
   end



   PRO spline_coeff, data, DEGREE = degree,	coefficients, $
   					 x_knots, y_knots, x, y, lo_x, up_x, lo_y, up_y

	coefficients = 0
	check_data, data, x, y, lo_x, up_x, lo_y, up_y, error
	if  error  then  return
	if  n_elements( degree ) eq 0  then  degree = 3
	if  degree mod 2 eq 0  then  degree = degree + 1
	x_knots = compute_knots( x, lo_x, up_x, degree, error )
	if  error  then  return
	y_knots = compute_knots( y, lo_y, up_y, degree, error )
	if  error  then  return
	b_splines, x, x_knots, degree, ax, nptx
	b_splines, y, y_knots, degree, ay, npty			; delta_t = 2.6
	coefficients = solve_sys( ax, nptx, ay, npty, data, degree )
													; delta_t = 10.6
	return
   end
