   ; STACK_MEDIAN: compute the median of a stack of frames.

   ; INPUT
   ;	stack: stack (cube) of 2-D frames
   ;	[MASK = ]: stack of boolean masks (same size as stack), to exclude
   ;		some pixels of some frames from the computation of the median
   ;	[/POSITIVE]: compute stack median considering only positive pixels;
   ;		if there is no such pixel available, the corresponding value is 0
   ;	[WEIGHTS = ]: array of weights for every frame in the stack
   ;	[N_STEP = ]: weights are transformed to integers, by partitioning
   ;		the range between the minimum and the maximum weight supplied.
   ;		N_STEP is the number of intervals (default N_STEP = 10)
   ; OUTPUT
   ;	Return 2-D median of the stack. Return the input stack if it is
   ;	neither a 3-D cube of frames nor a single 2-D frame.

   FUNCTION stack_median, stack, MASK = mask, POSITIVE = positive, $
   						  WEIGHTS = weights, N_STEP = n_step

	s = size( stack )
	if  s[0] ne 3  then  return, stack
	s1 = s[1]  &  s2 = s[2]  &  n_frames = s[3]
	; Mask some frames?
	masked = n_elements( mask ) ne 0
	; Weighed median?
	weighted = n_elements( weights ) ne 0
	if  weighted  then begin
	   if  n_elements( n_step ) eq 0  then  n_step = 10
	   w = weights > 0
	   w = w / ( max( w ) - min( w ) ) * n_step
	   w = round( w ) > 1
	endif else  w = replicate( 1, n_frames )
	; Compute median
	m = stack[*,*,0]  &  m = m - m
	for  i = 0, s2 - 1  do  for  j = 0, s1 - 1  do begin
	   slice = stack[j,i,*]  &  w_ji = w  &  n = n_frames
	   ; Mask undesired frames
	   if  masked  then begin
	      accept = where( mask[j,i,*] ne 0, n )
	      if  n ne 0  then begin
	         slice = slice[accept]  &  w_ji = w_ji[accept]
	      endif
	   endif
	   ; Retain only positive pixels in the slice
	   if  keyword_set( positive ) and n ne 0  then begin
	      accept = where( slice gt 0, n )
	      if  n eq 0  then begin
	         slice = 0.  &  w_ji = 1
	      endif else begin
	         slice = slice[accept]  &  w_ji = w_ji[accept]
	      endelse
	   endif
	   ; Replicate frames according to their weight
	   if  weighted and n ne 0  then begin
	      temp = replicate( slice[0], w_ji[0] )
	      for  k = 1, n - 1  do $
	         temp = add_elements( temp, replicate( slice[k], w_ji[k] ) )
	      slice = temp
	   endif
	   ; Compute median of pixel [j,i]
	   if  n ne 0  then  m[j,i] = median( slice, /EVEN )
	endfor
	return, m
   end
