   ; STARS: compute an image model given by a sum of shifted scaled replicas
   ;	of a template image. It is possible to create a model with a space-
   ;	variant (SV) template, computed by a user-supplied function.

   ; INPUT
   ;	psf: template
   ;	x, y: positions of shifted replicas of the template
   ;	i: scaling factors
   ;	sx, sy: x- and y- size of output image
   ;	[/NO_NORM]: avoid psf normalization to unitary volume
   ;	[SV = sv_fun]: string, containing the name of the function
   ;		to compute a local SVPSF at (x,y). It must be defined as follows:
   ;		result = sv_fun( psf, x_ref, y_ref, x, y, _EXTRA = extra )
   ;	[X_REF = x_ref, Y_REF = y_ref]: reference position to compute a SVPSF
   ; NOTE: the option of space-variability is not available if the reference
   ;	position (x_ref,y_ref) is not supplied
   ;	[_EXTRA = ]: see IMAGE_SHIFT and SV_FUN (if used)

   ; INPUT/OUTPUT
   ;	[AUX_DATA = ]: auxiliary data used by IMAGE_SHIFT for fractional
   ;		shift of the template. This optional input/output parameter may
   ;		be used when STARS is repeatedly called without changing the
   ;		template. Not available with SV PSF

   ; OUTPUT
   ;	Return the image model


   FUNCTION stars, psf, x, y, i, sx, sy, NO_NORM = no_norm,	  $
   				   SV = sv_fun, X_REF = x_ref, Y_REF = y_ref, $
   				   _EXTRA = extra, AUX_DATA = aux_data

	if  n_params() eq 5  then  sy = sx
	image = fltarr( sx, sy )  &  nstar = n_elements( i )
	if  n_elements( x ) ne n_elements( y ) or n_elements( y ) ne nstar or $
		mysize( psf, /N_DIM ) ne 2  then  return, image
	sv = n_elements( sv_fun ) ne 0 and $
	     n_elements( x_ref ) * n_elements( y_ref ) ne 0
	psfsize = mysize( psf, /DIM )  &  m = get_max( psf )
	for  n = 0L, nstar - 1  do begin
	   ix = round( x[n] )  &  iy = round( y[n] )
	   dx = x[n] - ix	   &  dy = y[n] - iy
	   if  sv  then begin
	      svpsf = call_function( sv_fun, psf, x_ref, y_ref, $
	      						 x[n], y[n], _EXTRA = extra )
	      m = get_max( svpsf )
	      shifted_psf = image_shift( svpsf, dx, dy, _EXTRA = extra )
	   endif else $
	      shifted_psf = image_shift( psf, dx, dy, _EXTRA = extra, $
	   								 AUX_DATA = aux_data )
	   image = psf_here( shifted_psf, image, ix, iy, i[n], $
	   				SSX = psfsize[0], SSY = psfsize[1], LSX = sx, LSY = sy, $
	   				XS = m[0], YS = m[1] )
	endfor
	if  not keyword_set( no_norm )  then  image = image / total( psf )
	return, image
   end
