/*
 * C Language interface.  Solaris.
 * (%M% %I% - %G% %U%)
 */
#include <sys/types.h>
#include <sys/mkdev.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/ioccom.h>
#include <fcntl.h>
#include <string.h>

#include "ugpib.h"


struct ibarg {
	int  size;          /* size of ibarg */
	int  handle;		
        int  ibsta;         /* return status */
        int  iberr;         /* return error code */
        int  ibcnt;         /* return count */
        void *buf;          /* buffer address */
        void *buf1;	    /* buffer address */
        void *buf2;         /* buffer address */
        int  n;             /* count or argument */
        int  n1;            /* count or argument */
        int  n2;            /* count or argument */
};

/* ioctl codes */
#define DIOC    ('d'<<8)

/* the following codes use the form: ioctl(f,IBRD,&ibarg);  */
#define IBDIAG   (DIOC|5) 	/* special */
#define IBTRG   (DIOC|10)
#define IBCLR   (DIOC|11)
#define IBCMD   (DIOC|12)
#define IBXTRC  (DIOC|14)
#define IBRSP   (DIOC|15)
#define IBRPP   (DIOC|16)
#define IBBNA   (DIOC|17)
#define IBWAIT  (DIOC|18)
#define IBONL   (DIOC|19)
#define IBRSV   (DIOC|20)
#define IBGTS   (DIOC|21)
#define IBCAC   (DIOC|22)
#define IBPOKE  (DIOC|23)
#define IBSGNL  (DIOC|24)
#define IBSRE   (DIOC|25)
#define IBRSC   (DIOC|26)
#define IBRD    (DIOC|27)
#define IBWRT   (DIOC|28)
#define IBLOC   (DIOC|29)
#define IBPCT   (DIOC|30)
#define IBSIC   (DIOC|31)
#define IBPPC   (DIOC|33)
#define IBLINES (DIOC|34)
#define IBEOT   (DIOC|35)
#define IBPAD   (DIOC|36)
#define IBSAD   (DIOC|37)
#define IBDMA   (DIOC|38)
#define IBEOS   (DIOC|39)
#define IBTMO   (DIOC|40)
#define IBWRTF  (DIOC|41)
#define IBRDF   (DIOC|42)
#define IBLLO   (DIOC|43)
#define IBSTAT  (DIOC|44)
#define IBLN    (DIOC|45)
#define IBDEV   (DIOC|46)
#define IBIST   (DIOC|32)

/* NI-488.2 calls */
#define ALLSPOLL        (DIOC|50)
#define DEVCLEARLIST    (DIOC|51)
#define ENABLELOCAL     (DIOC|52)
#define ENABLEREMOTE    (DIOC|53)
#define FINDRQS         (DIOC|54)
#define FINDLSTN        (DIOC|55)
#define PASSCONTROL     (DIOC|56)
#define PPOLLCONFIG     (DIOC|57)
#define PPOLLUNCONFIG   (DIOC|58)
#define PPOLL           (DIOC|59)
#define RECEIVE         (DIOC|60)
#define RCVRESPMSG      (DIOC|61)
#define RECEIVESETUP    (DIOC|62)
#define RESETSYS        (DIOC|63)
#define SENDCMDS        (DIOC|64)
#define SENDDATABYTES   (DIOC|65)
#define SENDIFC         (DIOC|66)
#define SENDLIST        (DIOC|67)
#define SENDLLO         (DIOC|68)
#define SENDSETUP       (DIOC|69)
#define SETRWLS         (DIOC|70)
#define TESTSYS         (DIOC|71)
#define TRIGGERLIST     (DIOC|72)
#define DEVCLEAR        (DIOC|73) 
#define READSTATUSBYTE  (DIOC|74)
#define SEND            (DIOC|75)
#define TESTSRQ         (DIOC|76)
#define TRIGGER         (DIOC|77)
#define WAITSRQ         (DIOC|78)

#define IBFIND		(DIOC|131)
#define IBCONFIG	(DIOC|132)
#define IBASK		(DIOC|133)

#define IBWRTA          (DIOC|150)
#define IBRDA           (DIOC|151)
#define IBCMDA          (DIOC|152)
#define IBSTOP		(DIOC|153)
#define IBWRTKEY	(DIOC|154)
#define IBRDKEY		(DIOC|155)

extern errno;
int ibsta, iberr;
int ibcnt;
long ibcntl ;
struct ibarg ibarg;

static int fp_gpib=-1;

/*
 * common function to set up ioctl, make call
 * and set global variables
 */
#ifdef __STDC__
static fn(int code, int f)
#else
static fn(code, f)
#endif
{
	char *path="/dev/gpib";

	if (fp_gpib==-1) {
		fp_gpib=open(path,O_RDWR);
		if (fp_gpib==-1)
			if ((code==IBFIND)||(code==IBDEV))
				return -1;
			else
				return oserr();
	}

	ibarg.size=sizeof(struct ibarg);
	ibarg.handle=f;
	if (ioctl(fp_gpib, code, &ibarg) < 0)
		return oserr();
	ibcnt = ibarg.ibcnt;
	ibcntl = ibcnt ;
	iberr = ibarg.iberr;
	ibsta = ibarg.ibsta;
	if ((code==IBFIND)||(code==IBDEV))
		return ibarg.handle;
	else
		return ibsta;
}

/*
 * operating system error:
 *      error code EDVR indicates OS error code is in ibcnt
 */
static oserr() {
	iberr = EDVR;
	ibcnt = errno;
	ibcntl = ibcnt ;
	return ibsta = ERR|CMPL;
}


#define IB_BDNAME_SIZE 256
#ifdef __STDC__
ibfind  (char *bdname)
#else
ibfind  (bdname) char *bdname;
#endif
{
	struct stat fs;
	char name[IB_BDNAME_SIZE];

	if (*bdname=='/') {
		if (stat(bdname,&fs)!=0) {
			iberr=EDVR;
			ibsta=ERR|CMPL;
			return -1;
		}
	} else {
		name[0]='/';
		name[1]='d';
		name[2]='e';
		name[3]='v';
		name[4]='/';
		strncpy(&name[5],bdname,IB_BDNAME_SIZE-5);
		name[IB_BDNAME_SIZE-1]='0';
		if (stat(name,&fs)!=0) {
			iberr=EDVR;
			ibsta=ERR|CMPL;
			return -1;
		}
	}
	if (S_ISCHR(fs.st_mode)) {
		ibarg.n=fs.st_rdev;
		return fn(IBFIND,0);
	} else {
		iberr=EDVR;
		ibsta=ERR|CMPL;
		return -1;
	}

}

#ifdef __STDC__
ibdev (int bindex, int pad, int sad, int tmo, int eot, int eos)
#else
ibdev (bindex,pad,sad,tmo,eot,eos)
#endif
{
	int info[6];
	info[0]=bindex;
	info[1]=pad;
	info[2]=sad;
	info[3]=tmo;
	info[4]=eot;
	info[5]=eos;
	ibarg.n=sizeof(info);
	ibarg.buf=&info[0];
	return fn(IBDEV,bindex);
}

/*
 * ibbna changes a device's access board.
 */
#ifdef __STDC__
ibbna(int g, char *bname)
#else
ibbna(g,bname) char bname[]; 
#endif
{
	struct stat fs;
	char name[IB_BDNAME_SIZE];

	if (bname[0]=='/') {
		if (stat(bname,&fs)!=0) {
			iberr=EARG;
			ibsta=ERR|CMPL;
			return ibsta;
		}
	} else {
		name[0]='/';
		name[1]='d';
		name[2]='e';
		name[3]='v';
		name[4]='/';
		strncpy(&name[5],bname,IB_BDNAME_SIZE-5);
		name[IB_BDNAME_SIZE-1]='0';
		if (stat(name,&fs)!=0) {
			iberr=EARG;
			ibsta=ERR|CMPL;
			return ibsta;
		}
	}
	if (S_ISCHR(fs.st_mode)) {
		ibarg.n=fs.st_rdev;
		return fn(IBBNA,g);
	} else {
		iberr=EARG;
		ibsta=ERR|CMPL;
		return ibsta;
	}
}


#ifdef __STDC__
ibdiag (int g, void *buf, long cnt)
#else
ibdiag  (g, buf, cnt) void *buf;
#endif
{ 
	ibarg.buf=buf;
	ibarg.n=cnt;
	return fn(IBDIAG, g); 
}

#ifdef __STDC__
ibrd (int g, void *buf, long cnt)
#else
ibrd  (g, buf, cnt) void *buf;
#endif
{ 
	ibarg.buf=buf;
	ibarg.n=cnt;
	return fn(IBRD, g); 
}

#ifdef __STDC__
ibrda (int g, void *buf, long cnt)
#else
ibrda  (g, buf, cnt) char *buf;
#endif
{ 
	ibarg.buf=buf;
	ibarg.n=cnt;
	return fn(IBRDA, g); 
}

#ifdef __STDC__
ibwrt (int g, void *buf, long cnt)
#else
ibwrt  (g, buf, cnt) void *buf;
#endif
{ 
	ibarg.buf=buf;
	ibarg.n=cnt;
	return fn(IBWRT, g); 
}

#ifdef __STDC__
ibwrta (int g, void *buf, long cnt)
#else
ibwrta  (g, buf, cnt) void *buf;
#endif
{ 
	ibarg.buf=buf;
	ibarg.n=cnt;
	return fn(IBWRTA, g); 
}

#ifdef __STDC__
ibcmd (int g, void *buf, long cnt)
#else
ibcmd (g, buf, cnt) void *buf;
#endif
{ 
	ibarg.buf=buf;
	ibarg.n=cnt;
	return fn(IBCMD, g); 
}

#ifdef __STDC__
ibcmda (int g, void *buf, long cnt)
#else
ibcmda (g, buf, cnt) void *buf;
#endif
{ 
	ibarg.buf=buf;
	ibarg.n=cnt;
	return fn(IBCMDA, g); 
}

#ifdef __STDC__
ibstop (int g)
#else
ibstop (g)
#endif
{ 
	return fn(IBSTOP, g); 
}

#ifdef __STDC__
ibtrg (int g)
#else
ibtrg (g)
#endif
{ 
	return fn(IBTRG, g); 
}

#ifdef __STDC__
ibclr (int g)
#else
ibclr (g)
#endif
{ 
	return fn(IBCLR, g); 
}

#ifdef __STDC__
ibpct (int g)
#else
ibpct (g)
#endif
{ 
	return fn(IBPCT, g); 
}

#ifdef __STDC__
ibllo (int g)
#else
ibllo (g)
#endif
{ 
	return fn(IBLLO, g); 
}

#ifdef __STDC__
ibloc (int g)
#else
ibloc (g)
#endif
{ 
	return fn(IBLOC, g); 
}

#ifdef __STDC__
ibsic (int g)
#else
ibsic (g)
#endif
{ 
	return fn(IBSIC, g); 
}

#ifdef __STDC__
ibrsp (int g, char *buf)
#else
ibrsp (g, buf) char *buf;
#endif
{ 
	ibarg.buf=buf;
	return fn(IBRSP, g); 
}

#ifdef __STDC__
ibrpp (int g, char *buf)
#else
ibrpp (g, buf) char * buf;
#endif
{ 
	ibarg.buf=buf;
	return fn(IBRPP, g); 
}

#ifdef __STDC__
iblines (int g, short *buf)
#else
iblines (g, buf) short * buf;
#endif
{ 
	ibarg.buf=buf;
	return fn(IBLINES, g); 
}

#ifdef __STDC__
ibwait (int g, int mask)
#else
ibwait (g, mask) int mask;
#endif
{ 
	ibarg.n=mask;
	return fn(IBWAIT, g); 
}

#ifdef __STDC__
ibrsv (int g, int v)
#else
ibrsv (g, v) int v;
#endif
{ 
	ibarg.n=v;
	return fn(IBRSV, g); 
}

#ifdef __STDC__
ibsre (int g, int v)
#else
ibsre (g, v) int v;
#endif
{ 
	ibarg.n=v;
	return fn(IBSRE, g); 
}

#ifdef __STDC__
ibonl (int g, int v)
#else
ibonl (g, v) int v;
#endif
{ 
	ibarg.n=v;
	return fn(IBONL, g); 
}

#ifdef __STDC__
ibgts (int g, int v)
#else
ibgts (g, v) int v;
#endif
{ 
	ibarg.n=v;
	return fn(IBGTS, g); 
}

#ifdef __STDC__
ibcac (int g, int v)
#else
ibcac (g, v) int v;
#endif
{ 
	ibarg.n=v;
	return fn(IBCAC, g); 
}

#ifdef __STDC__
ibpoke (int g, int v, int val)
#else
ibpoke (g, v, val) int v;
#endif
{ 
	ibarg.n=v;
	ibarg.n1 = val ;
	return fn(IBPOKE, g); 
}

#ifdef __STDC__
ibsgnl (int g, int v)
#else
ibsgnl (g, v) int v;
#endif
{ 
	ibarg.n=v;
	return fn(IBSGNL, g); 
}


#ifdef __STDC__
ibrsc (int g, int v)
#else
ibrsc (g, v) int v;
#endif
{ 
	ibarg.n=v;
	return fn(IBRSC, g); 
}

#ifdef __STDC__
ibppc (int g, int v)
#else
ibppc (g, v) int v;
#endif
{ 
	ibarg.n=v;
	return fn(IBPPC, g); 
}

#ifdef __STDC__
ibist (int g, int v)
#else
ibist (g, v) int v;
#endif
{ 
	ibarg.n=v;
	return fn(IBIST, g); 
}

#ifdef __STDC__
ibeot (int g, int v)
#else
ibeot (g, v) int v;
#endif
{ 
	ibarg.n=v;
	return fn(IBEOT, g); 
}

#ifdef __STDC__
ibpad (int g, int v)
#else
ibpad (g, v) int v;
#endif
{ 
	ibarg.n=v;
	return fn(IBPAD, g); 
}

#ifdef __STDC__
ibsad (int g, int v)
#else
ibsad (g, v) int v;
#endif
{ 
	ibarg.n=v;
	return fn(IBSAD, g); 
}

#ifdef __STDC__
ibdma (int g, int v)
#else
ibdma (g, v) int v;
#endif
{ 
	ibarg.n=v;
	return fn(IBDMA, g); 
}

#ifdef __STDC__
ibeos (int g, int v)
#else
ibeos (g, v) int v;
#endif
{ 
	ibarg.n=v;
	return fn(IBEOS, g); 
}

#ifdef __STDC__
ibtmo (int g, int v)
#else
ibtmo (g, v) int v;
#endif
{ 
	ibarg.n=v;
	return fn(IBTMO, g); 
}

#ifdef __STDC__
ibconfig(int g, int option, int v) 
#else
ibconfig(g, option, v) int option, v;
#endif
{
  	ibarg.n1 = option;
  	ibarg.n2 = v;
  	return fn(IBCONFIG,g);
}

#ifdef __STDC__
ibask(int g, int option, int *ret) 
#else
ibask(g, option, ret) int option, *ret;
#endif
{
        int result;

  	ibarg.n1 = option;
  	result= fn(IBASK,g);
  	*ret=ibarg.n2;
        return result;
}

	
#ifdef __STDC___
ibln (int g, int pad, int sad,short *listen) 
#else
ibln (g,pad,sad,listen) short *listen;
#endif
{
	ibarg.n=MakeAddr(pad,sad);	
	ibarg.buf=listen;
	return fn(IBLN, g);
}


/*
 * file I/O routines:
 */

#define FBSIZE  512             /* size of buffer for file I/O */
static char buf[FBSIZE*2];

/*
 * write file.
 * double buffering is used not for improved throughput, but
 * for determining the last buffer before it is sent.
 */
ibwrtf(g, file) char file[]; {
	int f, i, cnt[2], total=0;
	ushort wrtfail=0, sta=0, err=0;
	char eot;

	if ((f=open(file, O_RDONLY, 0)) < 0)
		return oserr();
	ibeot(g,0);
	eot=iberr;
	i= 0;
	if ((cnt[0]=read(f, buf, FBSIZE)) < 0) {
		oserr();
		goto bad;
	}
	while (cnt[i]) {
		i= !i ;
		if ((cnt[i]=read(f, &buf[i*FBSIZE], FBSIZE)) < 0) {
			oserr();
			goto bad;
		}
		if (cnt[i] == 0) { /* then the previous read reached the eof */
			ibeot(g,eot);
			if (ibsta&ERR)    /* restore eot mode */
				goto bad;
		}
		if ((ibwrt(g, &buf[!i * FBSIZE], cnt[!i])) & ERR) {
			wrtfail = 1;
			sta = ibsta;
			err = iberr;
			goto bad;
		}
		total += ibcnt;
	}
	goto ret;
bad:    
	ibeot(g,eot);
	if (wrtfail) {		/* restore the write fail status and err */
		ibsta = sta;
		iberr = err;
	}
ret:    
	close(f);
	ibcnt = total;
	return ibsta;
}

/*
 * read file.
 */
ibrdf(g, file) char file[]; {
	int f, total=0;

	if ((f=creat(file, 0644)) < 0) {
		iberr = EFSO;
		ibcnt = 0;
		ibsta = ERR|CMPL;
		return ibsta;
	}
	do {    if ((ibrd(g, buf, FBSIZE*2) & ERR) && (iberr == EDVR) )
			goto bad;
		if (write (f, buf, ibcnt) != ibcnt)
			goto bad;
		total += ibcnt;
	} while ((ibsta & (ERR|END)) == 0);
bad:    close(f);;
	ibcnt = total;
	return ibsta;
}

#ifdef __STDC__
ibxtrc  (int g, void *buf, long cnt)
#else
ibxtrc  (g,buf,cnt)
char *buf;
#endif
{
	ibarg.n=cnt;
	ibarg.buf=buf;
	return fn(IBXTRC, g);
}

#ifdef _STDC_
ibwrtkey(int g, void *buf, long cnt)
#else
ibwrtkey(g,buf,cnt)
int g ;
char *buf;
int cnt ;
#endif
{
	ibarg.n = cnt ;
	ibarg.buf = buf ;
	return fn(IBWRTKEY, g) ;
}


#ifdef _STDC_ 
ibrdkey(int g, void *buf, long cnt) 
#else 
ibrdkey(g,buf,cnt) 
int g ; 
char *buf;
int cnt ; 
#endif 
{
        ibarg.n = cnt ; 
        ibarg.buf = buf ; 
        return fn(IBRDKEY, g) ; 
} 

/*********************************************************
 * NI-488.2 functions 
 *********************************************************/

#ifdef __STDC__
void AllSpoll(int b, Addr4882_t *taddrs, short *res)
#else
void AllSpoll(b,taddrs,res) int b; Addr4882_t *taddrs; short *res; 
#endif
{
	ibarg.buf1 = taddrs;
	ibarg.buf2 = res;
	fn(ALLSPOLL,b);
	return ;
} 

#ifdef __STDC__
void DevClearList(int b, Addr4882_t *addrlist)
#else
void DevClearList(b,addrlist) Addr4882_t *addrlist; 
#endif
{
	ibarg.buf1 = addrlist;
	fn(DEVCLEARLIST,b);
	return ;
}

#ifdef __STDC__
void DevClear(int b, Addr4882_t addr)
#else
void DevClear(b,addr) int b; Addr4882_t addr;
#endif
{
	ibarg.n = addr;
	fn(DEVCLEAR,b);
	return ;
}		

#ifdef __STDC__
void EnableLocal(int b, Addr4882_t *laddrs)
#else
void EnableLocal(b,laddrs) Addr4882_t *laddrs;
#endif
{
	ibarg.buf1 = laddrs;
	fn(ENABLELOCAL,b);
	return ;
}

#ifdef __STDC__
void EnableRemote(int b,  Addr4882_t *laddrs)
#else
void EnableRemote(b,laddrs)  Addr4882_t *laddrs;
#endif
{
	ibarg.buf1 = laddrs;
	fn(ENABLEREMOTE,b);
	return ;
}

#ifdef __STDC__
void FindRQS(int b, Addr4882_t *taddrs, short *dev_stat)
#else
void FindRQS(b,taddrs,dev_stat) Addr4882_t *taddrs; short *dev_stat;
#endif
{
	int ret;

	ibarg.buf1 = taddrs;
	ret = fn(FINDRQS,b);
	*dev_stat = (short)ibarg.n;
	return ;
} 

#ifdef __STDC__
void FindLstn(int b,Addr4882_t *addrlist, Addr4882_t *reslist, int limit)
#else
void FindLstn(b,addrlist,reslist,limit) Addr4882_t *addrlist; Addr4882_t *reslist; int limit;
#endif
{
	ibarg.buf1 = addrlist;
	ibarg.buf2 = reslist;
	ibarg.n = limit;
	fn(FINDLSTN,b);
	return ;
} 

#ifdef __STDC__
void PassControl(int b, Addr4882_t talker)
#else
void PassControl(b,talker) Addr4882_t talker;
#endif
{
	ibarg.n = talker;
	fn(PASSCONTROL,b);
	return ;
}	

#ifdef __STDC__
void PPollConfig(int b, Addr4882_t laddr, int  dataline, int  sense)
#else
void PPollConfig(b,laddr,dataline,sense) Addr4882_t laddr; int dataline; int sense;
#endif
{
	ibarg.n = laddr;
	ibarg.n1 = dataline;
	ibarg.n2 = sense;
	fn(PPOLLCONFIG,b);
	return ;
}

#ifdef __STDC__
void PPollUnconfig(int b,Addr4882_t *laddrs)
#else
void PPollUnconfig(b,laddrs) Addr4882_t *laddrs;
#endif
{
	ibarg.buf1 = laddrs;
	fn(PPOLLUNCONFIG,b);
}	

#ifdef __STDC__
void PPoll(int b, short *res_ptr)
#else
void PPoll(b,res_ptr) short *res_ptr;
#endif
{
	int ret;

	ret=fn(PPOLL,b);
	*res_ptr = (ushort)ibarg.n;
	return ;
}

#ifdef __STDC__
void ReadStatusByte(int b,Addr4882_t talker, short *result)
#else
void ReadStatusByte(b,talker,result)Addr4882_t talker, *result;
#endif
{
	int ret;
	ibarg.n=talker;
	ret=fn(READSTATUSBYTE,b);
	*result = (ushort)ibarg.n1;
	return ;
} 

#ifdef __STDC__
void Receive(int b,Addr4882_t talker, void *buf, long cnt, int eotmode)
#else
void Receive(b,talker,buf,cnt,eotmode) Addr4882_t talker; void *buf; long cnt; int eotmode;
#endif
{
	ibarg.n1 = talker;
	ibarg.buf = buf;
	ibarg.n = cnt;
	ibarg.n2 = eotmode;
	fn(RECEIVE,b);
	return ;
}

#ifdef __STDC__
void RcvRespMsg	(int b, void *buf, long cnt, int eotmode)
#else
void RcvRespMsg	(b,buf,cnt,eotmode) void *buf;long cnt; int eotmode;
#endif
{
	ibarg.buf = buf;
	ibarg.n = cnt;
	ibarg.n1 = eotmode;
	fn(RCVRESPMSG,b);
}

#ifdef __STDC__
void ReceiveSetup(int b, Addr4882_t talker)
#else
void ReceiveSetup(b,talker) Addr4882_t talker;
#endif
{
	ibarg.n = talker;
	fn(RECEIVESETUP,b);
	return ;
}	

#ifdef __STDC__
void SendList(int b,Addr4882_t *listeners, void *buf, long cnt, int eotmode)
#else
void SendList(b,listeners,buf,cnt,eotmode)
Addr4882_t listeners[]; char *buf; int cnt; int eotmode;
#endif
{
	ibarg.buf1 = listeners;
	ibarg.buf = buf;
	ibarg.n = cnt;
	ibarg.n2 = eotmode;
	 fn(SENDLIST,b);
	return ;
}	

#ifdef __STDC__
void Send(int b,Addr4882_t listener, void *buf, long cnt, int eotmode)
#else
void Send(b,listener,buf,cnt,eotmode) Addr4882_t listener; void *buf; long cnt; int eotmode;
#endif
{
	ibarg.n1 = listener;
	ibarg.buf = buf;
	ibarg.n = cnt;
	ibarg.n2 = eotmode;
	fn(SEND,b);
	return ;
}	

#ifdef __STDC__
void SendCmds(int b, void *buf, long cnt)
#else
void SendCmds(b,buf,cnt) void *buf; long cnt;
#endif
{
	ibarg.buf = buf;
	ibarg.n = cnt;
	fn(SENDCMDS,b);
	return ;
}	

#ifdef __STDC__
void SendDataBytes(int b, void *buf, long cnt, int eotmode)
#else
void SendDataBytes(b,buf,cnt,eotmode) void *buf; long cnt; int eotmode;
#endif
{
	ibarg.buf = buf;
	ibarg.n = cnt;
	ibarg.n2 = eotmode;
	fn(SENDDATABYTES,b);
	return ;
}

#ifdef __STDC__
void SendIFC(int b)
#else
void SendIFC(b)
#endif
{
	fn(SENDIFC,b);
	return ;
}	

#ifdef __STC__
void ResetSys(int b,Addr4882_t *laddrs)
#else
void ResetSys(b,laddrs) Addr4882_t  *laddrs;
#endif
{
	ibarg.buf1 = laddrs;
	fn(RESETSYS,b);
	return ;
}

#ifdef __STDC__
void SendLLO(int b)
#else
void SendLLO(b)
#endif
{
	fn(SENDLLO,b);
	return ;
}	

#ifdef __STDC__
void SendSetup(int b,Addr4882_t *listen)
#else
void SendSetup(b,listen) Addr4882_t *listen;
#endif
{
	ibarg.buf1 = listen;
	fn(SENDSETUP,b);
	return ;
}	

#ifdef __STDC__
void SetRWLS(int b,Addr4882_t *laddrs)
#else
void SetRWLS(b,laddrs) Addr4882_t *laddrs;
#endif
{
	ibarg.buf1 = laddrs;
	fn(SETRWLS,b);
	return ;
}	

#ifdef __STDC__
void TestSRQ(int b, short *result)
#else
void TestSRQ(b,result) short *result;
#endif
{
	int ret;
	ret=fn(TESTSRQ,b);
	*result=ibarg.n;
	return;
}		

#ifdef __STDC__
void TestSys(int b, Addr4882_t *addrs, short *result)
#else
void TestSys(b,addrs,result) Addr4882_t *addrs; short *result;
#endif
{
	ibarg.buf1 = addrs;
	ibarg.buf2 = result;
	fn(TESTSYS,b);
	return ;
}	

#ifdef __STDC__
void TriggerList(int b,Addr4882_t *laddrs)
#else
void TriggerList(b,laddrs)Addr4882_t  *laddrs;
#endif
{
	ibarg.buf1 = laddrs;
	fn(TRIGGERLIST,b);
	return ;
}	

#ifdef __STDC__
void Trigger(int b,Addr4882_t laddr)
#else
void Trigger(b,laddr) Addr4882_t laddr;
#endif
{
	ibarg.n=laddr;
	fn(TRIGGER,b);
	return ;
}	

#ifdef __STDC__
void WaitSRQ(int b,short *result)
#else
void WaitSRQ(b,result)short *result;
#endif
{
	int ret;
	ret=fn(WAITSRQ,b);
	*result = ibarg.n;
	return;
}
