/*
 * User GPIB Include File
 * (%M% %I% - %G% %U%)
 */
#ifndef NI_UGPIB_H
#define NI_UGPIB_H


#include <sys/types.h>

typedef ushort Addr4882_t;

extern int ibsta,iberr;
extern int ibcnt;
extern long ibcntl;


/* GPIB status bit vector */
#define ERR     (1<<15) /* Error detected */
#define TIMO    (1<<14) /* timeout */
#define END     (1<<13) /* EOI or eos detected */
#define SRQI    (1<<12) /* SRQ detected by CIC */
#define RQS     (1<<11) /* Device requires service */
#define CMPL    (1<<8)  /* DMA completed (SH/AH synch'd) */
#define LOK     (1<<7)  /* local lockout state */
#define REM     (1<<6)  /* remote state */
#define CIC     (1<<5)  /* controller-in-charge */
#define ATN     (1<<4)  /* attention asserted */
#define TACS    (1<<3)  /* talker active */
#define LACS    (1<<2)  /* listener active */
#define DTAS    (1<<1)  /* device trigger state */
#define DCAS    (1<<0)  /* device clear state */

/* Error messages */
#define EDVR  0  /* System error */
#define ECIC  1  /* Not CIC (or lost CIC during command) */
#define ENOL  2  /* write detected no listeners */
#define EADR  3  /* board not addressed correctly */
#define EARG  4  /* bad argument to function call */
#define ESAC  5  /* function requires board to be SAC */
#define EABO  6  /* asynchronous operation was aborted */
#define ENEB  7  /* non-existent board */
#define EDMA  8  /* DMA hardware error detected */
#define EOIP 10  /* new I/O attempted with old I/O in progress  */
#define ECAP 11  /* no capability for intended opeation         */
#define EFSO 12  /* file system operation error                 */
#define EBUS 14  /* bus error */
#define ESTB 15  /* serial poll queue overflow */
#define ESRQ 16  /* SRQ line 'stuck' on */
/*488.2*/
#define ETAB 20  /* The return buffer is full */
#define ELCK 21  /* board or address is locked by another process */

/* GPIB commands */
#define GTL	0001     /* Go To Local                    */
#define SDC	0004     /* Selected Device Clear          */
#define PPC	0005     /* Parallel Poll Configure        */
#define GET	0010     /* Group Execute Trigger          */
#define TCT	0011     /* Take Control                   */
#define LLO	0021     /* Local Lock Out                 */
#define DCL	0024     /* Device Clear                   */
#define PPU	0025     /* Parallel Poll Unconfigure      */
#define SPE	0030     /* Serial Poll Enable             */
#define SPD	0031     /* Serial Poll Disable            */
#define UNL	0077     /* Unlisten                       */
#define UNT	0137     /* Untalk                         */
#define PPD 	0x70     /* Parallel Poll Disable          */


#define LAD     0040     /* Listen address mask	 	   */
#define TAD     0100     /* Talk address mask 		   */

/* Timeout values and meanings                          */
#define TNONE    0   /* Infinite timeout (disabled)     */
#define T10us    1   /* Timeout of 10 us (ideal)        */
#define T30us    2   /* Timeout of 30 us (ideal)        */
#define T100us   3   /* Timeout of 100 us (ideal)       */
#define T300us   4   /* Timeout of 300 us (ideal)       */
#define T1ms     5   /* Timeout of 1 ms (ideal)         */
#define T3ms     6   /* Timeout of 3 ms (ideal)         */
#define T10ms    7   /* Timeout of 10 ms (ideal)        */
#define T30ms    8   /* Timeout of 30 ms (ideal)        */
#define T100ms   9   /* Timeout of 100 ms (ideal)       */
#define T300ms  10   /* Timeout of 300 ms (ideal)       */
#define T1s     11   /* Timeout of 1 s (ideal)          */
#define T3s     12   /* Timeout of 3 s (ideal)          */
#define T10s    13   /* Timeout of 10 s (ideal)         */
#define T30s    14   /* Timeout of 30 s (ideal)         */
#define T100s   15   /* Timeout of 100 s (ideal)        */
#define T300s   16   /* Timeout of 300 s (ideal)        */
#define T1000s  17   /* Timeout of 1000 s (maximum)     */

/* iblines constants */
#define ValidDAV          0x1        
#define ValidNDAC         0x2       
#define ValidNRFD         0x4      
#define ValidIFC          0x8     
#define ValidREN         0x10   
#define ValidSRQ         0x20  
#define ValidATN         0x40 
#define ValidEOI         0x80
#define BusDAV          0x100  
#define BusNDAC         0x200 
#define BusNRFD         0x400     
#define BusIFC          0x800    
#define BusREN         0x1000  
#define BusSRQ         0x2000 
#define BusATN         0x4000
#define BusEOI         0x8000

/* fixed_info_struct definition */
struct fixed_info {
    unsigned char	handler_type;
    unsigned char	board_type;
    unsigned char	handler_version[4];
    unsigned int	handler_segment;
    unsigned int	handler_size;
};

/*
 *  This macro can be used to easily create an entry in address list
 *  that is required by many of the 488.2 functions.  An address list is
 *  just an array of unsigned integers.  The primary address goes in the
 *  lower 8-bits and the secondary address goes in the upper 8-bits.
 */
#define MakeAddr(pad, sad) ((Addr4882_t)(((u_char)(pad)) | (((u_char)(sad))<<8)))

/*
 *  The following two macros are used to "break apart" an address list
 *  entry.  They take an unsigned integer and return either the primary
 *  or secondary address stored in the integer.
 */
#define GetPAD(val)    ((u_char)(val))
#define GetSAD(val)    ((u_char)((val) >> 8))


/*
 *  This value is used to terminate an address list.  It should be
 *  assigned to the last entry.
 */
#define NOADDR          (Addr4882_t)0xFFFF


/* Secondary address constant used by IBLN() */
#define NO_SAD	       0	/* No secondary address */
#define ALL_SAD       0xFF	/* send all secondary addresses */


/* These are the values that are used by the Send 488.2 command. */
#define NULLend (unsigned int)0x00  /* Do nothing at the end of a transfer.*/
#define NLend   (unsigned int)0x01  /* Send NL with EOI after a transfer.  */
#define DABend  (unsigned int)0x02  /* Send EOI with the last DAB.         */


/* This value is used by the 488.2 Receive command.
 */
#define STOPend     (unsigned int)0x0100


#define IbcPAD		0x0001
#define IbcSAD		0x0002
#define IbcTMO		0x0003
#define IbcEOT		0x0004
#define IbcPPC		0x0005
#define IbcREADDR	0x0006
#define IbcAUTOPOLL	0x0007
#define IbcCICPROT	0x0008
#define IbcIRQ		0x0009
#define IbcSC		0x000a
#define IbcSRE		0x000b
#define IbcEOSrd	0x000c
#define IbcEOSwrt	0x000d
#define IbcEOScmp	0x000e
#define IbcEOSchar	0x000f
#define IbcPP2		0x0010
#define IbcTIMING	0x0011
#define IbcDMA		0x0012
#define IbcReadAdjust	0x0013
#define IbcWriteAdjust	0x0014
#define IbcEventQueue	0x0015
#define IbcSPollBit	0x0016
#define IbcSendLLO	0x0017
#define IbcSPollTime	0x0018
#define IbcPPollTime	0x0019
#define IbcNoEndBitOnEos 0x001a
#define IbcUnAddr	0x001b
#define IbcSignalNumber	0x001c

#define IbaPAD		IbcPAD
#define IbaSAD		IbcSAD
#define IbaTMO		IbcTMO
#define IbaEOT		IbcEOT
#define IbaPPC		IbcPPC
#define IbaREADDR	IbcREADDR
#define IbaAUTOPOLL	IbcAUTOPOLL
#define IbaCICPROT	IbcCICPROT
#define IbaIRQ		IbcIRQ	
#define IbaSC		IbcSC
#define IbaSRE		IbcSRE
#define IbaEOSrd	IbcEOSrd
#define IbaEOSwrt	IbcEOSwrt
#define IbaEOScmp	IbcEOScmp
#define IbaEOSchar	IbcEOSchar	
#define IbaPP2		IbcPP2	
#define IbaTIMING	IbcTIMING
#define IbaDMA		IbcDMA	
#define IbaReadAdjust	IbcReadAdjust
#define IbaWriteAdjust	IbcWriteAdjust
#define IbaEventQueue	IbcEventQueue
#define IbaSPollBit	IbcSPollBit
#define IbaSendLLO	IbcSendLLO
#define IbaSPollTime	IbcSPollTime
#define IbaPPollTime	IbcPPollTime
#define IbaNoEndBitOnEos IbcNoEndBitOnEos
#define IbaUnAddr	IbcUnAddr
#define IbaSignalNumber	IbcSignalNumber

#define IbaBNA		0x0200
#define IbaBaseAddr	0x0201
#define IbaDmaChannel	0x0202
#define IbaIrqLevel	0x0203
#define IbaBaud		0x0204
#define IbaParity	0x0205
#define IbaStopBits	0x0206
#define IbaDataBits	0x0207
#define IbaComPort	0x0208
#define	IbaComIrqLevel	0x0209
#define IbaComPortBase	0x020a



extern int ibask(int handle, int option, int *retval);
extern int ibbna(int handle, char *bdname);
extern int ibcac(int handle, int v);
extern int ibclr(int handle);
extern int ibcmd(int handle, void *buffer, long cnt);
extern int ibcmda(int handle, void *buffer, long cnt);
extern int ibconfig(int handle, int option, int value);
extern int ibdev(int boardID, int pad, int sad, int tmo, int eot, int eos);
extern int ibdma(int handle, int v);
extern int ibeos(int handle, int v);
extern int ibeot(int handle, int v);
extern int ibevent(int handle, short *event);
extern int ibfind(char *bdname);
extern int ibgts(int handle, int v);
extern int ibist(int handle, int v);
extern int iblines(int handle, short *lines);
extern int ibln(int handle, int padval, int sadval, short *listenflag);
extern int ibloc(int handle);
extern int ibonl(int handle, int v);
extern int ibpad(int handle, int v);
extern int ibpct(int handle);
extern int ibppc(int handle, int v);
extern int ibrd(int handle, void *buffer, long cnt);
extern int ibrda(int handle, void *buffer, long cnt);
extern int ibrdf(int handle, char *flname);
extern int ibrpp(int handle, char *ppr);
extern int ibrsc(int handle, int v);
extern int ibrsp(int handle, char *spr);
extern int ibrsv(int handle, int v);
extern int ibsad(int handle, int v);
extern int ibsic(int handle);
extern int ibsre(int handle, int v);
extern int ibsrq(void (*func)(void));
extern int ibstop(int handle);
extern int ibtmo(int handle, int v);
extern int ibtrap(int  mask, int mode);
extern int ibtrg(int handle);
extern int ibwait(int handle, int mask);
extern int ibwrt(int handle, void *buffer, long cnt);
extern int ibwrta(int handle, void *buffer, long cnt);
extern int ibwrtf(int handle, char *flname);
extern int ibpoke(int handle, int option, int value);
extern int ibdiag(int handle, void *buffer, long cnt);
extern int ibxtrc(int handle, void *buffer, long cnt);
extern int ibwrtkey(int handle, void *buffer, long  cnt);
extern int ibrdkey(int handle, void *buffer, long cnt);

extern void AllSpoll(int boardID, Addr4882_t *addrlist, short *resultlist);
extern void DevClear(int boardID, Addr4882_t address);
extern void DevClearList(int boardID, Addr4882_t *addrlist);
extern void EnableLocal(int boardID, Addr4882_t *addrlist);
extern void EnableRemote(int boardID, Addr4882_t *addrlist);
extern void FindLstn(int boardID, Addr4882_t *padlist, Addr4882_t *resultlist, int limit);
extern void FindRQS(int boardID, Addr4882_t *addrlist, short *result);
extern void PPoll(int boardID, short *result);
extern void PPollConfig(int boardID, Addr4882_t address, int dataLine, int lineSense);
extern void PPollUnconfig(int boardID, Addr4882_t *addrlist);
extern void PassControl(int boardID, Addr4882_t address);
extern void RcvRespMsg(int boardID, void *buffer, long cnt, int termination);
extern void ReadStatusByte(int boardID, Addr4882_t address, short *result);
extern void Receive(int boardID, Addr4882_t address, void *buffer, long cnt, int termination);
extern void ReceiveSetup(int boardID, Addr4882_t address);
extern void ResetSys(int boardID, Addr4882_t *addrlist);
extern void Send(int boardID, Addr4882_t address, void *buffer, long datacnt, int eotmode);
extern void SendCmds(int boardID, void *buffer, long cnt);
extern void SendDataBytes(int boardID, void *buffer, long cnt, int eotmode);
extern void SendIFC(int boardID);
extern void SendLLO(int boardID);
extern void SendList(int boardID, Addr4882_t *addrlist, void *buffer, long datacnt, int eotmode);
extern void SendSetup(int boardID, Addr4882_t *addrlist);
extern void SetRWLS(int boardID, Addr4882_t *addrlist);
extern void TestSRQ(int boardID, short *result);
extern void TestSys(int boardID, Addr4882_t *addrlist, short *resultlist);
extern void Trigger(int boardID, Addr4882_t address);
extern void TriggerList(int boardID, Addr4882_t *addrlist);
extern void WaitSRQ(int boardID, short *result);


#endif /*ifdef NI_UGPIB_H*/
