/*
 * C Language interface.  Solaris.
 * (%M% %I% - %G% %U%)
 */
#include <sys/types.h>
#include <sys/mkdev.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/ioccom.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>

#include "ugpib.h"




struct ibarg {
	int  size;          /* size of ibarg */
	int  handle;		
        int  ibsta;         /* return status */
        int  iberr;         /* return error code */
        int  ibcnt;         /* return count */
        void *buf;          /* buffer address */
        void *buf1;	    /* buffer address */
        void *buf2;         /* buffer address */
        int  n;             /* count or argument */
        int  n1;            /* count or argument */
        int  n2;            /* count or argument */
};

/* ioctl codes */
#define DIOC    ('d'<<8)

/* the following codes use the form: ioctl(f,IBRD,&ibarg);  */
#define IBDIAG   (DIOC|5) 	/* special */
#define IBTRG   (DIOC|10)
#define IBCLR   (DIOC|11)
#define IBCMD   (DIOC|12)
#define IBXTRC  (DIOC|14)
#define IBRSP   (DIOC|15)
#define IBRPP   (DIOC|16)
#define IBBNA   (DIOC|17)
#define IBWAIT  (DIOC|18)
#define IBONL   (DIOC|19)
#define IBRSV   (DIOC|20)
#define IBGTS   (DIOC|21)
#define IBCAC   (DIOC|22)
#define IBPOKE  (DIOC|23)
#define IBSGNL  (DIOC|24)
#define IBSRE   (DIOC|25)
#define IBRSC   (DIOC|26)
#define IBRD    (DIOC|27)
#define IBWRT   (DIOC|28)
#define IBLOC   (DIOC|29)
#define IBPCT   (DIOC|30)
#define IBSIC   (DIOC|31)
#define IBPPC   (DIOC|33)
#define IBLINES (DIOC|34)
#define IBEOT   (DIOC|35)
#define IBPAD   (DIOC|36)
#define IBSAD   (DIOC|37)
#define IBDMA   (DIOC|38)
#define IBEOS   (DIOC|39)
#define IBTMO   (DIOC|40)
#define IBWRTF  (DIOC|41)
#define IBRDF   (DIOC|42)
#define IBLLO   (DIOC|43)
#define IBSTAT  (DIOC|44)
#define IBLN    (DIOC|45)
#define IBDEV   (DIOC|46)
#define IBIST   (DIOC|32)

/* NI-488.2 calls */
#define ALLSPOLL        (DIOC|50)
#define DEVCLEARLIST    (DIOC|51)
#define ENABLELOCAL     (DIOC|52)
#define ENABLEREMOTE    (DIOC|53)
#define FINDRQS         (DIOC|54)
#define FINDLSTN        (DIOC|55)
#define PASSCONTROL     (DIOC|56)
#define PPOLLCONFIG     (DIOC|57)
#define PPOLLUNCONFIG   (DIOC|58)
#define PPOLL           (DIOC|59)
#define RECEIVE         (DIOC|60)
#define RCVRESPMSG      (DIOC|61)
#define RECEIVESETUP    (DIOC|62)
#define RESETSYS        (DIOC|63)
#define SENDCMDS        (DIOC|64)
#define SENDDATABYTES   (DIOC|65)
#define SENDIFC         (DIOC|66)
#define SENDLIST        (DIOC|67)
#define SENDLLO         (DIOC|68)
#define SENDSETUP       (DIOC|69)
#define SETRWLS         (DIOC|70)
#define TESTSYS         (DIOC|71)
#define TRIGGERLIST     (DIOC|72)
#define DEVCLEAR        (DIOC|73) 
#define READSTATUSBYTE  (DIOC|74)
#define SEND            (DIOC|75)
#define TESTSRQ         (DIOC|76)
#define TRIGGER         (DIOC|77)
#define WAITSRQ         (DIOC|78)

#define IBFIND		(DIOC|131)
#define IBCONFIG	(DIOC|132)
#define IBASK		(DIOC|133)

#define IBWRTA          (DIOC|150)
#define IBRDA           (DIOC|151)
#define IBCMDA          (DIOC|152)
#define IBSTOP		(DIOC|153)
#define IBWRTKEY	(DIOC|154)
#define IBRDKEY		(DIOC|155)

extern errno;
int ibsta, iberr;
int ibcnt;
long ibcntl ;
struct ibarg ibarg;

static int fp_gpib=-1;



/*
 * operating system error:
 *      error code EDVR indicates OS error code is in ibcnt
 */
static int oserr() {
	iberr = EDVR;
	ibcnt = errno;
	ibcntl = ibcnt ;
	return ibsta = ERR|CMPL;
}


/*
 * common function to set up ioctl, make call
 * and set global variables
 */

static int fn(int code, int f)

{
	char *path="/dev/gpib";

	if (fp_gpib==-1) {
		fp_gpib=open(path,O_RDWR);
		if (fp_gpib==-1)
			if ((code==IBFIND)||(code==IBDEV))
				return -1;
			else
				return oserr();
	}

	ibarg.size=sizeof(struct ibarg);
	ibarg.handle=f;
	if (ioctl(fp_gpib, code, &ibarg) < 0)
		return oserr();
	ibcnt = ibarg.ibcnt;
	ibcntl = ibcnt ;
	iberr = ibarg.iberr;
	ibsta = ibarg.ibsta;
	if ((code==IBFIND)||(code==IBDEV))
		return ibarg.handle;
	else
		return ibsta;
}




#define IB_BDNAME_SIZE 256

int ibfind  (char *bdname)
{
	struct stat fs;
	char name[IB_BDNAME_SIZE];

	if (*bdname=='/') {
		if (stat(bdname,&fs)!=0) {
			iberr=EDVR;
			ibsta=ERR|CMPL;
			return -1;
		}
	} else {
		name[0]='/';
		name[1]='d';
		name[2]='e';
		name[3]='v';
		name[4]='/';
		strncpy(&name[5],bdname,IB_BDNAME_SIZE-5);
		name[IB_BDNAME_SIZE-1]='0';
		if (stat(name,&fs)!=0) {
			iberr=EDVR;
			ibsta=ERR|CMPL;
			return -1;
		}
	}
	if (S_ISCHR(fs.st_mode)) {
		ibarg.n=fs.st_rdev;
		return fn(IBFIND,0);
	} else {
		iberr=EDVR;
		ibsta=ERR|CMPL;
		return -1;
	}

}


int ibdev (int bindex, int pad, int sad, int tmo, int eot, int eos)
{
	int info[6];
	info[0]=bindex;
	info[1]=pad;
	info[2]=sad;
	info[3]=tmo;
	info[4]=eot;
	info[5]=eos;
	ibarg.n=sizeof(info);
	ibarg.buf=&info[0];
	return fn(IBDEV,bindex);
}

/*
 * ibbna changes a device's access board.
 */

int ibbna(int g, char *bname)
{
	struct stat fs;
	char name[IB_BDNAME_SIZE];

	if (bname[0]=='/') {
		if (stat(bname,&fs)!=0) {
			iberr=EARG;
			ibsta=ERR|CMPL;
			return ibsta;
		}
	} else {
		name[0]='/';
		name[1]='d';
		name[2]='e';
		name[3]='v';
		name[4]='/';
		strncpy(&name[5],bname,IB_BDNAME_SIZE-5);
		name[IB_BDNAME_SIZE-1]='0';
		if (stat(name,&fs)!=0) {
			iberr=EARG;
			ibsta=ERR|CMPL;
			return ibsta;
		}
	}
	if (S_ISCHR(fs.st_mode)) {
		ibarg.n=fs.st_rdev;
		return fn(IBBNA,g);
	} else {
		iberr=EARG;
		ibsta=ERR|CMPL;
		return ibsta;
	}
}



int ibdiag (int g, void *buf, long cnt)
{ 
	ibarg.buf=buf;
	ibarg.n=cnt;
	return fn(IBDIAG, g); 
}

int ibrd(int g, void *buf, long cnt)
{ 
	ibarg.buf=buf;
	ibarg.n=cnt;
	return fn(IBRD, g); 
}


int ibrda (int g, void *buf, long cnt)
{ 
	ibarg.buf=buf;
	ibarg.n=cnt;
	return fn(IBRDA, g); 
}

int ibwrt (int g, void *buf, long cnt)
{ 
	ibarg.buf=buf;
	ibarg.n=cnt;
	return fn(IBWRT, g); 
}

int ibwrta (int g, void *buf, long cnt)
{ 
	ibarg.buf=buf;
	ibarg.n=cnt;
	return fn(IBWRTA, g); 
}

int ibcmd (int g, void *buf, long cnt)
{ 
	ibarg.buf=buf;
	ibarg.n=cnt;
	return fn(IBCMD, g); 
}

int ibcmda (int g, void *buf, long cnt)
{ 
	ibarg.buf=buf;
	ibarg.n=cnt;
	return fn(IBCMDA, g); 
}

int ibstop (int g)
{ 
	return fn(IBSTOP, g); 
}

int ibtrg (int g)
{ 
	return fn(IBTRG, g); 
}

int ibclr (int g)
{ 
	return fn(IBCLR, g); 
}

int ibpct (int g)
{ 
	return fn(IBPCT, g); 
}

int ibllo (int g)
{ 
	return fn(IBLLO, g); 
}


int ibloc (int g)
{ 
	return fn(IBLOC, g); 
}


int ibsic (int g)
{ 
	return fn(IBSIC, g); 
}

int ibrsp (int g, char *buf)
{ 
	ibarg.buf=buf;
	return fn(IBRSP, g); 
}

int ibrpp (int g, char *buf)
{ 
	ibarg.buf=buf;
	return fn(IBRPP, g); 
}

int iblines (int g, short *buf)
{ 
	ibarg.buf=buf;
	return fn(IBLINES, g); 
}

int ibwait (int g, int mask)
{ 
	ibarg.n=mask;
	return fn(IBWAIT, g); 
}

int ibrsv (int g, int v)
{ 
	ibarg.n=v;
	return fn(IBRSV, g); 
}

int ibsre (int g, int v)
{ 
	ibarg.n=v;
	return fn(IBSRE, g); 
}

int ibonl (int g, int v)
{ 
	ibarg.n=v;
	return fn(IBONL, g); 
}

int ibgts (int g, int v)
{ 
	ibarg.n=v;
	return fn(IBGTS, g); 
}

int ibcac (int g, int v)
{ 
	ibarg.n=v;
	return fn(IBCAC, g); 
}

int ibpoke (int g, int v, int val)
{ 
	ibarg.n=v;
	ibarg.n1 = val ;
	return fn(IBPOKE, g); 
}

int ibsgnl (int g, int v)
{ 
	ibarg.n=v;
	return fn(IBSGNL, g); 
}


int ibrsc (int g, int v)
{ 
	ibarg.n=v;
	return fn(IBRSC, g); 
}

int ibppc (int g, int v)
{ 
	ibarg.n=v;
	return fn(IBPPC, g); 
}

int ibist (int g, int v)
{ 
	ibarg.n=v;
	return fn(IBIST, g); 
}

int ibeot (int g, int v)
{ 
	ibarg.n=v;
	return fn(IBEOT, g); 
}

int ibpad (int g, int v)
{ 
	ibarg.n=v;
	return fn(IBPAD, g); 
}

int ibsad (int g, int v)
{ 
	ibarg.n=v;
	return fn(IBSAD, g); 
}

int ibdma (int g, int v)
{ 
	ibarg.n=v;
	return fn(IBDMA, g); 
}

int ibeos (int g, int v)
{ 
	ibarg.n=v;
	return fn(IBEOS, g); 
}

int ibtmo (int g, int v)
{ 
	ibarg.n=v;
	return fn(IBTMO, g); 
}

int ibconfig(int g, int option, int v) 
{
  	ibarg.n1 = option;
  	ibarg.n2 = v;
  	return fn(IBCONFIG,g);
}

int ibask(int g, int option, int *ret) 
{
        int result;

  	ibarg.n1 = option;
  	result= fn(IBASK,g);
  	*ret=ibarg.n2;
        return result;
}

	
int ibln (int g, int pad, int sad,short *listen) 
{
	ibarg.n=MakeAddr(pad,sad);	
	ibarg.buf=listen;
	return fn(IBLN, g);
}


/*
 * file I/O routines:
 */

#define FBSIZE  512             /* size of buffer for file I/O */
static char buf[FBSIZE*2];

/*
 * write file.
 * double buffering is used not for improved throughput, but
 * for determining the last buffer before it is sent.
 */
int ibwrtf(int g,char file[]) {
	int f, i, cnt[2], total=0;
	ushort wrtfail=0, sta=0, err=0;
	char eot;

	if ((f=open(file, O_RDONLY, 0)) < 0)
		return oserr();
	ibeot(g,0);
	eot=iberr;
	i= 0;
	if ((cnt[0]=read(f, buf, FBSIZE)) < 0) {
		oserr();
		goto bad;
	}
	while (cnt[i]) {
		i= !i ;
		if ((cnt[i]=read(f, &buf[i*FBSIZE], FBSIZE)) < 0) {
			oserr();
			goto bad;
		}
		if (cnt[i] == 0) { /* then the previous read reached the eof */
			ibeot(g,eot);
			if (ibsta&ERR)    /* restore eot mode */
				goto bad;
		}
		if ((ibwrt(g, &buf[!i * FBSIZE], cnt[!i])) & ERR) {
			wrtfail = 1;
			sta = ibsta;
			err = iberr;
			goto bad;
		}
		total += ibcnt;
	}
	goto ret;
bad:    
	ibeot(g,eot);
	if (wrtfail) {		/* restore the write fail status and err */
		ibsta = sta;
		iberr = err;
	}
ret:    
	close(f);
	ibcnt = total;
	return ibsta;
}

/*
 * read file.
 */
int ibrdf(int g, char file[]) {
	int f, total=0;

	if ((f=creat(file, 0644)) < 0) {
		iberr = EFSO;
		ibcnt = 0;
		ibsta = ERR|CMPL;
		return ibsta;
	}
	do {    if ((ibrd(g, buf, FBSIZE*2) & ERR) && (iberr == EDVR) )
			goto bad;
		if (write (f, buf, ibcnt) != ibcnt)
			goto bad;
		total += ibcnt;
	} while ((ibsta & (ERR|END)) == 0);
bad:    close(f);;
	ibcnt = total;
	return ibsta;
}

int ibxtrc  (int g, void *buf, long cnt)
{
	ibarg.n=cnt;
	ibarg.buf=buf;
	return fn(IBXTRC, g);
}

int ibwrtkey(int g, void *buf, long cnt)
{
	ibarg.n = cnt ;
	ibarg.buf = buf ;
	return fn(IBWRTKEY, g) ;
}


int ibrdkey(int g, void *buf, long cnt) 
{
        ibarg.n = cnt ; 
        ibarg.buf = buf ; 
        return fn(IBRDKEY, g) ; 
} 

/*********************************************************
 * NI-488.2 functions 
 *********************************************************/

void AllSpoll(int b, Addr4882_t *taddrs, short *res)
{
	ibarg.buf1 = taddrs;
	ibarg.buf2 = res;
	fn(ALLSPOLL,b);
	return ;
} 

void DevClearList(int b, Addr4882_t *addrlist)
{
	ibarg.buf1 = addrlist;
	fn(DEVCLEARLIST,b);
	return ;
}

void DevClear(int b, Addr4882_t addr)
{
	ibarg.n = addr;
	fn(DEVCLEAR,b);
	return ;
}		

void EnableLocal(int b, Addr4882_t *laddrs)
{
	ibarg.buf1 = laddrs;
	fn(ENABLELOCAL,b);
	return ;
}

void EnableRemote(int b,  Addr4882_t *laddrs)
{
	ibarg.buf1 = laddrs;
	fn(ENABLEREMOTE,b);
	return ;
}

void FindRQS(int b, Addr4882_t *taddrs, short *dev_stat)
{
	int ret;

	ibarg.buf1 = taddrs;
	ret = fn(FINDRQS,b);
	*dev_stat = (short)ibarg.n;
	return ;
} 

void FindLstn(int b,Addr4882_t *addrlist, Addr4882_t *reslist, int limit)
{
	ibarg.buf1 = addrlist;
	ibarg.buf2 = reslist;
	ibarg.n = limit;
	fn(FINDLSTN,b);
	return ;
} 

void PassControl(int b, Addr4882_t talker)
{
	ibarg.n = talker;
	fn(PASSCONTROL,b);
	return ;
}	

void PPollConfig(int b, Addr4882_t laddr, int  dataline, int  sense)
{
	ibarg.n = laddr;
	ibarg.n1 = dataline;
	ibarg.n2 = sense;
	fn(PPOLLCONFIG,b);
	return ;
}

void PPollUnconfig(int b,Addr4882_t *laddrs)
{
	ibarg.buf1 = laddrs;
	fn(PPOLLUNCONFIG,b);
}	

void PPoll(int b, short *res_ptr)
{
	int ret;

	ret=fn(PPOLL,b);
	*res_ptr = (ushort)ibarg.n;
	return ;
}

void ReadStatusByte(int b,Addr4882_t talker, short *result)
{
	int ret;
	ibarg.n=talker;
	ret=fn(READSTATUSBYTE,b);
	*result = (ushort)ibarg.n1;
	return ;
} 

void Receive(int b,Addr4882_t talker, void *buf, long cnt, int eotmode)
{
	ibarg.n1 = talker;
	ibarg.buf = buf;
	ibarg.n = cnt;
	ibarg.n2 = eotmode;
	fn(RECEIVE,b);
	return ;
}

void RcvRespMsg	(int b, void *buf, long cnt, int eotmode)
{
	ibarg.buf = buf;
	ibarg.n = cnt;
	ibarg.n1 = eotmode;
	fn(RCVRESPMSG,b);
}

void ReceiveSetup(int b, Addr4882_t talker)
{
	ibarg.n = talker;
	fn(RECEIVESETUP,b);
	return ;
}	

void SendList(int b,Addr4882_t *listeners, void *buf, long cnt, int eotmode)
{
	ibarg.buf1 = listeners;
	ibarg.buf = buf;
	ibarg.n = cnt;
	ibarg.n2 = eotmode;
	 fn(SENDLIST,b);
	return ;
}	

void Send(int b,Addr4882_t listener, void *buf, long cnt, int eotmode)
{
	ibarg.n1 = listener;
	ibarg.buf = buf;
	ibarg.n = cnt;
	ibarg.n2 = eotmode;
	fn(SEND,b);
	return ;
}	

void SendCmds(int b, void *buf, long cnt)
{
	ibarg.buf = buf;
	ibarg.n = cnt;
	fn(SENDCMDS,b);
	return ;
}	

void SendDataBytes(int b, void *buf, long cnt, int eotmode)
{
	ibarg.buf = buf;
	ibarg.n = cnt;
	ibarg.n2 = eotmode;
	fn(SENDDATABYTES,b);
	return ;
}

void SendIFC(int b)
{
	fn(SENDIFC,b);
	return ;
}	

void ResetSys(int b,Addr4882_t *laddrs)
{
	ibarg.buf1 = laddrs;
	fn(RESETSYS,b);
	return ;
}

void SendLLO(int b)
{
	fn(SENDLLO,b);
	return ;
}	

void SendSetup(int b,Addr4882_t *listen)
{
	ibarg.buf1 = listen;
	fn(SENDSETUP,b);
	return ;
}	

void SetRWLS(int b,Addr4882_t *laddrs)
{
	ibarg.buf1 = laddrs;
	fn(SETRWLS,b);
	return ;
}	

void TestSRQ(int b, short *result)
{
	int ret;
	ret=fn(TESTSRQ,b);
	*result=ibarg.n;
	return;
}		

void TestSys(int b, Addr4882_t *addrs, short *result)
{
	ibarg.buf1 = addrs;
	ibarg.buf2 = result;
	fn(TESTSYS,b);
	return ;
}	

void TriggerList(int b,Addr4882_t *laddrs)
{
	ibarg.buf1 = laddrs;
	fn(TRIGGERLIST,b);
	return ;
}	

void Trigger(int b,Addr4882_t laddr)
{
	ibarg.n=laddr;
	fn(TRIGGER,b);
	return ;
}	

void WaitSRQ(int b,short *result)
{
	int ret;
	ret=fn(WAITSRQ,b);
	*result = ibarg.n;
	return;
}
