// E.S.O.
// bgaillar  20/10/99  created 
//
//
//KE486 driver
//purpose : this unit is an high level interface to access to the ammeter
//ke486 on the CCD test bench
//At first, initialize the device by calling the init function 
//
// setFilter
//                 0 : Digital and analog filters disabled
//                 1 : Digital filter enabled, analog filter disabled
//                 2 : Dgital filter disabled, analog filter enabled
//                 3 : Digital and analog filter enabled
          
// setRange        
//                 0 : Enable autorange
//                 1 : 2nA range
//                 2 : 20nA range
//                 3 : 200nA range
//                 4 : 2microA range
//                 5 : 20microA range
//                 6 : 200microA range
//                 7 : 2mA range
//                 8 : no range (IDDCO error)
//                 9 : no range (IDDCO error)
//                 10 : disable autorange
//               
// setIntegration  
//                 0 : Fast integration period (4-1/2d resolution)
//                 1 : Line cycle integration period (5-1/2d resolution)
// E.S.O.
// bgaillar  20/10/99  created 
//
// setReadoutDelay
//                      Delay trigger by "n" seconds,
//                      where "n" = 0 to 999.999 seconds
//
// mesureSerie
//                      func 
//                            0 : Multiple talk
//                            1 : one shot on talk
//                            2 : Multiple on get
//                            3 : One shot on get
//                            4 : Multiple on X
//                            5 : One-shot on X
//                            6 : Multiple on external trigger
//                            7 : One-shot on external trigger
//                            8 : Multiple on operate
//                            9 : One-shot on operate
//
//                      numberOfPoints 
//                            beetween 1 and 512
//                      
//                      interval (use to set the multiple trigger interval)
//                            beetween 0.010sec to 999.999sec 
//
// setOutPutStream     Set the ouput standard message stream.NULL if
//                     no message. 

#include <stdio.h>
#include "ke486drv.h"
#include "ke486_err.h"


//ke486 adress beetween 0 and 30    
#define GPIBAddress 14



#define ERRMSQ 0x8000


//-- INSTRUMENT-DEPENDENT COMMAND ARRAYS 
//	The following is an example of how instrument-dependent command
//          arrays can  be used in the code.
//
// static char primDispFunc[] = {"AAC","ADC","OHMS","VAC","VDC"};
// static char secDispFunc[]  = {"AAC2","ADC2","OHMS2","VAC2","VDC2"};
//
//  When formatting strings to send to your instrument, these arrays can be
//  used to simplify your code, for example:
//
//  if ((ke486_err = printf (instrSession, "%s; %s;",
//          primDispFunc[func1], secDispFunc[func2])) < 0)
//      return ke486_err;
//
//  The variables func1 and func2 are being used as indexing parameters that
//  are passed into the function.
//  Insert a check on Func in the calling program, to avoid passing real
//  values, which will be truncated but will not generate error messages


static char *zeroCheck[] = {"C0","C1","C2"};
static char *filter[] = {"P0","P1","P2","P3"};
static char *range[] = {"R0","R1","R2","R3","R4","R5","R6","R7","R8","R9","R10"}; 
static char *integration[] = {"S0","S1"};  
static char *trigger[] = {"T0","T1","T2","T3","T4","T5","T6","T7"}; 

Ke486::Ke486() {
    int interI;

    device=-1;

    // It opens the amp1
    if (!((interI=ibfind("/dev/amp1")) & ERRMSQ)) 
	device=interI;
}

Ke486::Ke486(ostream * outputS) {
    int interI;

    device=-1;
    this->outputS=outputS;

    // It opens the amp1
    if (outputS!=NULL) 
	(*outputS) <<"Opening /dev/amp1\n";
    if (!((interI=ibfind("/dev/amp1")) & ERRMSQ)) 
	device=interI;
}

Ke486::~Ke486() {
    if (device!=-1) {
    // ibloc places device in local    
    ibloc(device);
    ibonl(device,0);
    }
}

// Function: SetOutPutStream
// Purpose:  Set the ouput standard message stream.NULL if
//           no message. 
void Ke486::setOutPutStream(ostream * outputS) {
    this->outputS=outputS;
}






// Function: Read data
// Purpose:  This function reads a buffer of data from the GPIB interface
//           using the 488.2 library function ibrd.
//           The return value is equal to the global error variable
int Ke486::readBuffer(char *cmd,int cnt) {    
    int errorCode=0; 
    if (ibrd(device,cmd,(long)cnt) & ERRMSQ)
        errorCode=101;
    return errorCode;
}


// Function: Write data
// Purpose:  This function writes a buffer of data to the GPIB interface
//           using the 488.2 library function ibwrt.
//           The return value is equal to the global error variable

int Ke486::writeBuffer (char *cmd, int cnt) {
    int errorCode=0;

    if (ibwrt(device,cmd,(long)cnt) & ERRMSQ)
        return errorCode=102;
    
    // if check(0X20)== 0 the 5th bit has been set. An error has occured.
    // The error_ident function performs a serial poll and returns the
    // error string coming from the device.
   
    if (check(0x20)==0) {
    	errorCode=errorIdent();
    }  
    return errorCode;
}
    

int Ke486::displayMessage(char * message) {
    int errorCode=0;
    int messageLength;
    char cmd[255];
    
    if (sprintf(cmd,"D%sX",message)<=0)
	return 40;
    messageLength=strlen(message);
    
    if ((errorCode=writeBuffer("cmd",messageLength))!=0) 
	return errorCode;	
    return errorCode;
}



// Function: Check Status
// Purpose:  this function checks or waits for scan or buffered
//			 read completion
int Ke486::check(int Mask) {
    char Spr;
    int errorCode=0;
    

    if (ibrsp(device,&Spr) & ERRMSQ)
        errorCode=1000;

    // The Mask format is built according to the specs in the Ke486 user manual, page 4-21
    // We always check that the complete mask is verified, which insures that all required
    // conditions are fulfilled. The mask can be given both as a decimal value (i.e 40, which
    // corresponds to bit 5 and bit 3 up) or a hexadecimal one (i.e. 0x28).
    // Spr can be different but the condition is verified when all the bits of the Mask are set
    // also by the device (Spr).
    // Important: condition ke486_err=0 happens when the Mask is perfectly matched! If the
    // calling function is checking for an error condition (i.e. error present if the Mask is
    // met, remember to set ke486_err from 300  back to 0. This conforms to the rule which
    // associates 0 error to no error present.
       
    else
	if (((Spr) & Mask)==Mask)  // condition is met
	    errorCode=0;
	else
	    errorCode=200;
    return errorCode;
}



// Function: Make Zero check
// Purpose:  This function cancels any internal offset that might
//           affect accuracy
int Ke486::makeZeroCheck(int Func) {
    int nbcmd;
    char cmd[255];
    int errorCode=0;
            
    if ((Func<0) || (Func>2))
        return 30;

    // append a terminator X to the string and format
    if (sprintf(cmd,"%sX",zeroCheck[Func])==0)
	return 40;
    nbcmd=strlen(cmd);
    if ((errorCode=writeBuffer(cmd,nbcmd))!=0)
	return errorCode;

    return errorCode;	
}


// Function: Set Filter
// Purpose:  This function sets the filter type on the input signal
int Ke486::setFilter(int Func)
{  
    int nbcmd;
    char cmd[255];
    int errorCode=0;
    
    if ((Func<0) || (Func>3))
        return 30;

    // append a terminator X to the string and format
    if (sprintf(cmd,"%sX",filter[Func])==0)
	return 40;
    nbcmd=strlen(cmd);

    if ((errorCode=writeBuffer(cmd,nbcmd))!=0)
    	return errorCode;
    return errorCode;	
}



// Function: Set Range
// Purpose:  This function sets the current measurement range

int Ke486::setRange(int Func)
{   
    int nbcmd;
    char cmd[255];
    int errorCode=0;
    if ((Func<0) || (Func>10))
        return 30;

    // append a terminator X to the string and format
    if (sprintf(cmd,"%sX",range[Func])==0)
	return 40;
    nbcmd=strlen(cmd);

    if ((errorCode=writeBuffer(cmd,nbcmd))!=0)
    	return errorCode;
    	
    //  perform zero correction    	
   
    if ((errorCode=makeZeroCheck(2))!=0)
	return errorCode;
   
    return errorCode;	
}




// Function: Set Integration period 
// Purpose:  This function sets the integration period and display
//           resolution.
int Ke486::setIntegration(int Func) {   
    int nbcmd;
    char cmd[255];
    int errorCode=0;
    if ((Func<0) || (Func>1))
        return 30;
    // append a terminator X to the string and format
    if (sprintf(cmd,"%sX",integration[Func])==0)
	return 40;
    nbcmd=strlen(cmd);

    if ((errorCode=writeBuffer(cmd,nbcmd))!=0)
    	return errorCode;

    return errorCode;	
}






// Function: Set readout delay
// Purpose:  This function sets the trigger delay time
int Ke486::setReadoutDelay(float delay) {
    int nbcmd;
    char cmd[255];
    int errorCode=0;
    if ((delay<0.0) || (delay>999.999))
        return 31;


    if ((errorCode=sprintf(cmd,"W%fX",delay))!=0)
	return 40;
    nbcmd=strlen(cmd);

    if ((errorCode=writeBuffer(cmd,nbcmd))!=0)
    	return errorCode;

    return errorCode;	
}


// Function: Set trigger mode
// Purpose:  This function sets the trigger mode and source
int Ke486::setTrigger(int Func) {
    int nbcmd;
    char cmd[255];
    int errorCode=0;
    if ((Func<0) || (Func>7))
        return 30;
    // append a terminator X to the string and format
    if (sprintf(cmd,"%sX",trigger[Func])==0)
	return 40;
    nbcmd=strlen(cmd);

    if ((errorCode=writeBuffer(cmd,nbcmd))!=0)
    	return errorCode;

    return errorCode;
}





// Function: measure
// Purpose:  This function takes one reading from the ammeter
int Ke486::measure(int Func,float &value) {
    char cmd[255];
    int errorCode=0;
    
    if ((Func<0) || (Func>7))
        return 30;

    // check the current status of the ammeter's settings


    if ((errorCode=writeBuffer("U0X",3))!=0)
    	return errorCode;

    if ((errorCode=readBuffer(cmd,43))!=0)
    	return errorCode;

    if (cmd[6]!='0')  {
    
    
    // set readout from A/D converter
    if ((errorCode=writeBuffer("B0X",3))!=0) 
    	return errorCode;
    }	
    
    
    

    
    if (cmd[10]!='1')  {
     // set ASCII reading without prefix */
    if ((errorCode=writeBuffer("G1X",3))!=0)
	return errorCode;	
    }

    // trigger mode  = 7 (single shot external) is not coded
    // trigger modes 1,3,5 are valid ones
    // trigger modes 2,4,6 are not valid (multiple readings only)

    switch (Func)  {
    case 1:
	// single shot on TALK
	if ((errorCode=writeBuffer("T1X",3))!=0) 
	    return errorCode;
	// send trigger
	if (ibtrg(device) & ERRMSQ)
	    return 7;
	break;
    case 3:	
	// single shot on GET
	if ((errorCode=writeBuffer("T3X",3))!=0) 
	    return errorCode;
	
	// send trigger
	if (ibtrg(device) & ERRMSQ)
	    return 7;
	break;
    case 5:
	//single shot on X terminator
	//send the trigger
	if ((errorCode=writeBuffer("T5X",3))!=0)
	    return errorCode;
	break;
    default:
	return 8;
 }
    
    // acquire data. Each data packet consists of 13 bytes (with G1 option set)
    
    if ((errorCode=readBuffer(cmd,13))!=0)		   
        return errorCode;
   
    if (sscanf(cmd,"%G",&value)!=1)
	return 40;
    
    
    return errorCode;	
}




// Function: getString
// Purpose:  This function get an error string
void getString(int num,char separator,char * str,char * strBack) {
    int i=0;int j=0;
    strBack[0]=0;
    for (;((i<num-1) && (str[j]!=0));i++) {
    for (;((str[j]!=0) && (str[j++]!=separator)););
    }
    if (str[j]!=0) {
    for (i=0;((str[j]!=0) && (str[j]!=separator));strBack[i++]=str[j++]);
    strBack[j+1]=0;
    } 
}



// Function: mesureSerie
// Purpose:  This function reads multiple data entries
int Ke486::mesureSerie(int Func, int NumberOfPoints, float Interval, float *DataArray) {
    
    int i;

    
    int nbcmd;
    char cmd[7000];
    char interStr[1000];
    int errorCode=0;

   
    
    if ((NumberOfPoints<1) || (NumberOfPoints>512))
        return 30;
    if ((Interval<0.0) || (Interval>999.999))
        return 31;

    
    switch(Func) {
    case 0:
    case 2:
    case 4:
	
	// temporarily remove timeout setting from the device. Reading a 512 data buffer
	// requires more than 100 sec. This command prevents timeout errors during download
	

	if (ibtmo(device,0) & ERRMSQ)
	    return 17;
	

	// multiple read on X
	if ((errorCode=writeBuffer("T4",2))!= 0)
	    return errorCode;
	

	// Prepare string to :
	// 1)Arm the buffer (N setting)
	// 2)Set to default(175msec) if Interval is less than 10 ms. (Q setting)
	//              Use Interval in all other cases.
	// 3)The last "X" sends the trigger


	if (Interval< 0.010) 
	    if (sprintf(cmd,"Q0X")==0)
		return 40;
	    else
		if (sprintf(cmd,"Q%fX",Interval)==0)
		    return 40; 


	nbcmd=strlen(cmd);
	if ((errorCode=writeBuffer(cmd,nbcmd))!=0)
	    return errorCode;


	
	if (sprintf(cmd,"N%dX",NumberOfPoints)==0)
	    return 40;

	
	//if (sprintf(cmd,"I%dX",NumberOfPoints)==0)
	//    return 40;
	nbcmd=strlen(cmd);
	if ((errorCode=writeBuffer(cmd,nbcmd))!=0)
	    return errorCode;
	
	// wait for data to be stored and device to be ready. Mask 1E corresponds to bytes
	// 1 to 4 all set. See Ke486 user manual page 4-21
	
	if ((errorCode=writeBuffer("B2X",3))!=0)
		return errorCode;
	while (check(0x1E)!=0);
			     
	// select all reading from data store
	// Download the complete string; it is faster than taking measurements one by one
	// restore the timeout to 30 seconds
	// return to A/D converter as source
	if ((errorCode=readBuffer(cmd,6700))!=0)
	    return errorCode;

	
	for (i=0;i<NumberOfPoints;i++) {
	getString(i+1,',',cmd,interStr);
	if (sscanf(interStr,"%G",&DataArray[i])==0)
	    return 40;
	}
	
	if (ibtmo(device,14) & ERRMSQ)
	    return 17;
	
	if ((errorCode=writeBuffer("B0X",3))!=0) 
	    return errorCode;
	break;
    case 6:
	
	// multiple read on external trigger
	if ((errorCode=writeBuffer("T6", 2))!=0)
	    return errorCode;	
	



	if (Interval<=0.010) 
	    if (sprintf(cmd,"Q0.010X")==0)
		return 40;
	    else
		if (sprintf(cmd,"Q%fX",Interval)==0)
		    return 40; 
	nbcmd=strlen(cmd);
	// send the string to arm the buffer and start storing readings when trigger is received
	if ((errorCode=writeBuffer(cmd,nbcmd))!=0)
	    return errorCode;
	
	if (sprintf(cmd,"N%dX", NumberOfPoints)==0)
		return 40;
	nbcmd=strlen(cmd);
	// send the string to arm the buffer and start storing readings when trigger is received
	
	if ((errorCode=writeBuffer(cmd,nbcmd))!=0)
	    return errorCode;
	

	
	// no data is read here. The function ke486_get_buffer should be called when needed from the
	// calling program. Hence DataArray is returned as zeroes
	
	for (i=0;i<NumberOfPoints;i++)
	    DataArray[i]=0.0;
	break;	
    default:
	return 8;
    }
    
    
    return 0;
}


// Function: get buffer
// Purpose:  This function returns a specified number of readings from buffer
int Ke486::getMesureSerieBuffer(int nbRead,float *DataArray) {
    char cmd[255];
    int i;
    char interStr[256];
    int errorCode=0;
     // temporarily remove timeout setting from the device. Reading a 512 data buffer
    // requires more than 100 sec. This command prevents timeout errors during download
    
    if (ibtmo(device,0) & ERRMSQ)
	return 17;
    // select all readings from data store
    
    if ((errorCode=writeBuffer("B2X",3))!=0)
	return errorCode;
    
    // Download the complete string; it is faster than taking measurements one by one
    
    if ((errorCode=readBuffer(cmd,6700))!= 0)
	return errorCode;
 
    for (i=0;i<nbRead;i++) {
    getString(i+1,',',cmd,interStr);
    if (sscanf(interStr,"%G",&DataArray[i])!=1)
	return 40;
    }

    // restore the timeout to 30 seconds
    
    if (ibtmo(device,14) & ERRMSQ)
	return 17;
    
    //return to A/D converter as source
    
    if ((errorCode=writeBuffer("B0X", 3))!=0) 
	return errorCode;	
    return errorCode;
    
}













// Function: Send error messages corresponding to the error code
char * Ke486::getErrorString(int errorCode,char * errorString) {
    switch (errorCode) {
    case 1:sprintf(errorString,"%s",E1); break;
    case 2:sprintf(errorString,"%s",E2); break;
    case 7:sprintf(errorString,"%s",E7); break;
    case 8:sprintf(errorString,"%s",E8); break;
    case 17:sprintf(errorString,"%s",E17); break;
    case 30:sprintf(errorString,"%s",E30); break;
    case 31:sprintf(errorString,"%s",E31); break;
    case 40:sprintf(errorString,"%s",E40); break;
    case 101:sprintf(errorString,"%s",E101); break;
    case 102:sprintf(errorString,"%s",E102); break;
    case 200:sprintf(errorString,"%s",E200); break;
    case 300:sprintf(errorString,"%s",E300); break;
    case 301:sprintf(errorString,"%s",E301); break;
    case 302:sprintf(errorString,"%s",E302); break;
    case 303:sprintf(errorString,"%s",E303); break;
    case 304:sprintf(errorString,"%s",E304); break;
    case 305:sprintf(errorString,"%s",E305); break;
    case 306:sprintf(errorString,"%s",E306); break;
    case 307:sprintf(errorString,"%s",E307); break; 
    case 308:sprintf(errorString,"%s",E308); break;    
    case 309:sprintf(errorString,"%s",E309); break;
    case 310:sprintf(errorString,"%s",E310); break;
    case 311:sprintf(errorString,"%s",E311); break;
    case 312:sprintf(errorString,"%s",E312); break;

    default:sprintf(errorString,"%s",def);
    }
    return errorString;

}




// Function: Checks errors from the device
// Purpose:  This function identifies an error generated by the device
//           It is called by the ke486_write_data function whenever a
//           serial poll detects an error occurence (5th bit set).
//           It returns a string indicating the type of error.
//			 All error codes are described in Ke 486 User manual page 4-34
int Ke486::errorIdent() {
    char string[14];
    char cmd[256];
    int errorCode=0;
    int i;

    // Query the error string from the device
     
    if (ibwrt(device,"U1X",3)& ERRMSQ) 
        errorCode=102;
    
    if (ibrd (device,cmd,16) & ERRMSQ)
        errorCode=101;

    if (scanf(&(cmd[3]),"%s",string)==0)
	return 40;

    // Build the error code
    for (i=0;(i<13);i++)
	if (string[i]=='1')
	    return 300+i;
	    
    
    return 0;
}






// Function: Initialize
// Purpose:  This function opens the instrument, queries the instrument
//           for its ID, and initializes the instrument to a known state.
int Ke486::init() {
    char cmd[255];
    int errorCode=0;
    
    
    if (device==-1)
	return 1;
    
    
    // check if the remote status is enabled. If check(0X20)== 0 the 5th bit has been set.
    //(keythley Man. pag 4-22. In this case, REN is set sending the Listen address.
    // The error_ident function performs a serial poll and returns the
    // error string coming from the device.
    
    if (outputS!=NULL)
	(*outputS) << "init ke486...\n";
    
    if (check(0x20)==0) {
    errorCode=errorIdent();
    if (ibsre(0,GPIBAddress) & ERRMSQ) {
    // send listen address 7. See appendix A Gpib Software manual, for message MLA7. The character "'"
    // must be changed if the the primary GPIB address of the ke486 is not anymore 7
    ibcmd(device,"@?'",3); 
    }
    }
    else
	errorCode=0;
    if (outputS!=NULL)
	(*outputS) << "reset ke486\n";
    // clears the device
    if (ibclr(device) & ERRMSQ)  {
     return 2;
    }    	

    

    if (outputS!=NULL)
	(*outputS) << "ke486: ID query\n";
    // ID query
    if ((errorCode=writeBuffer("U2X",3))!=0)	 
	return errorCode;
    if ((errorCode=readBuffer(cmd,6))!=0)
	return errorCode;
   
    
    //only 1 string should be scanned
    if (outputS!=NULL)
	(*outputS) << "ke486: ID= " << cmd << "\n";
    
   
    // set standard configuration
    // set ASCII format without prefix
    if ((errorCode=writeBuffer("G1X", 3))!=0)  	 
    	return errorCode;
    


    // set terminator sequence to none
    if ((errorCode=writeBuffer("Y4X",3))!=0)		 
    	return errorCode;
    

    // enable autorange and disable zero check    	
    if((errorCode=makeZeroCheck(2))!=0)
	return errorCode;
    if ((errorCode=makeZeroCheck(0))!=0)
	return errorCode;
   
    // read out the complete status string and print it to screen
    if ((errorCode=writeBuffer("U0X",3))!=0)
    	return errorCode;
    if ((errorCode=readBuffer(cmd,41))!=0)
    	return errorCode;
    

    //Filter (enable digital analog filter)
    if ((errorCode=writeBuffer("P3X",3))!=0)
    	return errorCode;
    

    if (outputS!=NULL)
	(*outputS) << "ke486: init ended.\n";

    
    return errorCode;
}





