// E.S.O.
// bgaillar  20/10/99  created 
//
//
// ms257 driver
//purpose : this unit is an high level interface to access to the monochromator
//ms257 on the CCD test bench
//At first, initialize the device by calling the init function 
//
// init                  Initialize the monochromator
//
//  setWavelength        Position the current grating to the current wavelength
//                       Wavelength must be within the range 0 to MAXW
//
// setPortout            Select input port (B,C)
//
// setBandpass           Bandpath in NM.
//
// setFilter             Select position of filter wheel 1, in front of the 
//                       monochromator. Value beetween 1 and 5.      
//
// setShutter            Set the shutter state (1: open, 0: close)
//
// setOutPutStream       Set the ouput standard message stream.NULL if
//                       no message. 

#include "ms257drv.h"
#include "ms257_err.h"
#include <stdio.h>

#define ERRMSQ 0x8000


// Purpose:  This function opens the instrument, queries the instrument
//           for its ID, and initializes the instrument to a known state.
Ms257::Ms257() {
    int interI;
    device1=-1;
    device2=-1;
    
    outputS=NULL;
    
    // It opens the mono1 and mono2 devices    
    printf("Opening /dev/mono1\n");
    if (!((interI=ibfind("/dev/mono1")) & ERRMSQ)) 
	device1=interI;
   
    printf("Opening /dev/mono2\n");
    if (!((interI=ibfind("/dev/mono2")) & ERRMSQ)) 
	device2=interI;
}


Ms257::Ms257(ostream * outputS) {
    int interI;
    device1=-1;
    device2=-1;
    
    this->outputS=outputS;
    
    if (outputS!=NULL)
	(*outputS) << "Opening /dev/mono1\n";
    if (!((interI=ibfind("/dev/mono1")) & ERRMSQ)) 
	device1=interI;
   
    if (outputS!=NULL)
	(*outputS) << "Opening /dev/mono2\n";
    if (!((interI=ibfind("/dev/mono2")) & ERRMSQ)) 
	device2=interI;
}


// Function: Close
Ms257::~Ms257() {
    if (device1!=-1)
	ibonl(device1,0);
    if (device2!=-1)
	ibonl(device2,0);
}

// Purpose:  This function write data to a monochromator
//           using the 488.2 library function ibrd.
int Ms257::writeBuffer(int dev,char *string,int cnt) {
    int device=(dev==1) ? device1 : device2;
    if (device1==-1) 
	return 101;
    if (device2==-1)
	return 201;

    if (ibwrt(device,string,cnt) & ERRMSQ)
	switch (dev) {
	case 1:return 110;break;
	default:return 210;
	}
    return 0;    
}

// last character sent by the instrument is always '>'
const char EOS_BYTE = '>';


// Function: Read buffer
// Purpose:  This function reads a buffer of data from a monochromator
//           using the 488.2 library function ibrd.
int Ms257::readBuffer(int dev,char *string,int & cnt) {
    int i=0;
    char interChar;   
    int device=((dev==1) ? device1 : device2);
    int errorCode=0;


    if (device1==-1) 
	return 101;
    if (device2==-1)
	return 201;
	    

    // Always eliminate the first two leading characters
    for (i=0;i<2;i++)
	if (ibrd(device,&interChar,1) & ERRMSQ)	     
	    switch (dev) {
	    case 1:return 111;break;
	    default:return 211;
	    }

    //ibrd(device,&interChar,1);		    

    // Read the rest of the string one character by one
    for (i=0,cnt=0;i<30;i++) { 	
    if (ibrd(device,(string+i),1) & ERRMSQ)	     
	switch (dev) {
	case 1:return 111;break;
	default:return 211;
	}
    cnt=cnt+1;
    
    // If the first character is '>' then exit the function and return no error
    // This condition checks that the previous write operation has been performed
    // without errors.

    if (string[i]==EOS_BYTE && i==0)
    return 0;	           
    else
    if (string[i]==EOS_BYTE)
	break;	
    }    
    string[cnt-1]=0;
    //Communication error
    if (string[0]=='E') {
    sscanf((string+1),"%d",&errorCode);
    errorCode+=301;
    return errorCode;
    }
    return 0;
    
}


int Ms257::init() {
    char * temp1="1:500:2:700:3";
    
    
    char cmd[200];
    int nbcmd;
    
    if (outputS!=NULL)
	(*outputS) << "init ms257...\n";
    

    if (device1==-1) 
	return 101;
    if (device2==-1)
	return 201;

    if (outputS!=NULL)
	(*outputS) << "reset ms257\n";

    if (ibclr(device1) & ERRMSQ)
	return 102;
    
    if (ibclr(device2) & ERRMSQ)
	return 202;
    
    writeBuffer(1,"?VER\r",5);
    readBuffer(1,cmd,nbcmd);
    
    
    if (outputS!=NULL)
	(*outputS) << "ms257: ID= " << cmd << "\n";
   

   //  Checks on system:
       //units should be nm
	   //  A query on output port on mono2 = B should
	   //  be added in case the fiber experiment has
	   //  been mounted.
	   
   writeBuffer(1,"?UNITS\r",7);
   readBuffer(1,cmd,nbcmd);
   	
   if (cmd[0]!='N') {
   writeBuffer(1,"=UNITS NM\r",10);
   readBuffer(1,cmd,nbcmd);
   if (outputS!=NULL)
       (*outputS) << " ms257: UNITS reset to--> " << cmd << "\n";
   }

   
   writeBuffer(2,"?UNITS\r",7);
   readBuffer(2,cmd,nbcmd);
      	
   if (cmd[0]!='N') {
   writeBuffer(2,"=UNITS NM\r",10);
   readBuffer(2,cmd,nbcmd);
   if (outputS!=NULL)
       (*outputS) << "ms257: UNITS reset to--> " << cmd << "\n";
   }


// Settings on init:
//	Filter wheels on zero position (no filter) 
//	Bandpass = 10		
//  Wavelenght = 400 nm				
// Select output port B for Monochromator 2 

   writeBuffer(2,"!PORTOUT B\r",11);
   readBuffer(2,cmd,nbcmd);
   

   // Neutral density filter wheel
   writeBuffer(1,"!FILT2 1\r",9);
   readBuffer(1,cmd,nbcmd);
   
   if (outputS!=NULL)
       (*outputS) << "ms257: Setting on neutral filter\n";

 
// Wavelength set to 632

   writeBuffer(1,"!GW 632\r",8);
   readBuffer(1,cmd,nbcmd);
  
   writeBuffer(2,"!GW 632\r",8);
   readBuffer(2,cmd,nbcmd);


   if (outputS!=NULL)
       (*outputS) << "ms257: setting wavelength on 632nm \n";


// Order sorting filter wheel. Position 0 enables AUTO filter selection.

   writeBuffer(1,"!FILT1 0\r",9);
   readBuffer(1,cmd,nbcmd);
  
// Shutter

   writeBuffer(1,"=SHTRTYPE S\r",12);
   readBuffer(1,cmd,nbcmd);
   
   writeBuffer(1,"!SHUTTER 0\r",11);
   readBuffer(1,cmd,nbcmd);
   

//  Checks on system:
//  Are Changeover Points set correctly for order sorting filter wheel?
//  If not, set the correct values.
//  Oriel filters 51290 and 51330 are used to sort the orders:
//  		- from 250 -  500nm: 	filter 1 (no filter)
//			- from 500 -  700nm:	filter 2 (51290/cut-on wl: 475nm)
//			- from 700 - 1100nm:	filter 3 (51330/cut-on wl: 665nm)

   
   writeBuffer(1,"?CHNGF1\r",8);
   readBuffer(1,cmd,nbcmd);
   
   if (strcmp(temp1,cmd)!=0) {
   writeBuffer(1,"=CHNGF1 1:500:2:700:3\r",22);
   readBuffer(1,cmd,nbcmd); 
   
   }
   
   

// Bandpass set to 10 nm for each slit (A,B,C)
   if (outputS!=NULL)
       (*outputS) << "ms257: set band pass to 10nm\n";
   writeBuffer(1,"=BANDPASS 10\r",13);
   readBuffer(1,cmd,nbcmd); 
   writeBuffer(2,"=BANDPASS 10\r",13);
   readBuffer(2,cmd,nbcmd); 

   
   writeBuffer(1,"?MAXW\r",6);
   readBuffer(1,cmd,nbcmd);
   sscanf(cmd,"%f",&maxW);

   return 0;

}



// Function: SetOutPutStream
// Purpose:  Set the ouput standard message stream.NULL if
//           no message. 
void Ms257::setOutPutStream(ostream * outputS) {
    this->outputS=outputS;
}



// Function: SetWavelength
// Purpose:  Set the monochromator wavelength value 
int Ms257::setWavelength(float wavelength) {
    char cmd[200];
    int nbcmd;
    int errorCode=0;
    
    if ((wavelength<=0) ||
	(wavelength>maxW))
	return 15;
    
    sprintf(cmd,"!GW %g\r",wavelength);
    nbcmd=strlen(cmd);
    if (outputS!=NULL)    
	(*outputS) << "command= " << cmd << "\n";


    if ((errorCode=writeBuffer(1,cmd,nbcmd))!=0)
	return errorCode;
    if ((errorCode=readBuffer(1,cmd,nbcmd))!=0)
	return errorCode;

    sprintf(cmd,"!GW %g\r",wavelength);
    nbcmd=strlen(cmd);
    if ((errorCode=writeBuffer(2,cmd,nbcmd))!=0)
	return errorCode;
    if ((errorCode=readBuffer(2,cmd,nbcmd))!=0)
	return errorCode;

    return 0;
}


// Function: SetPortout
// Purpose:  Set the outport state (B or C)
int Ms257::setPortout(char port) { 
    char cmd[200];
    int nbcmd;
    int errorCode=0;

    if ((port!='B') && (port!='C') && (port!=0))
	return 13;    

    sprintf(cmd,"!PORTOUT %c\r",port);
    nbcmd=strlen(cmd);
    if (outputS!=NULL)
	(*outputS) << "command= " << cmd << "\n";

    if ((errorCode=writeBuffer(2,cmd,nbcmd))!=0)
	return errorCode;
    if ((errorCode=readBuffer(2,cmd,nbcmd))!=0)
	return errorCode;
    return 0;
}


// Function: SetBandPass
// Purpose:  Set the monochromator bandpath (in Nanometer)
int Ms257::setBandpass(float bandpass) {
    char cmd[200];
    int nbcmd;
    int errorCode=0;

    if (bandpass<0)
	return 15;   

    sprintf(cmd,"=BANDPASS %g\r",bandpass);
    nbcmd=strlen(cmd);
    if (outputS!=NULL)
	(*outputS) << "command= " << cmd << "\n";

    
    if ((errorCode=writeBuffer(1,cmd,nbcmd))!=0)
	return errorCode;
    if ((errorCode=readBuffer(1,cmd,nbcmd))!=0)
	return errorCode;
    
    sprintf(cmd,"=BANDPASS %g\r",bandpass);
    nbcmd=strlen(cmd);
    if ((errorCode=writeBuffer(2,cmd,nbcmd))!=0)
	return errorCode;
    if ((errorCode=readBuffer(2,cmd,nbcmd))!=0)
	return errorCode;
    return 0;
}

// Function: SetFilter
// Purpose:  Set the filter wheel position.
int Ms257::setFilter(int filterNumber) {
    char cmd[200];
    int nbcmd;
    int errorCode=0;
       

    sprintf(cmd,"!FILT2 %d\r",filterNumber);
    nbcmd=strlen(cmd);
    if (outputS!=NULL)
	(*outputS) << "command= " << cmd << "\n";

    
    if ((errorCode=writeBuffer(1,cmd,nbcmd))!=0)
	return errorCode;
    if ((errorCode=readBuffer(1,cmd,nbcmd))!=0)
	return errorCode;
    return 0;
}

// Function: SetShutetr
// Purpose:  Set the shutter state (0: close, 1:open).
int Ms257::setShutter(int shutterState) {
    char cmd[200];
    int nbcmd;
    int errorCode=0;
    
    sprintf(cmd,"!SHUTTER %d\r",shutterState);
    nbcmd=strlen(cmd);
    if (outputS!=NULL)
	(*outputS) << "command= " << cmd << "\n";
    
    if ((errorCode=writeBuffer(1,cmd,nbcmd))!=0)
	return errorCode;
    if ((errorCode=readBuffer(1,cmd,nbcmd))!=0)
	return errorCode;
    return 0;
}

// Function: Send error messages corresponding to the error code
char * Ms257::getErrorString(int errorCode,char * errorString) {
    switch (errorCode) {
    case 13: sprintf(errorString,"%s",E13); break;
    case 14: sprintf(errorString,"%s",E14); break;
    case 15: sprintf(errorString,"%s",E15); break;
    case 101: sprintf(errorString,"%s",E101); break;
    case 201: sprintf(errorString,"%s",E201); break;
    case 102: sprintf(errorString,"%s",E102); break;
    case 202: sprintf(errorString,"%s",E202); break;
    case 110: sprintf(errorString,"%s",E101); break;
    case 210: sprintf(errorString,"%s",E201); break;
    case 111: sprintf(errorString,"%s",E102); break;
    case 211: sprintf(errorString,"%s",E202); break;
	
    case 301:sprintf(errorString,"%s",E301); break;
    case 302:sprintf(errorString,"%s",E302); break;
    case 303:sprintf(errorString,"%s",E303); break;
    case 401:sprintf(errorString,"%s",E401); break;
    case 402:sprintf(errorString,"%s",E402); break;
    case 501:sprintf(errorString,"%s",E501); break;
	
    default: sprintf(errorString,"%s",def);
    }    
    return errorString;
} 

