/******************************************************************************
* Calculate fluxes (counts) in an elliptical aperture array;
* each ellipse is assumed to have the same centre, ellipticity and orientation;
* the array major axes are assumed to be in ascending order
******************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "const.h"
#include "img.h"
#include "geom.h"

#define  ACC       0.001     /* Relative accuracy with which areas are 
			       determined */

void apfluxe(image *img, ellipse apert[], int nap, double bg, double flux[]) {

  int          i, j, iap;
  double       nsub0, f, *maj;
  square       *pix;
  grect        pstamp;
  int          sidxndval(double *, int, double);
  double       ang_img2phys(image *, double), sqinel_sub(square, ellipse, int);
  void         errormsg(char *, ...), warnmsg(char *, ...),
               img2phys(image *, double, double, double *, double *);


  /* Make a grect from the postage stamp in physical coordinates */
  img2phys(img, (double)(img->nx + 1) / 2.0, (double)(img->ny + 1) / 2.0,
	   &(pstamp.c.x), &(pstamp.c.y));
  pstamp.alpha = ang_img2phys(img, 0.0);
  /* This isn't quite right */
  pstamp.w = img->nx;
  pstamp.h = img->ny;

  /* Make the pixels */
  if ((pix = (square *)calloc(img->npts, sizeof(square))) == NULL)
    errormsg("apfluxe(): Couldn't allocate memory!");
  for (i = 0; i < img->npts; i++) {
    img2phys(img, 1.0 + i % img->nx, 1.0 + i / img->nx, &(pix[i].c.x),
	     &(pix[i].c.y));
    pix[i].s = 1.0;
  }

  /* Copy out the major axes */
  if ((maj = (double *)calloc(nap, sizeof(double))) == NULL)
    errormsg("apfluxe(): Couldn't allocate memory!");
  for (i = 0; i < nap; i++)
    maj[i] = apert[i].maj;

  /* Reset fluxes */
  for (i = 0; i < nap; i++)
    flux[i] = 0.0;

  nsub0 = pow(sqrt(PI) * ACC * apert[0].maj / apert[0].min, -2.0/3.0);
  for (j = 0; j < img->npts; j++) {
    /* For each pixel, find the "closest" aperture */
    iap = sidxndval(maj, nap, sqrt(pow(pix[j].c.x - apert[0].c.x, 2.0) + 
				   pow(pix[j].c.y - apert[0].c.y, 2.0)));

    /* Work outwards until pixel is fully within aperture */
    f = 0.0;
    for (i = iap; i < nap && f < 1.0; i++) {
      f = sqinel_sub(pix[j], apert[i], ceil(nsub0 / apert[i].maj));
      if (!isnan(img->i[j]))
	flux[i] += f * (img->i[j] - bg);
    }
    for (; i < nap; i++) {
      if (!isnan(img->i[j]))
	flux[i] += (img->i[j] - bg);
    }

    /* Work inwards until pixel is fully outside aperture */
    f = 1.0;
    for (i = iap-1; i >= 0 && f > 0.0; i--) {
      f = sqinel_sub(pix[j], apert[i], ceil(nsub0 / apert[i].maj));
      if (!isnan(img->i[j]))
	flux[i] += f * (img->i[j] - bg);
    }
  }

  /* Clean up */
  free(pix);
  free(maj);

  return;
}
