/******************************************************************************
* Determine the flux centre of an ELT PSF
******************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <cpgplot.h>
#include "const.h"
#include "img.h"
#include "geom.h"
#include "iact.h"

#define  CONTRAST  4.0


point eltpsffc(image *img, char *imgname, int plotopt) {

  int          i, npix, ci;
  double       ps, cl, r0, see, sig, rdlc, con;
  char         buffer[L_WORD];
  extern char  *progname;
  extern icom  com;
  image        simg;
  eannulus     apert;
  grect        apertg;
  point        cip, cfp;
  plenv        env;
  int          getdkeyw(char *, char *, double *), cpgwinrename(char *),
               iscale(image *, float *, float *, int);
  double       apsbean1(image *, eannulus, double); 
  point        fluxcent(image *, double);
  void         errormsg(char *, ...), rfitsimg(char *, int, image *),
               ipplot(image *, plenv *), cpgpoint(point, int), cpggrect(grect),
               iwindowing(float *, float *, plenv *, image *, int),
               getcom(int, int, float, float, icom *);


  /* Read in the full image if necessary */
  if (img->i == 0)
    rfitsimg(imgname, 3, img);

  /* Get some keywords from the header */
  if (getdkeyw(imgname, "CDELT1", &ps))
    errormsg("eltpsffc(): Can't read keyword CDELT1!");
  if (getdkeyw(imgname, "WAVELENG", &cl))
    errormsg("eltpsffc(): Can't read keyword WAVELENG!");
  if (getdkeyw(imgname, "R0", &r0))
    errormsg("eltpsffc(): Can't read keyword R0!");

  /* Centre of the image should be at (nx + 1)/2
     but Miska seems to have centered things on 2049 */
  cip.x = ceil((img->nx + 1.0) / 2.0);
  cip.y = ceil((img->ny + 1.0) / 2.0);

  /* Seeing at cl in mas */
  see = 0.98 * cl / (r0 * pow(cl / 0.5e-6, 1.2)) / RPMAS;
  sig = see / FWHMSIG;

  /* Radius of the DLC in mas */
  rdlc = 1.22 * cl / 42.0 / RPMAS;

  /* Read in a central sub-image of size 2*npix+1, where npix is slightly 
     larger than the radius of the DLC */
  npix = (int)ceil(1.2 * rdlc / ps);
  sprintf(buffer, "%s[%d:%d,%d:%d]", imgname, (int)cip.x - npix, 
	  (int)cip.x + npix, (int)cip.y - npix, (int)cip.y + npix);
  rfitsimg(buffer, 3, &simg);

  /* Calculate contrast between central pixel and first minimum */
  apert.c.x = npix + 1;
  apert.c.y = npix + 1;
  apert.alpha = 0;
  apert.majout = rdlc / ps + 0.5;
  apert.majin = rdlc / ps - 0.5;
  apert.minout = apert.majout;
  apert.minin = apert.majin;
  con = simg.i[npix*simg.nx+npix] / apsbean1(&simg, apert, 0.0);

  /* Decide whether to use the present sub-image or a larger one */
  if (con < CONTRAST) {
    free(simg.i);
    wcsfree(simg.wcs);
    npix = (int)ceil(1.2 * 0.5 * see / ps);
    sprintf(buffer, "%s[%d:%d,%d:%d]", imgname, (int)cip.x - npix, 
	    (int)cip.x + npix, (int)cip.y - npix, (int)cip.y + npix);
    rfitsimg(buffer, 3, &simg);
  }

  /* Calculate central point */
  cfp = fluxcent(&simg, 0.0);
  cfp.x += cip.x - npix - 1;
  cfp.y += cip.y - npix - 1;

  if (plotopt) {

    /* Announce current centre */
    fprintf(stderr, "\nImage shows suggested sub-image size for centering.\n");
    fprintf(stderr, "Current centre = %.3f %.3f\n", cfp.x, cfp.y);
    fprintf(stderr, "Use cursor to change sub-image size.\n");
    fprintf(stderr, "Click lower left corner or Ctrl-q to quit.\n");

    /* Some setup */
    env.grey = 1;
    env.con = 0;
    env.phys = 0;
    strcpy(env.xlab, "x");
    strcpy(env.ylab, "y");
    sprintf(env.title, "%s: centering", imgname);

    /* Open the graphics device */
    if (cpgopen("/xwin") <= 0)
      errormsg("eltpsffc(): Couldn't open the device for PGPLOT!");
    cpgask(0);
    cpgsfs(2);
    sprintf(buffer, "%s: flux centering", progname);
    cpgwinrename(buffer);

    /* Display the full image */
    env.xmin = cip.x - 2*npix;
    env.xmax = cip.x + 2*npix;
    env.ymin = cip.y - 2*npix;
    env.ymax = cip.y + 2*npix;
    iscale(img, &(env.zmin), &(env.zmax), 1);
    ipplot(img, &env);
    
    /* Display the current sub-image size and centre */
    apertg.c = cip;
    apertg.alpha = 0.0;
    apertg.h = 2 * npix + 1;
    apertg.w = 2 * npix + 1;
    ci = 2;
    cpgsci(ci);
    cpggrect(apertg);
    cpgpoint(cfp, 2);
    cpgsci(1);
    
    while (1) {

      getcom(0, 0, com.x, com.y, &com);

      if (com.c[0]+64 == 'Q') {
	/* Quit */
	break;

      } else if (com.c[0] == 'w') {
	/* Window the plot */
	iwindowing(&com.x, &com.y, &env, img, 0);
	ipplot(img, &env);
	cpgsci(ci);
	cpggrect(apertg);
	cpgpoint(cfp, 2);
	cpgsci(1);

      } else if (com.c[0] == 's') {
	/* Scale the plot */
	i = 4;
	while ((i = iscale(img, &(env.zmin), &(env.zmax), i)))
	  if (i == 4)
	    ipplot(img, &env);
	cpgsci(ci);
	cpggrect(apertg);
	cpgpoint(cfp, 2);
	cpgsci(1);

      } else if (com.c[0] == 'A') {
	/* Read in a new sub-image */
	free(simg.i);
	wcsfree(simg.wcs);
	npix = ((cip.x - com.x) + (cip.y - com.y)) / 2.0;
	sprintf(buffer, "%s[%d:%d,%d:%d]", imgname, (int)cip.x - npix, 
		(int)cip.x + npix, (int)cip.y - npix, (int)cip.y + npix);
	rfitsimg(buffer, 3, &simg);

	/* New centre */
	cfp = fluxcent(&simg, 0.0);
	cfp.x += cip.x - npix - 1;
	cfp.y += cip.y - npix - 1;
	fprintf(stderr, "Current centre = %.3f %.3f\n", cfp.x, cfp.y);

	/* New sub-image and centre */
	apertg.h = 2 * npix + 1;
	apertg.w = 2 * npix + 1;
	ci++;
	cpgsci(ci);
	cpggrect(apertg);
	cpgpoint(cfp, 2);
	cpgsci(1);
      } else 
	fprintf(stderr, "\nSorry, don't know that command.\n");
    }

    fprintf(stderr, "\n");
    cpgclos();
  }

  /* Clean up */
  free(simg.i);
  wcsfree(simg.wcs);

  return cfp;
}
