/******************************************************************************
* Fit a simulated PSF image
******************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <cpgplot.h>
#include "iact.h"
#include "eltpsffit.h"


char      *progname, psfimgname[L_WORD];
icom      com;


/******************************************************************************
* Print the usage message
******************************************************************************/
 
void usage(void) {

  fprintf(stderr, "\nUsage: %s [options] psf.fits\n", progname);
  fprintf(stderr, "\nFit a simulated PSF image.\n");
  fprintf(stderr, "\nOptions include:\n \
\t-help\t\tprint this message\n \
\t-ws size\twindow size in cm\n \
\t-f file\t\tread commands from file\n \
\t-nodisp\t\tturn off display\n \
\n");
  exit(3);
}


/******************************************************************************
* Main program
******************************************************************************/

int main(int argc, char *argv[]) {

  int        i;
  float      ws;
  double     *fitdata1d;
  char       ofile[L_WORD], buffer[L_WORD], *buf;
  FILE       *data_file;
  image      psfimg;
  point      cent;
  ellipse    *apert;
  int        cpgwinrename(char *), mvreswin(int, int, int, int, char *);
  FILE       *faskropen(char *, char *, int);
  point      eltpsffc(image *, char *, int);
  void       errormsg(char *, ...), rfitsimg(char *, int, image *),
             rasciimg(char *, int, image *), get_input(char *, char *, ...),
             apsbean(image *, ellipse *, int, double, double *),
             eltpsffit1d(double *, int, int), eltpsffits2d(image *);


  progname = strrchr(argv[0], '/');
  progname = (progname == NULL) ? argv[0] : progname + 1;
  
  /* Look for -help switch */
  if (argc < 2)
    usage();
  for (i = 1; i < argc; i++)
    if (!strcmp(argv[i], "-help"))
      usage();

  /* Some setup */
  com.cf = com.nodisp = com.drag = 0;
  com.csize = L_COM;
  com.cfile = NULL;
  ws = WS;

  /* Look for other switches */
  for (i = 1; i < argc-1; i++)
    if (!strcmp(argv[i], "-f")) {
      com.cf = 1;
      strcpy(ofile, argv[++i]);
    } else if (!strcmp(argv[i], "-ws")) {
      if (!sscanf(argv[++i], "%f", &ws))
	errormsg("Bad window size: %s", argv[i]);
    } else if (!strcmp(argv[i], "-nodisp")) {
      com.nodisp = 1;
    } else
      errormsg("Unknown command line option: %s", argv[i]);

  /* Get the image name */
  strcpy(psfimgname, argv[argc-1]);
  
  /* Read in the image */
  if (strstr(psfimgname, ".fit") != NULL)
    rfitsimg(psfimgname, 3, &psfimg);
  else
    rasciimg(psfimgname, 3, &psfimg);

  /* Some more setup */
  if (com.cf) {
    if ((com.cfile = faskropen("Command file name?", ofile, 3)) == NULL)
      errormsg("Couldn't open %s!", ofile);
  } else if (com.nodisp) {
    com.cf = 1;
    com.cfile = stdin;
  }

  /* Open the plotting device */
  if (com.nodisp) {
    if (cpgopen("/null") <= 0)
      errormsg("Couldn't open the device for PGPLOT!");
  } else {
#ifndef NO_PGPATCH
    if (cpgopen("/xjwin") <= 0)
      errormsg("Couldn't open the device for PGPLOT!");
#else
    if (cpgopen("/xwin") <= 0)
      errormsg("Couldn't open the device for PGPLOT!");
#endif
    cpgask(0);
    cpgpap(ws/2.54, 0.8);
    cpgscr(0, 0.184, 0.31, 0.31);
    cpgscr(15, 0.0, 0.0, 0.0);
    cpgwinrename(progname);
    mvreswin(300, 15, -1, -1, progname);
  }

  /* Make some space */
  if ((fitdata1d = (double *)calloc(psfimg.nx, sizeof(double))) == NULL)
    errormsg("Couldn't allocate memory!");

  com.c[0] = 'n';
  get_input("\nDo you want to fit a 1D model to a cut along the x-axis?", "%c",
	    com.c);
  if (com.c[0] == 'y') {
    /* Copy the central line of the image */
    for (i = 0; i < psfimg.nx; i++)
      fitdata1d[i] = psfimg.i[(psfimg.ny/2 + 1) * psfimg.nx + i];

    /* Fit this */
    eltpsffit1d(fitdata1d, psfimg.nx, 1);
  }

  com.c[0] = 'n';
  get_input("\nDo you want to fit a 1D model to a cut along the y-axis?", "%c",
	    com.c);
  if (com.c[0] == 'y') {
    /* Copy the central column of the image */
    for (i = 0; i < psfimg.ny; i++)
      fitdata1d[i] = psfimg.i[i * psfimg.nx + psfimg.nx/2 + 1];
    
    /* Fit this */
    eltpsffit1d(fitdata1d, psfimg.ny, 2);
  }

  com.c[0] = 'y';
  get_input("\nDo you want to fit a 1D model to the radial profile?", "%c",
	    com.c);
  if (com.c[0] == 'y') {
    strcpy(buffer, psfimgname);
    buf = strstr(buffer, ".fit");
    strcpy(buf, ".1dp");

    if ((data_file = faskropen("1D PSF profile file name?", buffer, 4)) 
	== NULL) {

      /* No profile found, make it */
      cent = eltpsffc(&psfimg, psfimgname, 0);

      /* Make apertures */
      if ((apert = (ellipse *)calloc(psfimg.nx/2, sizeof(ellipse))) == NULL)
	errormsg("Couldn't allocate memory!");
      
      for (i = 0; i < psfimg.nx/2; i++) {
	apert[i].c = cent;
	apert[i].maj = 0.5 + (double)i;
	apert[i].min = apert[i].maj;
	apert[i].alpha = 0.0;
      }
      
      /* Calculate fluxes */
      apsbean(&psfimg, apert, psfimg.nx/2, 0.0, fitdata1d);
      free(apert);
      
    } else {
      /* Read in profile */
      i = 0;
      while (i < psfimg.nx && fgets(buffer, L_WORD, data_file) != NULL)
	if (buffer[0] != '#') {
	  if (sscanf(buffer, "%*s %lf", &(fitdata1d[i])) != 1)
	    errormsg("Having trouble reading the data!");
	  i++;
	}
      fclose(data_file);
      if (i != psfimg.nx/2)
	errormsg("Something is very wrong!");
    }
      
    /* Fit this */
    eltpsffit1d(fitdata1d, psfimg.nx/2, 3);
  }

  com.c[0] = 'y';
  get_input("\nDo you want to fit a spherically symmetric 2D model to the\
 image?", "%c", com.c);
  if (com.c[0] == 'y') {
    /* Fit this */
    eltpsffits2d(&psfimg);
  }

  /* Clean up */
  cpgend();
  free(psfimg.i);
  if (psfimg.wcs != 0)
    wcsfree(psfimg.wcs);
  free(fitdata1d);

  return 0;
}
