/******************************************************************************
* Fit a 1D PSF
******************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <cpgplot.h>
#include "const.h"
#include "iact.h"
#include "eltpsffit.h"


float          xfit[N_PTS1DF], yfit[N_PTS1DF];
data1dim       data1d;
env1dim        env1d;
fit1dim        fit1d;


void eltpsffit1d(double *oydata, int onpts, int type) {

  int          i, imin, imax;
  double       help;
  extern char  psfimgname[];
  extern icom  com;
  int          sidxnval(float *, int, float),
               getdkeyw(char *, char *, double *);
  void         errormsg(char *, ...), nferrormsg(char *, ...), epf1dplot(void),
               epf1dwindowing(icom *, env1dim *, data1dim *),
               epf1dfs(void), epf1dwmodel(void), epf1drmodel(void),
               getcom(int, int, float, float, icom *), epf1daddc(icom *),
               epf1ddelc(icom *), epf1deditc(icom *), epf1ddofit(void),
               epf1dse(icom *);


  /* Some setup */
  data1d.y = oydata;
  data1d.npts = onpts;
  fit1d.nc = 0;
  fit1d.c = NULL;
  fit1d.type = type;

  /* Make space for the various data arrays */
  if ((data1d.x = (double *)calloc(data1d.npts, sizeof(double))) == NULL)
    errormsg("eltpsffit1d(): Couldn't allocate memory!");
  if ((data1d.s = (double *)calloc(data1d.npts, sizeof(double))) == NULL)
    errormsg("eltpsffit1d(): Couldn't allocate memory!");
  if ((data1d.r = (double *)calloc(data1d.npts, sizeof(double))) == NULL)
    errormsg("eltpsffit1d(): Couldn't allocate memory!");
  if ((data1d.d = (unsigned short *)calloc(data1d.npts, 
					   sizeof(unsigned short))) == NULL)
    errormsg("eltpsffit1d(): Couldn't allocate memory!");

  if ((data1d.fx = (float *)calloc(data1d.npts, sizeof(float))) == NULL)
    errormsg("eltpsffit1d(): Couldn't allocate memory!");
  if ((data1d.fy = (float *)calloc(data1d.npts, sizeof(float))) == NULL)
    errormsg("eltpsffit1d(): Couldn't allocate memory!");
  if ((data1d.ly = (float *)calloc(data1d.npts, sizeof(float))) == NULL)
    errormsg("eltpsffit1d(): Couldn't allocate memory!");
  if ((data1d.fs = (float *)calloc(data1d.npts, sizeof(float))) == NULL)
    errormsg("eltpsffit1d(): Couldn't allocate memory!");
  if ((data1d.ls = (float *)calloc(data1d.npts, sizeof(float))) == NULL)
    errormsg("eltpsffit1d(): Couldn't allocate memory!");
  if ((data1d.fr = (float *)calloc(data1d.npts, sizeof(float))) == NULL)
    errormsg("eltpsffit1d(): Couldn't allocate memory!");
  if ((data1d.lr = (float *)calloc(data1d.npts, sizeof(float))) == NULL)
    errormsg("eltpsffit1d(): Couldn't allocate memory!");
  
  /* Prepare data arrays */
  if (getdkeyw(psfimgname, "CDELT1", &help))
    errormsg("eltpsffit1d(): Couldn't get CDELT1 keyword!");
  for (i = 0; i < data1d.npts; i++) {
    if (fit1d.type == 1 || fit1d.type == 2)
      data1d.x[i] = (double)(i + 1) * help;
    else if (fit1d.type == 3)
      data1d.x[i] = (double)i * help;
    else
      errormsg("eltpsffit1d(): Unknown fit type: %d!", fit1d.type);
    data1d.d[i] = 0;
    data1d.fx[i] = data1d.x[i];
    data1d.fy[i] = data1d.y[i];
    data1d.ly[i] = (data1d.y[i] <= 0.0) ? 0.0 : log10(data1d.y[i]);
    data1d.s[i] = 1.0 / sqrt((double)data1d.npts);
    data1d.fs[i] = data1d.s[i];
    data1d.ls[i] = (data1d.y[i] <= 0.0) ? 1.0e6 : 
      data1d.s[i] / data1d.y[i] * LOGE;
    data1d.r[i] = 0.0;
    data1d.fr[i] = 0.0;
    data1d.lr[i] = 0.0;  
  }

  /* Initial plotting setup */
  env1d.sym = 1;
  env1d.noerrb = 1;
  env1d.ch = 0.6;
  env1d.ls = 1;
  env1d.noerrb = 1;
  env1d.ic = 1;
  env1d.res = 0;
  env1d.logp = 1;
  if (fit1d.type == 1) {
    strcpy(env1d.title, "1D PSF cross-section");
    strcpy(env1d.xlab, "x (mas)");
  } else if (fit1d.type == 2) {
    strcpy(env1d.title, "1D PSF cross-section");
    strcpy(env1d.xlab, "y (mas)");
  } else if (fit1d.type == 3) {
    strcpy(env1d.title, "1D PSF radial profile");
    strcpy(env1d.xlab, "r (mas)");
  }
  if (env1d.logp)
    strcpy(env1d.ylab, "log(PSF)");
  else
    strcpy(env1d.ylab, "PSF");
  
  if (env1d.logp) {
    data1d.yd = data1d.ly;
    data1d.ys = data1d.ls;
  } else {
    data1d.yd = data1d.fy;
    data1d.ys = data1d.fs;
  }
  com.c[0] = 'a';
  epf1dwindowing(&com, &env1d, &data1d);
  com.x = (env1d.xmin + env1d.xmax) / 2.0;
  com.y = (env1d.ymin + env1d.ymax) / 2.0;
  for (i = 0; i < N_PTS1DF; i++)
    xfit[i] = data1d.x[0] + (data1d.x[data1d.npts-1] - data1d.x[0]) * (float)i 
      / (float)(N_PTS1DF-1);
  
  /* Read in a previously saved model, if there is one */
  epf1drmodel();

  /* Fill in the residuals */
  epf1dfs();

  /* Initial plot */
  epf1dplot();

  /* Enter interactive loop */
  if (!(com.cf))
    fprintf(stderr, "\nType \"?\" for a list of commands.\n");
  while (1) {
    
    getcom(0, 1, com.x, com.y, &com);

    if (com.c[0]+64 == 'Q') {
      /* Quit */
      break;

    } else if (com.c[0] == '?') {
      /* Help menu */
      fprintf(stderr, "\nList of interactive commands:\n\n");
      fprintf(stderr, "\t?\t\tprint this list\n \
\tw\t\twindow the plot\n \
\tf\t\trefresh the plot\n \
\ta\t\tadd a new model component\n \
\tlmb\t\tedit a model component\n \
\td\t\tdelete a model component\n \
\tx\t\tfix/release a fit component's parameter\n \
\tF\t\tfit\n \
\tCtrl-s\t\tsave the current model\n \
\tCtrl-r\t\tread a previously saved model\n \
\tc\t\ttoggle plotting of individual components\n \
\te\t\ttoggle plotting of fitting weights (= error bars)\n \
\tr\t\ttoggle plotting of residuals\n \
\tl\t\ttoggle log scale\n \
\ts\t\tshow 'fit' statistics of current model\n \
\tz\t\tset fitting weights\n\
\tspc\t\tprint info on data point nearest to cursor\n \
\tk\t\treject data points from fit\n \
\tu\t\tunreject data points\n \
\ti\t\tunreject all data points\n \
\tCtrl-i\t\tinterrupt command file mode\n \
\tCtrl-c\t\tcontinue with command file\n \
\tCtrl-q\t\tdone fitting\n\n");

    } else if (com.c[0] == 'w' || com.c[0] == 'Y' || com.c[0] == 'P' || 
	       com.c[0]+64 == 'Y' || com.c[0]+64 == 'P' || 
	       com.c[0] == 'y' || com.c[0] == 'p' || com.c[0] == 'X' ||
	       com.c[0] == 'D' || (com.c[0] == 'A' && com.drag)) {
      /* Window the plot */
      epf1dwindowing(&com, &env1d, &data1d);
      epf1dplot();

    } else if (com.c[0] == 'f') {
      /* Refresh the plot */
      epf1dplot();

    } else if (com.c[0] == 'e') {
      /* Toggle plotting of errorbars */
      env1d.noerrb = !env1d.noerrb;
      epf1dplot();

    } else if (com.c[0] == ' ') {
      /* Info on point */
      i = sidxnval(data1d.fx, data1d.npts, com.x);
      printf("\nData point: i = %d\n", i);
      printf("x = %g   y = %g   sig = %g   res = %g\n", data1d.x[i],
	     data1d.y[i], data1d.s[i], data1d.r[i]);
      printf("Cursor:     x = %g   y = %g\n", com.x, com.y);

    } else if (com.c[0] == 's') {
      /* Print fit statistics */
      epf1dfs();

    } else if (com.c[0] == 'z') {
      /* Set errors */
      epf1dse(&com);

    } else if (com.c[0] == 'k' || com.c[0] == 'u') {
      /* Delete data points */
      imin = sidxnval(data1d.fx, data1d.npts, com.x);
      fprintf(stderr, "\n... again ...\n");
      cpgsci(7);
      getcom(4, 1, com.x, com.y, &com);
      cpgsci(1);
      if (com.c[0] == 'k' || com.c[0] == 'u') {
	imax = sidxnval(data1d.fx, data1d.npts, com.x);
	if (imax < imin) {
	  i = imax;
	  imax = imin;
	  imin = i;
	}
	if (com.c[0] == 'k')
	  for (i = imin; i < imax; i++)
	    data1d.d[i] = 1;
	else
	  for (i = imin; i < imax; i++)
	    data1d.d[i] = 0;
	epf1dplot();
      }

    } else if (com.c[0] == 'i') {
      /* Undelete all data points */
      for (i = 0; i < data1d.npts; i++)
	data1d.d[i] = 0;
      epf1dplot();

    } else if (com.c[0] == 'a' && 
	       (data1d.yd == data1d.fy || data1d.yd == data1d.ly)) {
      /* Add a new component */
      epf1daddc(&com);
      epf1dplot();

    } else if (com.c[0] == 'd' && 
	       (data1d.yd == data1d.fy || data1d.yd == data1d.ly) &&
	       env1d.ic) {
      /* Delete a component */
      epf1ddelc(&com);
      epf1dplot();

    } else if ((com.c[0] == 'A' || com.c[0] == 'x') && 
	       (data1d.yd == data1d.fy || data1d.yd == data1d.ly) && 
	       env1d.ic) {
      /* Edit a component */
      epf1deditc(&com);
      epf1dplot();

    } else if (com.c[0] == 'r') {
      /* Toggle plotting of residuals */
      if (env1d.res) {
	env1d.res = 0;
	if (env1d.logp) {
	  strcpy(env1d.ylab, "log(PSF)");
	  data1d.yd = data1d.ly;
	} else {
	  strcpy(env1d.ylab, "PSF");
	  data1d.yd = data1d.fy;
	}
      } else {
	env1d.res = 1;
	if (env1d.logp) {
	  strcpy(env1d.ylab, "log(PSF / fit)");
	  data1d.yd = data1d.lr;
	} else {
	  strcpy(env1d.ylab, "PSF - fit");
	  data1d.yd = data1d.fr;
	}
      }
      com.c[0] = '=';
      epf1dwindowing(&com, &env1d, &data1d);
      epf1dplot();

    } else if (com.c[0] == 'l') {
      /* Toggle log scale */
      if (env1d.logp) {
	env1d.logp = 0;
	if (env1d.res) {
	  strcpy(env1d.ylab, "PSF - fit");
	  data1d.yd = data1d.fr;
	} else {
	  strcpy(env1d.ylab, "PSF");
	  data1d.yd = data1d.fy;
	}
	data1d.ys = data1d.fs;
      } else {
	env1d.logp = 1;
	if (env1d.res) {
	  strcpy(env1d.ylab, "log(PSF / fit)");
	  data1d.yd = data1d.lr;
	} else {
	  strcpy(env1d.ylab, "log(PSF)");
	  data1d.yd = data1d.ly;
	}
	data1d.ys = data1d.ls;
      }
      com.c[0] = '=';
      epf1dwindowing(&com, &env1d, &data1d);
      epf1dplot();

    } else if (com.c[0] == 'c') {
      /* Toggle plotting of individual components */
      env1d.ic = !(env1d.ic);
      epf1dplot();

    } else if (com.c[0] == 'F') {
      /* Fit */
      epf1ddofit();
      epf1dplot();

    } else if (com.c[0]+64 == 'S') {
      /* Save current model */
      epf1dwmodel();

    } else if (com.c[0]+64 == 'R') {
      /* Read model */
      epf1drmodel();
      epf1dfs();
      epf1dplot();

    } else 
      fprintf(stderr, "\nSorry, don't know that command.\n");
  }

  /* Save current model */
  epf1dwmodel();

  /* Clean up */
  free(data1d.x);
  free(data1d.fx);
  free(data1d.fy);
  free(data1d.ly);
  free(data1d.s);
  free(data1d.fs);
  free(data1d.ls);
  free(data1d.r);
  free(data1d.fr);
  free(data1d.lr);
  free(data1d.d);
  if (fit1d.nc > 0)
    free(fit1d.c);

  return;
}
