/****************************************************************************
* Write out the current model
****************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <fitsio.h>
#include "eltpsffit.h"


void epfs2dwmodel(void) {

  int                i, status;
  float              *help;
  double             prof0, prof, contrast;
  char               *buf, buffer[L_WORD];
  extern char        psfimgname[];
  FILE               *data_file;
  fitsfile           *fptr;
  extern datas2dim   datas2d;
  extern fits2dim    fits2d;
  double             epfs2deval(double, int), epfs2dreval(double, int);
  FILE               *faskwopen(char *, char *, int);
  void               errormsg(char *, ...), nferrormsg(char *, ...),
                     ferrormsg(int, char *, ...);


  if (fits2d.nc != 0) {

    /* Write out the model */
    /* Make the file name */ 
    strcpy(buffer, psfimgname);
    buf = strstr(buffer, ".fit");
    strcpy(buf, ".s2df");
 
    /* Open the file */
    if ((data_file = faskwopen("s2D fit file name?", buffer, 4)) == NULL)
      errormsg("epfs2dwmodel(): Couldn't open %s!", buffer);
  
    /* Write out */
    fprintf(data_file, "# Description of spherically symmetric 2D model\n");
    fprintf(data_file,
	    "# comp h             x [mas]       y [mas]      pi  w [mas]       q           rr [mas]\n");
    for (i = 0; i < fits2d.nc; i++)
      fprintf(data_file,
	      "%-4s  %12.5e  %12.5e  %12.5e  %2d  %11.5e  %11.4e  %11.5e\n",
	      fits2d.c[i].type, fits2d.c[i].h, fits2d.c[fits2d.c[i].pi].p.x, 
	      fits2d.c[fits2d.c[i].pi].p.y, fits2d.c[i].pi, fits2d.c[i].w,
	      fits2d.c[i].q, fits2d.c[i].rr);

    fclose(data_file);
    fprintf(stderr, "\nSaved current model.\n");

    /* Write out the radial profile of PSF fit */
    /* Make the file name */
    strcpy(buf, ".s2dfp");
    
    /* Open the file */
    if ((data_file = faskwopen("s2D radial profile file name?", buffer, 4)) 
	== NULL)
      errormsg("epfs2dwmodel(): Couldn't open %s!", buffer);

    /* Write out */
    fprintf(data_file,
	    "# Radial PSF profile of spherically symmetric 2D model\n");
    fprintf(data_file, "# r [mas]      PSF\n");
    prof0 = epfs2dreval(0.0, fits2d.nc);
    contrast = P_CONT;
    for (i = 0; i < fits2d.nc; i++)
      if (!strcmp(fits2d.c[i].type, "c") && fits2d.c[i].h / prof0 > contrast) {
	contrast = 1.01 * fits2d.c[i].h / prof0;
	nferrormsg("Model contains a constant above the desired\n\
contrast level. Resetting contrast to %e.", contrast);
      }
    prof = prof0;
    for (i = 0; prof / prof0 > contrast; i++) {
      prof = epfs2dreval((double)i * P_STEP * datas2d.img->wcs->cdelt[0],
			 fits2d.nc);
      fprintf(data_file, "%e   %.8e\n", 
	      (double)i * P_STEP * datas2d.img->wcs->cdelt[0], prof);
    }

    fclose(data_file);
    fprintf(stderr, "\nSaved radial profile of current model.\n");

    /* Write out the full 2D PSF */
    /* Make the file name */
    sprintf(buffer, "!%s", psfimgname);
    buf = strstr(buffer, ".fit");
    strcpy(buf, "_s2df.fits");
    
    /* Open the data file and create header */
    status = 0;
    if (fits_create_template(&fptr, buffer, psfimgname, &status))
      ferrormsg(status, "epfs2dwmodel(): Couldn't create %s!", buffer);
    if (fits_write_key(fptr, TSTRING, "COMMENT", 
		       "Spherically symmetric 2D model", 0, &status))
      ferrormsg(status, "epfs2dwmodel(): Couldn't write keyword to %s!",
		buffer);
    
    /* Make some space */
    if ((help = (float *)calloc(datas2d.img->npts, sizeof(float))) == NULL)
      errormsg("epfs2dwmodel(): Couldn't allocate memory!");

    /* Make the fitted image */
    for (i = 0; i < datas2d.img->npts; i++)
      help[i] = epfs2deval(datas2d.x[i], fits2d.nc);
    
    /* Write out */
    if (fits_write_img(fptr, TFLOAT, 1, datas2d.img->npts, help, &status))
      ferrormsg(status, "epfs2dwmodel(): Couldn't write data to %s!", buffer);
    
    /* Close the fits file */
    if (fits_close_file(fptr, &status))
      ferrormsg(status, "epfs2dwmodel(): Couldn't close %s!", buffer);
    free(help);
    fprintf(stderr, "\nSaved image of current model.\n");
  }

  return;
}
