/*****************************************************************************
* Open a file for reading with optional file name query, checking whether
* the file exists, system escape, and remote file access (with or without 
* buffering). A filename of "-" signifies stdin.
* Options:
* 1 = don't have a default filename
* 2 = have a default, but query the user
* 3 = have a default, query only if the file doesn't exist
* 4 = have a default, use it in any case (like fopen)
*****************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#ifdef J_RFA
#include <curl/curl.h>
#endif


FILE *faskropen(char *query, char *filename, int opt) {

  int        remote;
  char       c;
  FILE       *data_file;
  void       errormsg(char *, ...), nferrormsg(char *, ...), 
             get_input(char *, char *, ...);
#if defined J_RFA
  CURL       *curl;
  CURLcode   res;
  char       errbuf[CURL_ERROR_SIZE];
#elif defined J_RFA_NB
  FILE       *remote_fopen(char *);
#endif


  /* Queries */
  if (opt == 1)
    get_input(query, "!%s", filename);
  else if (opt == 2)
    get_input(query, "%s", filename);
  else if (opt == 3 || opt == 4)
    ;
  else
    errormsg("faskropen(): Unknown option: %d", opt);

  /* Handle stdin */
  if (!strcmp(filename, "-")) {
    data_file = tmpfile();
    while ((c = getc(stdin)) != EOF)
      putc(c, data_file);
    rewind(data_file);
    return data_file;
  }

  remote = 0;
  if (strstr(filename, "://") != NULL)
    remote = 1;

  if (!remote) {
    /* Local file */
    while (opt != 4 && (access(filename, R_OK) == -1 || filename[0] == '!')) {
      if (filename[0] == '!')
	system(filename+1);
      else
	nferrormsg("File %s doesn't exist or isn't readable!", filename);
      get_input(query, "!%s", filename);
    }
    return fopen(filename, "r");

  } else {
    /* Remote file */
#if defined J_RFA
    if ((curl = curl_easy_init()) == NULL)
      errormsg("faskropen(): Couldn't initialise curl!");
    /*
    curl_easy_setopt(curl, CURLOPT_VERBOSE, 1);
    */
    curl_easy_setopt(curl, CURLOPT_ERRORBUFFER, errbuf);
    curl_easy_setopt(curl, CURLOPT_FAILONERROR, 1);
    do {
      if ((data_file = tmpfile()) == NULL)
	errormsg("faskropen(): Couldn't open temporary file!");
      curl_easy_setopt(curl, CURLOPT_WRITEDATA, data_file);
      curl_easy_setopt(curl, CURLOPT_URL, filename);  
      if ((res = curl_easy_perform(curl))) {
	fclose(data_file);
	if (opt == 4) {
	  curl_easy_cleanup(curl);
	  return NULL;
	}
	if (filename[0] == '!')
	  system(filename+1);
	else
	  nferrormsg("Remote file transfer failed for\n%s!\n%s", filename,
		     errbuf);
	get_input(query, "!%s", filename);
      }
    } while (res);
    curl_easy_cleanup(curl);
    rewind(data_file);
    return data_file;

#elif defined J_RFA_NB
    return remote_fopen(filename);

#else
    errormsg("faskropen(): No support for remote file access!");

#endif
  }

  /* Should never get here */
  return NULL;
}
