/******************************************************************************
* Multiple component fitting function;
* Uses NR index convention
******************************************************************************/

#include <string.h>
#include <math.h>
#include <gsl/gsl_vector.h>
#include <gsl/gsl_matrix.h>
#include <gsl/gsl_sf_bessel.h>
#include "const.h"
#include "nlfit.h"
#include "eltpsffit.h"


int fepfmc1d(const gsl_vector *a, void *data, gsl_vector *f, gsl_matrix *J) {

  int                i, j, k, pi;
  size_t             ipts, npts;
  double             *x, *y, *sig, *yf, arg, arg2, qarg, g, dg, b1, qb1, b0,
                     qb0, dq, q2, omq2;
  extern fit1dim     fit1d;
  void               errormsg(char *, ...), nferrormsg(char *, ...);


  /* Data */
  npts = ((fit_data *)data)->n;
  x = ((fit_data *)data)->x;
  y = ((fit_data *)data)->y;
  sig = ((fit_data *)data)->sig;
  yf = ((fit_data *)data)->yf;

  /* Initialise function values */
  for (ipts = 0; ipts < npts; ipts++)
    yf[ipts] = 0.0;

  /* Initialise Jacobian */
  gsl_matrix_set_zero(J);
 
  /* Step through each component */
  j = 0;
  for (i = 0; i < fit1d.nc; i++) {

    /* Find the index in a of the current component's position */
    pi = 0;
    for (k = 0; k < fit1d.c[i].pi; k++)
      pi += fit1d.c[k].npar;
    pi += fit1d.c[k].npar - 1;
  
    /* Negative width is out of bounds */
    if (strcmp(fit1d.c[i].type, "c") && a->data[j+1] <= 0.0) {
      nferrormsg("fepfmc1d(): Negative width in %s component!", 
		 fit1d.c[i].type);
      for (ipts = 0; ipts < npts; ipts++)
	gsl_vector_set(f, ipts, 1.0e30);
      return GSL_SUCCESS;
    }

    /* Calculation of value and derivatives */
    if (!strcmp(fit1d.c[i].type, "g")) {
      for (ipts = 0; ipts < npts; ipts++) {
	arg = (x[ipts] - a->data[pi]) / a->data[j+1];
	g = exp(-arg * arg);
	dg = 2.0 * arg * g;
	yf[ipts] += a->data[j] * g;
	if (fit1d.c[i].hfit)
	  gsl_matrix_set(J, ipts, j, g / sig[ipts]);
	if (fit1d.c[i].wfit)
	  gsl_matrix_set(J, ipts, j+1,
			 a->data[j] * arg / a->data[j+1] * dg / sig[ipts]);
	if (fit1d.c[fit1d.c[i].pi].pfit)
	  gsl_matrix_set(J, ipts, pi,
			 a->data[j] / a->data[j+1] * dg  / sig[ipts]
			 + gsl_matrix_get(J, ipts, pi));
      }
      
    } else if (!strcmp(fit1d.c[i].type, "l")) {
      for (ipts = 0; ipts < npts; ipts++) {
	arg = (x[ipts] - a->data[pi]) / a->data[j+1];
	g = 1.0 / (arg * arg + 1.0);
	dg = 2.0 * arg * g * g;
	yf[ipts] += a->data[j] * g;
	if (fit1d.c[i].hfit)
	  gsl_matrix_set(J, ipts, j, g / sig[ipts]);
	if (fit1d.c[i].wfit)
	  gsl_matrix_set(J, ipts, j+1,
			 a->data[j] * arg / a->data[j+1] * dg / sig[ipts]);
	if (fit1d.c[fit1d.c[i].pi].pfit)
	  gsl_matrix_set(J, ipts, pi,
			 a->data[j] / a->data[j+1] * dg / sig[ipts]
			 + gsl_matrix_get(J, ipts, pi));
      }

    } else if (!strcmp(fit1d.c[i].type, "m")) {
      if (a->data[j+2] <= 0.0) {
	nferrormsg("fepfmc1d(): Negative Moffat index!");
	for (ipts = 0; ipts < npts; ipts++)
	  gsl_vector_set(f, ipts, 1.0e30);
	return GSL_SUCCESS;
      }
      for (ipts = 0; ipts < npts; ipts++) {
	arg = (x[ipts] - a->data[pi]) / a->data[j+1];
	arg2 = arg * arg + 1.0;
	g = 1.0 / pow(arg2, a->data[j+2]);
	dg = 2.0 * arg * a->data[j+2] / arg2 * g;
	yf[ipts] += a->data[j] * g;
	if (fit1d.c[i].hfit)
	  gsl_matrix_set(J, ipts, j, g / sig[ipts]);
	if (fit1d.c[i].wfit)
	  gsl_matrix_set(J, ipts, j+1,
			 a->data[j] * arg / a->data[j+1] * dg / sig[ipts]);
	if (fit1d.c[i].qfit)
	  gsl_matrix_set(J, ipts, j+2,
			 -a->data[j] * log(arg2) * g / sig[ipts]);
	if (fit1d.c[fit1d.c[i].pi].pfit)
	  gsl_matrix_set(J, ipts, pi,
			 a->data[j] / a->data[j+1] * dg / sig[ipts]
			 + gsl_matrix_get(J, ipts, pi));
      }

    } else if (!strcmp(fit1d.c[i].type, "s")) {
      for (ipts = 0; ipts < npts; ipts++) {
	if (x[ipts] == a->data[pi]) {
	  arg = 0.0;
	  g = 1.0;
	  dg = 0.0;
	} else {
	  arg = (x[ipts] - a->data[pi]) / a->data[j+1];
	  g = sin(arg) / arg;
	  dg = (g - cos(arg)) / arg;
	}
	yf[ipts] += a->data[j] * g;
	if (fit1d.c[i].hfit)
	  gsl_matrix_set(J, ipts, j, g / sig[ipts]);
	if (fit1d.c[i].wfit)
	  gsl_matrix_set(J, ipts, j+1,
			 a->data[j] * arg / a->data[j+1] * dg / sig[ipts]);
	if (fit1d.c[fit1d.c[i].pi].pfit)
	  gsl_matrix_set(J, ipts, pi,
			 a->data[j] / a->data[j+1] * dg / sig[ipts]
			 + gsl_matrix_get(J, ipts, pi));
      }

    } else if (!strcmp(fit1d.c[i].type, "a")) {
      for (ipts = 0; ipts < npts; ipts++) {
	if (x[ipts] == a->data[pi]) {
	  arg2 = 0.0;
	  arg = 0.0;
	  g = 1.0;
	  dg = 0.0;
	} else {
	  arg2 = (x[ipts] - a->data[pi]) * RPMAS;
	  arg =  sin(arg2) / (a->data[j+1] * RPMAS);
	  g = 2.0 * gsl_sf_bessel_J1(arg) / arg;
	  dg = 4.0 / arg * g * (g - gsl_sf_bessel_J0(arg));
	  g = g * g;
	}
	if (fabs(arg2 / (a->data[j+1] * RPMAS)) < AIRC) {
	  yf[ipts] += a->data[j] * g;
	  if (fit1d.c[i].hfit)
	    gsl_matrix_set(J, ipts, j, g / sig[ipts]);
	  if (fit1d.c[i].wfit)
	    gsl_matrix_set(J, ipts, j+1,
			   a->data[j] * arg / a->data[j+1] * dg / sig[ipts]);
	  if (fit1d.c[fit1d.c[i].pi].pfit)
	    gsl_matrix_set(J, ipts, pi,
			   a->data[j] * cos(arg2) / a->data[j+1] * dg 
			   / sig[ipts]
			   + gsl_matrix_get(J, ipts, pi));
	}
      }
      
    } else if (!strcmp(fit1d.c[i].type, "o")) {
      if (a->data[j+2] <= 0.0 || a->data[j+2] >= 1.0) {
	nferrormsg("fepfmc1d(): Contrast parameter out of range: %e!", 
		   a->data[j+2]);
	for (ipts = 0; ipts < npts; ipts++)
	  gsl_vector_set(f, ipts, 1.0e30);
	return GSL_SUCCESS;
      }
      for (ipts = 0; ipts < npts; ipts++) {
	if (x[ipts] == a->data[pi]) {
	  arg2 = 0.0;
	  arg = 0.0;
	  g = 1.0;
	  dg = 0.0;
	  dq = 0.0;
	} else {
	  arg2 = (x[ipts] - a->data[pi]) * RPMAS;
	  arg = sin(arg2) / (a->data[j+1] * RPMAS);
	  qarg = a->data[j+2] * arg;
	  b1 = gsl_sf_bessel_J1(arg) / arg;
	  qb1 = gsl_sf_bessel_J1(qarg) / qarg;
	  b0 = gsl_sf_bessel_J0(arg);
	  qb0 = gsl_sf_bessel_J0(qarg);
	  q2 = a->data[j+2] * a->data[j+2];
	  omq2 = 1.0 - q2;
	  g = 2.0 / omq2 * (b1 - q2 * qb1);
	  dg = 4.0 / arg * g * (g - (b0 - q2 * qb0) / omq2);
	  dq = 4.0 * a->data[j+2] / omq2 * g * (g - qb0);
	  g = g * g;
	}
	if (fabs(arg2 / (a->data[j+1] * RPMAS)) < AIRC) {      
	  yf[ipts] += a->data[j] * g;
	  if (fit1d.c[i].hfit)
	    gsl_matrix_set(J, ipts, j, g / sig[ipts]);
	  if (fit1d.c[i].wfit)
	    gsl_matrix_set(J, ipts, j+1,
			   a->data[j] * arg / a->data[j+1] * dg / sig[ipts]);
	  if (fit1d.c[i].qfit)
	    gsl_matrix_set(J, ipts, j+2,
			   a->data[j] * dq / sig[ipts]);
	  if (fit1d.c[fit1d.c[i].pi].pfit)
	    gsl_matrix_set(J, ipts, pi,
			   a->data[j] * cos(arg2) / a->data[j+1] * dg 
			   / sig[ipts]
			   + gsl_matrix_get(J, ipts, pi));
	}
      }

    } else if (!strcmp(fit1d.c[i].type, "c")) {
      for (ipts = 0; ipts < npts; ipts++) {
	yf[ipts] += a->data[j];
	if (fit1d.c[i].hfit)
	  gsl_matrix_set(J, ipts, j, 1.0 / sig[ipts]);
      }

    } else
      errormsg("fepfmc1d(): Unknown component type: %s", fit1d.c[i].type);

    /* Increase counter */
    j += fit1d.c[i].npar;
  }

  /* Set the output vector */
  for (ipts = 0; ipts < npts; ipts++)
    gsl_vector_set(f, ipts, (yf[ipts] - y[ipts]) / sig[ipts]);

  return GSL_SUCCESS;
}
