/******************************************************************************
* Find plotting scale for an image; return value is used to indicate whether
* the function should be called again during interactive use.
* Options:
* 1 = minmax
* 2 = minmax of central image region
* 3 = sigma clipping
* 4 = choose interactively
******************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include <cpgplot.h>
#include "img.h"

#define  N_CENT  20     /* Size of the central image region */
#define  N_MAXP  10000  /* Maximum number of points used in sigma clipping */
#define  N_BIN   10000  /* Maximum number of bins */


static int     nbin;
static float   xb[N_BIN+1], h[N_BIN], xmin, xmax, ymin, ymax;
static char    scstring[50];


/******************************************************************************
* Plotting subroutine for interactive use
******************************************************************************/

void plot(float zmin, float zmax) {


  /* Plot the histogram */
  cpgbbuf();
  cpgscr(0, 0.184, 0.31, 0.31);
  cpgeras();
  cpgscr(0, 0.0, 0.0, 0.0);
  cpgswin(xmin, xmax, ymin, ymax);
  cpgsci(0);
  cpgrect(xmin, xmax, ymin, ymax);
  cpgsci(1);
  cpgbin(nbin, xb, h, 0);
  cpgsci(5);
  cpgbox("BCTS", 0.0, 0, "BCTS", 0.0, 0);
  cpgsci(7);
  cpgbox("N", 0.0, 0, "N", 0.0, 0);
  cpgsci(3);
  cpglab("Data value", "Log frequency", " ");
  cpgmtxt("T", -1.6, 0.65, 0.0, scstring);
  cpgsci(7);
  cpgmove(zmin, ymin);
  cpgdraw(zmin, ymax);
  cpgmove(zmax, ymin);
  cpgdraw(zmax, ymax);
  cpgsci(1);
  cpgebuf();

  return;
}


/******************************************************************************
* Scaling routine
******************************************************************************/

int iscale(image *img, float *zmin, float *zmax, int opt) {

  int            i, j, ix, iy, n, odev, r;
  static int     dev=0;
  float          *buf, dist[4];
  static float   x, y, bins;
  char           com;
  static image   *oimg;
  int            sigclip(float *, float *, int, float),
                 idxval(float *, int, float),
                 cpgwinrename(char *), mvreswin(int, int, int, int, char *);
  float          jmin(float *, int), jmax(float *, int);
  double         percentile(float *, int, double);
  void           errormsg(char *, ...),
                 bin(float *, int, float *, float *, int);


  if (img->npts < 2)
    errormsg("iscale(): Not enough points in image: %d", img->npts);

  if (opt == 2 && (img->nx <= N_CENT || img->ny <= N_CENT))
    opt = 1;

  if (opt == 1) {
    *zmin = jmin(img->i, img->npts);
    *zmax = jmax(img->i, img->npts);
    r = 4;

  } else if (opt == 2) {
    /* Make some work space */
    if ((buf = (float *)calloc(img->npts, sizeof(float))) == NULL)
      errormsg("iscale(): Couldn't allocate memory!");

    /* Copy the central region of the image */
    n = 0;
    for (iy = img->ny/2 - N_CENT/2; iy < img->ny/2 + N_CENT/2; iy++)
      for (ix = img->nx/2 - N_CENT/2; ix < img->nx/2 + N_CENT/2; ix++) {
	i = ix + iy * img->nx;
	if (!isnan(img->i[i]) && img->i[i] > -9000.0)
	  buf[n++] = img->i[i];
      }
    *zmin = jmin(buf, n);
    *zmax = jmax(buf, n);

    /* Clean up */
    free(buf);
    r = 4;

  } else if (opt == 3) {

    if (img->npts > N_MAXP) {
      /* Make some work space */
      if ((buf = (float *)calloc(N_MAXP, sizeof(float))) == NULL)
	errormsg("iscale(): Couldn't allocate memory!");

      /* Sample the image */
      n = (int)ceil((float)(img->npts) / (float)N_MAXP);
      j = 0;
      for (i = 0; i < img->npts; i += n)
	if (!isnan(img->i[i]) && img->i[i] > -9000.0)
	  buf[j++] = img->i[i];
      i = j;
    } else {
      /* Make some work space */
      if ((buf = (float *)calloc(img->npts, sizeof(float))) == NULL)
	errormsg("iscale(): Couldn't allocate memory!");

      /* Copy the image */
      j = 0;
      for (i = 0; i < img->npts; i++)
	if (!isnan(img->i[i]) && img->i[i] > -9000.0)
	  buf[j++] = img->i[i];
      i = j;
    }
    
    /* Sigma clip */
    do {
      n = i;
      i = sigclip(buf, buf, n, 3.0);
    } while (i != n);

    if (n != 0) {
      *zmin = jmin(buf, n);
      *zmax = jmax(buf, n);
      *zmax += 3.0 * (*zmax - *zmin);
    }

    /* Clean up */
    free(buf);
    r = 4;

  } else if (opt == 4) {
    cpgqid(&odev);

    if (dev == 0) {
      cpgqitf(&i);
      if (i == 0)
	sprintf(scstring, "Scaling: linear");
      if (i == 1)
	sprintf(scstring, "Scaling: log");
      if (i == 2)
	sprintf(scstring, "Scaling: square-root");

      /* Open the graphics device */
      if ((dev = cpgopen("/xwin")) <= 0)
	errormsg("iscale(): Couldn't open the device for PGPLOT!");
      cpgpap(4.0, 0.5);
      cpgask(0);
      cpgsch(2.2);
      cpgsvp(0.08, 0.99, 0.18, 0.99);
      cpgwinrename("Image scaling");
      mvreswin(395, 15, 0, 0, "Image scaling");
    } else
      cpgslct(dev);
    
    if (img != oimg) {
      oimg = img;

      /* Make the histogram of the data */
      xmin = jmin(img->i, img->npts);
      xmax = jmax(img->i, img->npts);
      if (img->npts < 20 * N_BIN) {
	bins = 2.0 * (percentile(img->i, img->npts, 0.75) 
		      - percentile(img->i, img->npts, 0.25)) 
	  * pow((double)img->npts, -1.0/3.0);
	nbin = (int)((xmax - xmin) / bins);
      } else
	nbin = N_BIN + 1;
      if (nbin > N_BIN)
	nbin = N_BIN;
      bins = (xmax - xmin) / (float)nbin;
      for (i = 0; i < nbin; i++) {
	xb[i] = xmin + (float)i * bins;
	h[i] = 0.0;
      }
      xb[nbin] = xmax;
      bin(img->i, img->npts, h, xb, nbin);
      
      for (i = 0; i < nbin; i++)
	h[i] = (h[i] == 0.0) ? 0.0 : log10(h[i]);

      ymax = 1.1 * jmax(h, nbin);
      ymin = 0.0;

      xmin = (*zmax + *zmin) / 2.0 - 1.0 * fabs(*zmax - *zmin);
      xmax = (*zmax + *zmin) / 2.0 + 3.0 * fabs(*zmax - *zmin);

      x = (xmin + xmax) / 2.0;
      y = (ymin + ymax) / 2.0;
    }

    /* Plot the histogram */
    plot(*zmin, *zmax);
    
    while (1) {
    
      cpgcurs(&x, &y, &com);

      dist[0] = fabs(x - xmin);
      dist[1] = fabs(x - xmax);
      dist[2] = fabs(x - *zmin);
      dist[3] = fabs(x - *zmax);
      i = idxval(dist, 4, jmin(dist, 4));
      
      if (com == 'q') {
	r = 0;
	break;

      } else if (com == '?') {
	fprintf(stderr, "\nList of commands in scaling mode:\n\n");
	fprintf(stderr, "\t?\t\tprint this list\n \
\t1\t\tminmax\n \
\t2\t\tminmax of central %d x %d pixel region\n \
\t3\t\tsigma clipping\n \
\tlmb\t\tdrag closest limit or y-axis\n \
\trmb\t\tdrag limits together\n \
\ti\t\tlinear scaling\n \
\tl\t\tlogarithmic scaling\n \
\ts\t\tsquare-root scaling\n \
\ta\t\tset x-axis limits to histogram extremes\n \
\tq\t\tquit scaling\n", N_CENT, N_CENT);

      } else if (com == 'a') {
	xmin = xb[0];
	xmax = xb[nbin-1];
	plot(*zmin, *zmax);

      } else if (com == 'i') {
	cpgslct(odev);
	cpgsitf(0);
	sprintf(scstring, "Scaling: linear");
	r = 4;
	break;

      } else if (com == 'l') {
	cpgslct(odev);
	cpgsitf(1);
	sprintf(scstring, "Scaling: log");
	r = 4;
	break;

      } else if (com == 's') {
	cpgslct(odev);
	cpgsitf(2);
	sprintf(scstring, "Scaling: square-root");
	r = 4;
	break;

      } else if (isdigit(com)) {
	r = com - '0';
	if (r > 0 && r < 4)
	  break;

      } else if (com == 'X') {
	cpgsci(7);
	x = (*zmax + *zmin) / 2.0;
	cpgband(6, 1, x, y, &x, &y, &com);
	cpgsci(1);
	dist[0] = (*zmax - *zmin) / 2.0;
	*zmax = x + dist[0];
	*zmin = x - dist[0];
	r = 4;
	break;

      } else if (i == 0) {
	cpgsci(0);
	cpgbox("", 0.0, 0, "BST", 0.0, 0);
	cpgsci(5);
	cpgband(6, 1, xmin, y, &xmin, &y, &com);
	cpgsci(1);
	plot(*zmin, *zmax);

      } else if (i == 1) {
	cpgsci(0);
	cpgbox("", 0.0, 0, "CST", 0.0, 0);
	cpgsci(5);
	cpgband(6, 1, xmax, y, &xmax, &y, &com);
	cpgsci(1);
	plot(*zmin, *zmax);
	
      } else if (i == 2) {
	cpgsci(0);
	cpgmove(*zmin, ymin);
	cpgdraw(*zmin, ymax);
	cpgsci(7);
	cpgband(6, 1, *zmin, y, zmin, &y, &com);
	cpgsci(1);
	plot(*zmin, *zmax);
	x = *zmin;
	r = 4;
	break;
	
      } else if (i == 3) {
	cpgsci(0);
	cpgmove(*zmax, ymin);
	cpgdraw(*zmax, ymax);
	cpgsci(7);
	cpgband(6, 1, *zmax, y, zmax, &y, &com);
	cpgsci(1);
	plot(*zmin, *zmax);
	x = *zmax;
	r = 4;
	break;
      }
    }

    cpgslct(odev);

  } else {
    errormsg("iscale(): Unknown option: %d", opt);
    /* To stop compiler from complaining */
    r = 4;
  }

  return r;
}
