/*****************************************************************************
* Nonlinear fitting
*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <gsl/gsl_blas.h>
#include <gsl/gsl_vector.h>
#include <gsl/gsl_matrix.h>
#include <gsl/gsl_multifit_nlin.h>
#include "nlfit.h"

gsl_vector   *dummy1;
gsl_matrix   *dummy2;
int          (*fitfunc)(const gsl_vector *, void *, gsl_vector *,
			gsl_matrix *);


/******************************************************************************
* Fitting function 
******************************************************************************/

int fitfunc_f(const gsl_vector *a, void *data, gsl_vector *f) {
  
  int    fepfmc1d(const gsl_vector *, void *, gsl_vector *, gsl_matrix *);
  
  
  return fepfmc1d(a, data, f, dummy2);
}


int fitfunc_df(const gsl_vector *a, void *data, gsl_matrix *J) {

  int    fepfmc1d(const gsl_vector *, void *, gsl_vector *, gsl_matrix *);

  
  return fepfmc1d(a, data, dummy1, J);
}


/******************************************************************************
* Print information on the current iteration
******************************************************************************/

void print_state(size_t iter, int status, gsl_multifit_fdfsolver *s) {

  size_t    i;


  fprintf(stderr, "iteration: %3u   %s   chi^2 = %e\n", iter, 
	  gsl_strerror(status), pow(gsl_blas_dnrm2(s->f), 2.0));
  for (i = 0; i < (s->x)->size; i++)
    fprintf(stderr, "a[%u] = %e\n", i, gsl_vector_get(s->x, i));

  return;
}

/*****************************************************************************/

void nlfit(double x[], double y[], double sig[], int onpts, double a[],
	   double erra[], int ona, double *chisq,
	   int (*ofitfunc)(const gsl_vector *, void *, gsl_vector *,
			   gsl_matrix *)) {

  int                                 status;
  size_t                              i, npts, na, iter;
  double                              *dummy3;
  fit_data                            data;
  const gsl_multifit_fdfsolver_type   *T;
  gsl_multifit_fdfsolver              *s;
  gsl_multifit_function_fdf           f;
  gsl_vector_view                     va;
  gsl_matrix                          *covar;
  void                                errormsg(char *, ...),
                                      nferrormsg(char *, ...);


  if (onpts <= 0)
    errormsg("nlfit(): Bad number of data points: %d!", onpts);
  if (ona <= 0)
    errormsg("nlfit(): Bad number of parameters: %d!", ona);
  if (onpts < ona)
    errormsg("nlfit(): Not enough data points for this model!");

  /* Initial setup */
  npts = (size_t)onpts;
  na = (size_t)ona;
  fitfunc = ofitfunc;
  covar = gsl_matrix_alloc(na, na);
  dummy1 = gsl_vector_alloc(npts);
  dummy2 = gsl_matrix_alloc(npts, na);
  if ((dummy3 = (double *)calloc(npts, sizeof(double))) == NULL)
    errormsg("nlfit(): Couldn't allocate memory!");
  
  /* Data */
  data.n = npts;
  data.x = x;
  data.y = y;
  data.sig = sig;
  data.yf = dummy3;

  /* Fitting function */
  f.f = &fitfunc_f;
  f.df = &fitfunc_df;
  f.fdf = fitfunc;
  f.n = npts;
  f.p = na;
  f.params = &data;

  /* Initial guess */
  va = gsl_vector_view_array(a, na);

  /* Set up the solver */
  T = gsl_multifit_fdfsolver_lmsder;
  s = gsl_multifit_fdfsolver_alloc(T, npts, na);
  gsl_multifit_fdfsolver_set(s, &f, &va.vector);

  /* Iterate */
  iter = 0;
  status = 0;
  print_state(iter, status, s);
  do {
    iter++;
    status = gsl_multifit_fdfsolver_iterate(s);
    print_state(iter, status, s);
    if (status)
      /* Iterator can't make any progress; not sure what this means */
      break;
    status = gsl_multifit_test_delta(s->dx, s->x, GSL_DBL_EPSILON, 1.0e-5);
  } while (status == GSL_CONTINUE && iter < N_MAX);

  if (iter == N_MAX)
    nferrormsg("nlfit(): Hit maximum number of iterations!");

  /* Parameters */
  for (i = 0; i < na; i++)
    a[i] = gsl_vector_get(s->x, i);

  /* Errors */
  gsl_multifit_covar(s->J, 0.0, covar);
  for (i = 0; i < na; i++)
    erra[i] = sqrt(gsl_matrix_get(covar, i, i));
  
  /* chi^2 */
  *chisq = pow(gsl_blas_dnrm2(s->f), 2.0);

  /* Clean up */
  gsl_multifit_fdfsolver_free(s);
  gsl_matrix_free(covar);
  gsl_vector_free(dummy1);
  gsl_matrix_free(dummy2);
  free(dummy3);

  return;
}
