/******************************************************************************
* Read the image header of a 2d fits image;
* if the image is gzipped but this routine is called without .gz in the image
* name (which will work because of cfitsio's file name syntax) then it will 
* not use a .hdr file even if it is present;
* opt only gets used with the actual image, not with the .hdr file:
* if I don't know the image name (opt=1 or 2) it's unlikely a header file 
* exists anyway; if a mistake was made in the image name (opt=3) then the 
* routine will complain about not finding the image rather than not finding 
* the header
******************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <fitsio.h>
#include <wcs.h>
#include "img.h"


void rfitsimghdr(char *imgname, int opt, image *img) {

  int        i, status, dim, zip, fits;
  long       naxes[2];
  double     det;
  char       *header, *buf, buffer[L_WORD], imgn[L_WORD], headn[L_WORD];
  FILE       *data_file;
  fitsfile   *fptr;
  FILE       *faskropen(char *, char *, int);
  fitsfile   *fitsaskropen(char *, char *, int);
  void       errormsg(char *, ...), ferrormsg(int, char *, ...),
             warnmsg(char *, ...), zpx2zpn(char *);


  zip = fits = 0;

  /* Strip any .gz or .Z off the image name */
  strcpy(imgn, imgname);
  if ((buf = strstr(imgn, ".gz")) != NULL) {
    *buf = '\0';
    zip = 1;
  } else if ((buf = strstr(imgn, ".Z")) != NULL) {
    *buf = '\0';
    zip = 1;
  }
  
  /* Make the name of the header file */
  strcpy(headn, imgn);
  if ((buf = strstr(headn, ".fit")) != NULL) {
    *buf = '\0';
    fits = 1;
  }
  strcat(headn, ".hdr");

  if ((data_file = faskropen("Header file?", headn, 4)) != NULL && 
      (!fits || (fits && zip))) {
    /* Use header file */
    
    /* Allocate memory for the header (180 * 80 + 1 = 14401) */
    if ((header = (char *)calloc(14401, sizeof(char))) == NULL)
      errormsg("rfitsimghdr(): Couldn't allocate memory!");

    buf = header;
    i = 0;
    while (i < 180 && fgets(buffer, L_WORD, data_file) != NULL) {
      i++;

      strncpy(buf, buffer, 80);

      /* Pad with white spaces */
      while (*buf != '\0' && *buf != '\n' && buf != header + i*80)
	buf++;
      while (buf != header + i*80) {
	*buf = ' ';
	buf++;
      }
    }

    /* Close the header file */
    fclose(data_file);

    /* Pad the rest with white spaces */
    while (buf != header + 14400) {
      *buf = ' ';
      buf++;
    }
    header[14400] = '\0';

    /* Check the dimension of the image */
    for (i = 0; i < 180; i++)
      if (!strncmp(header + i*80, "NAXIS   ", 8)) {
	if ((buf = strchr(header + i*80, '=')) == NULL || 
	    buf != header + i*80 + 8)
	  errormsg("rfitsimghdr(): NAXIS card is malformed!");
	if (sscanf(buf+1, "%d", &dim) != 1)
	  errormsg("rfitsimghdr(): Couldn't read NAXIS keyword from %s!",
		   headn);
	if (dim != 2)
	  errormsg("rfitsimghdr(): %s doesn't have 2 dimensions!", headn);
	break;
      }
    if (i == 180)
      errormsg("rfitsimghdr(): Couldn't find NAXIS keyword in %s!", headn);

    /* Get the size of the image */
    for (i = 0; i < 180; i++)
      if (!strncmp(header + i*80, "NAXIS1  ", 8)) {
	if ((buf = strchr(header + i*80, '=')) == NULL || 
	    buf != header + i*80 + 8)
	  errormsg("rfitsimghdr(): NAXIS1 card is malformed!");
	if (sscanf(buf+1, "%ld", &naxes[0]) != 1)
	  errormsg("rfitsimghdr(): Couldn't read NAXIS1 keyword from %s!",
		   headn);
	break;
      }
    if (i == 180)
      errormsg("rfitsimghdr(): Couldn't find NAXIS1 keyword in %s!", headn);
    for (i = 0; i < 180; i++)
      if (!strncmp(header + i*80, "NAXIS2  ", 8)) {
	if ((buf = strchr(header + i*80, '=')) == NULL || 
	    buf != header + i*80 + 8)
	  errormsg("rfitsimghdr(): NAXIS2 card is malformed!");
	if (sscanf(buf+1, "%ld", &naxes[1]) != 1)
	  errormsg("rfitsimghdr(): Couldn't read NAXIS2 keyword from %s!",
		   headn);
	break;
      }
    if (i == 180)
      errormsg("rfitsimghdr(): Couldn't find NAXIS2 keyword in %s!", headn);

    /* Get the LTV keywords */
    img->ltv[0] = img->ltv[1] = 0.0;
    for (i = 0; i < 180; i++)
      if (!strncmp(header + i*80, "LTV1    ", 8)) {
	if ((buf = strchr(header + i*80, '=')) == NULL || 
	    buf != header + i*80 + 8)
	  errormsg("rfitsimghdr(): LTV1 card is malformed!");
	if (sscanf(buf+1, "%lf", &(img->ltv[0])) != 1)
	  errormsg("rfitsimghdr(): Couldn't read LTV1 keyword from %s!",
		   headn);
	break;
      }
    for (i = 0; i < 180; i++)
      if (!strncmp(header + i*80, "LTV2    ", 8)) {
	if ((buf = strchr(header + i*80, '=')) == NULL || 
	    buf != header + i*80 + 8)
	  errormsg("rfitsimghdr(): LTV2 card is malformed!");
	if (sscanf(buf+1, "%lf", &(img->ltv[1])) != 1)
	  errormsg("rfitsimghdr(): Couldn't read LTV2 keyword from %s!",
		   headn);
	break;
      } 

    /* Get the LTM keywords */
    img->ltm[0][0] = img->ltm[1][1] = 1.0;
    img->ltm[1][0] = img->ltm[0][1] = 0.0;
    for (i = 0; i < 180; i++)
      if (!strncmp(header + i*80, "LTM1_1  ", 8)) {
	if ((buf = strchr(header + i*80, '=')) == NULL || 
	    buf != header + i*80 + 8)
	  errormsg("rfitsimghdr(): LTM1_1 card is malformed!");
	if (sscanf(buf+1, "%lf", &(img->ltm[0][0])) != 1)
	  errormsg("rfitsimghdr(): Couldn't read LTM1_1 keyword from %s!",
		   headn);
	break;
      }
    for (i = 0; i < 180; i++)
      if (!strncmp(header + i*80, "LTM1_2  ", 8)) {
	if ((buf = strchr(header + i*80, '=')) == NULL || 
	    buf != header + i*80 + 8)
	  errormsg("rfitsimghdr(): LTM1_2 card is malformed!");
	if (sscanf(buf+1, "%lf", &(img->ltm[0][1])) != 1)
	  errormsg("rfitsimghdr(): Couldn't read LTM1_2 keyword from %s!",
		   headn);
	break;
      }
    for (i = 0; i < 180; i++)
      if (!strncmp(header + i*80, "LTM2_1  ", 8)) {
	if ((buf = strchr(header + i*80, '=')) == NULL || 
	    buf != header + i*80 + 8)
	  errormsg("rfitsimghdr(): LTM2_1 card is malformed!");
	if (sscanf(buf+1, "%lf", &(img->ltm[1][0])) != 1)
	  errormsg("rfitsimghdr(): Couldn't read LTM2_1 keyword from %s!",
		   headn);
	break;
      }
    for (i = 0; i < 180; i++)
      if (!strncmp(header + i*80, "LTM2_2  ", 8)) {
	if ((buf = strchr(header + i*80, '=')) == NULL || 
	    buf != header + i*80 + 8)
	  errormsg("rfitsimghdr(): LTM2_2 card is malformed!");
	if (sscanf(buf+1, "%lf", &(img->ltm[1][1])) != 1)
	  errormsg("rfitsimghdr(): Couldn't read LTM2_2 keyword from %s!",
		   headn);
	break;
      }

  } else if (fits) {
    /* Use the fits image */

    status = 0;

    /* Open the fits file */
    fptr = fitsaskropen("Fits file to read?", imgname, opt);

    /* Check the dimension of the image */
    if (fits_get_img_dim(fptr, &dim, &status))
      ferrormsg(status, "rfitsimghdr(): Couldn't read NAXIS keyword from %s!",
		imgn);
    if (dim != 2)
      errormsg("rfitsimghdr(): %s doesn't have 2 dimensions!", imgn);

    /* Get the size of the image */
    if (fits_get_img_size(fptr, 2, naxes, &status))
      ferrormsg(status, "rfitsimghdr(): Couldn't determine size of %s!", imgn);

    /* Get the LTV keywords */
    img->ltv[0] = img->ltv[1] = 0.0;
    if (fits_read_keys_dbl(fptr, "LTV", 1, dim, img->ltv, &i, &status))
      status = 0;

    /* Get the LTM keywords */
    img->ltm[0][0] = img->ltm[1][1] = 1.0;
    img->ltm[1][0] = img->ltm[0][1] = 0.0;
    if (fits_read_keys_dbl(fptr, "LTM1_", 1, dim, img->ltm[0], &i, &status))
      status = 0;
    if (fits_read_keys_dbl(fptr, "LTM2_", 1, dim, img->ltm[1], &i, &status))
      status = 0;

    /* Get the coordinate system keywords */
    if (fits_get_image_wcs_keys(fptr, &header, &status))
      ferrormsg(status, "rfitsimghdr(): Couldn't read coordinate system\
 keywords from %s!", imgn);

    /* Close the fits file */
    if (fits_close_file(fptr, &status))
      ferrormsg(status, "rfitsimghdr(): Couldn't close %s!", imgn); 

  } else
    errormsg("rfitsimghdr(): Found no header for %s!", imgn);

  img->nx = naxes[0];
  img->ny = naxes[1];
  img->npts = naxes[0] * naxes[1];

  /* Inverse of LTM */
  det = img->ltm[1][0];
  img->ltm[1][0] = img->ltm[0][1];
  img->ltm[0][1] = det;
  det = img->ltm[0][0] * img->ltm[1][1] - img->ltm[0][1] * img->ltm[1][0];
  if (det != 0.0) {
    img->ltmi[0][0] = img->ltm[1][1] / det;
    img->ltmi[0][1] = -img->ltm[0][1] / det;
    img->ltmi[1][0] = -img->ltm[1][0] / det;
    img->ltmi[1][1] = img->ltm[0][0] / det;
  } else
    warnmsg("rfitsimghdr(): LTM has no inverse!");

  /* Fix -ZPX projection */
  zpx2zpn(header);

  /* Fill in the WCS structure */
  img->wcs = wcsinit(header);

  /* Clean up */
  free(header);

  return;
}
