package org.eso.phase3.validator;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Interface on any client checking for archived files. A client can be local 
 * (i.e. connecting directly with the database), or connecting to a remote server. 
 * A client could also perform only a dummy check (i.e. returning always true for 
 * any input).
 * @author dsforna
 * 
 */
public interface ArchivedFilesClient
{
    /**
     * These are the allowed status of a file id returned (as string) by the
     * HTTP archived file server.
     * */
    public static final String ARCHIVED = "ARCHIVED";
    public static final String DATABASE_ERROR = "DATABASE_ERROR";
    public static final String NOT_ARCHIVED = "NOT_ARCHIVED";
    public static final List<String> ALLOWED_RETURN_VALUES = Arrays.asList(
            ARCHIVED, NOT_ARCHIVED, DATABASE_ERROR);

    /**
     * These constant must either match what is defined in the server part, or
     * being directly imported from the interface of the server part.
     * FILE_ID_PARAMETER_NAME: fileid parameter name in the input form.
     * FILE_ID_STATUS_SEPARATOR: fileid and its status are separated by this
     * string in the response
     */
    public static final String FILE_ID_PARAMETER_NAME = "fileid";;

    public static final String FILE_ID_STATUS_SEPARATOR = "|";


    /** Any unknown return value will be set to this: */
    public static final String UNKNOWN_RETURNED_VALUE_ERROR = "UNKNOWN_RETURNED_VALUE_ERROR";
    
    /**
     * Return the subset of the input set containing only the IDs which are
     * archived remotely.
     * 
     * @param fileIds
     *            the input set of files IDs to check.
     * @return the archived set.
     * @throws ValidatorException
     *             in case of HTTP connection error.
     */
    public Set<String> getArchived(Set<String> fileIds)
            throws ValidatorException;

    public boolean hasError();

    /**
     * @return if this client can pass request to the server.
     */
    public boolean isAvailable();

    /**
     * Check for each input file if it is archived or not.
     * 
     * @param fileIds
     *            set of files to check.
     * @return Map (key: file in input, value: status returned by the archived
     *         service). If the returned status is not declared in this
     *         interface, it is overwritten with UNKNOWN_RETURNED_VALUE_ERROR
     * @throws ValidatorException
     *             in case of HTTP connection error.
     */
    public Map<String, String> status(Set<String> fileIds)
            throws ValidatorException;;
}
