package org.eso.phase3.validator;

import java.util.*;
import java.util.Map.Entry;

import org.apache.log4j.Logger;

/**
 * Utility class to check for circular definitions in a set of products. 
 * Used to validate the provenance information of a release.
 *  
 * @author dsforna
 */
public class DependencyCheck
{
    /** Apache Log4J logger for this class namespace. */
    private static final Logger logger = Logger.getLogger(DependencyCheck.class);

    /**Remove the input components from the input potentialLoopMap. If any key 
     * (i.e. product) of the map is then left with an empty value, remove it 
     * from the map because it means it cannot be part of a circular dependency.
     * @param possibleLoops
     * @param components
     * @return if at least a key of the map was removed.
     */
    private boolean removeElements(Map<String, Set<String>> possibleLoops, 
            Set<String> components)
    {
        boolean productRemoved = false;
        
        Iterator<Entry<String, Set<String>>> it = possibleLoops.entrySet().iterator();
        while (it.hasNext()) {
        	Entry<String, Set<String>> entry = it.next();
        	Set<String> value = entry.getValue();
        	if (value != null) {
        		value.removeAll(components);
        		if (value.isEmpty()) {
                    logger.debug("Removing from possible loops the product "+ entry.getKey() 
                            + " because it does not have any dependency left.");
                    productRemoved = true;
                    it.remove();
                }
        	}
        }
        return productRemoved;
    }
    
    /**
     * Check if the provenance definitions in input have circular dependencies.
     * The algorithm considers all the definition potentially circular and then 
     * progressively removes any product or component which does not have 
     * dependencies. If at the end all the products are removed there is no 
     * circular dependency, otherwise there is at least one (note that there 
     * might be more than one) circular dependency among the remaining products. 
     * @param provenanceMap the input provenance definition with entries: 
     * (product, set of components)
     * @return the set of products with at least a circular dependency. If there 
     * is no circular dependency this set is empty.
     */
    public Set<String> findLoops(Map<String, Set<String>> provenanceMap)
    {
        logger.debug("Looking for circular definition.");
        Map<String, Set<String>> loopMap = new HashMap<String, Set<String>>();
        Set<String> components = new HashSet<String>();
        
        for (Entry<String, Set<String>> entry: provenanceMap.entrySet()) {
        	Set<String> loopMapComponents =  new HashSet<String>();
            // (fix for DFS09571):
            Set<String> prodComponents = entry.getValue();
            {
            	loopMapComponents.addAll(prodComponents);
            	components.addAll(prodComponents);
            }
            loopMap.put(entry.getKey(), loopMapComponents);
        }
        
        boolean removed = true;
        Set<String> products = loopMap.keySet();
        while (removed)
        {
            removed = false;
            Set<String> toRemove = new HashSet<String>();
            Iterator<String> itComponents = components.iterator();
            while (itComponents.hasNext())
            {
                String s = itComponents.next();
                if (! products.contains(s)) 
                {
                    logger.debug(s 
                            + " does not depend on any components therefore does not have a circular definition.");
                    removed = true; 
                    itComponents.remove();
                    toRemove.add(s);
                }
            }
            if (removeElements(loopMap, toRemove))
            {
                removed = true; 
            }
        }

        if (loopMap.keySet().size()> 0)
        {
            logger.error("Found at least one circular dependency.");
        }
        else 
        {
            logger.debug("No circular dependency found.");
        }
        return loopMap.keySet();
    }
    
    /**
     * This main is used as test program. 
     */
    public static void main(String[] args)
    {
        ValidationUtil.configureLog4j();
        logger.trace("");
        DependencyCheck check = new DependencyCheck();
        Map<String, Set<String>> provenanceMap;
        Set<String> loops;
        
        Map<String, Set<String>> okMap = new HashMap<String, Set<String>>();
        okMap.put("s", new HashSet<String>(Arrays.asList("a1", "a2","b1", "b2", "c", "e1", "e2")));
        okMap.put("b2", new HashSet<String>(Arrays.asList("a2","b1", "e2")));
        okMap.put("a2", new HashSet<String>(Arrays.asList("a1")));
        okMap.put("c", new HashSet<String>(Arrays.asList("a1", "e2")));
        okMap.put("e2", new HashSet<String>(Arrays.asList("e1")));
        okMap.put("b1", new HashSet<String>(Arrays.asList("a1", "a2","e1")));
        
        provenanceMap=okMap;
        loops = check.findLoops(provenanceMap);
        System.out.println("Loops in OkMap (should be empty): " + Arrays.toString(loops.toArray()));

        Map<String, Set<String>> nokMap = new HashMap<String, Set<String>>();
        nokMap.put("s", new HashSet<String>(Arrays.asList("a1", "a2","b1", "b2", "c", "e1", "e2")));
        nokMap.put("b2", new HashSet<String>(Arrays.asList("a2","b1", "e2")));
        nokMap.put("a2", new HashSet<String>(Arrays.asList("a1")));
        nokMap.put("c", new HashSet<String>(Arrays.asList("a1", "a2", "e2")));
        nokMap.put("e2", new HashSet<String>(Arrays.asList("e1")));
        nokMap.put("b1", new HashSet<String>(Arrays.asList("a1", "a2","e1")));
        nokMap.put("e1", new HashSet<String>(Arrays.asList("s")));

        provenanceMap=nokMap;
        loops = check.findLoops(provenanceMap);
        System.out.println("Loops in NOkMap (should be 1 loop): " + Arrays.toString(loops.toArray()));
    }
}
