package org.eso.phase3.validator;

import java.io.File;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.eso.oca.fits.DataTransportFormatHandler;
import org.eso.phase3.validator.ValidationReport.STATUS;
import org.eso.phase3.validator.ValidatorStat.StatType;

/**
 * Validator for a single FITS file of the release. This validator first uses
 * the external fits verify tool to assess the general compliance with the FITS
 * standard, then it invokes the OCA parser to check if the meta-data of the
 * FITS file are compliant with the ESO requirements.
 * 
 * @author dsforna
 */
public class FitsValidator implements ValidationStep
{
    /** Apache Log4J logger for this class namespace. */
    private static final Logger logger = Logger.getLogger(FitsValidator.class);

    /**
     * @param fitsValidator
     * @return
     */
    public static boolean isValid(FitsValidator fitsValidator)
    {
        return fitsValidator.isValid();
    }


    /**Category of the fits file, as declared in the release's structure.*/
    private final String fileCategory;

    /**Path and name on disk of the fits file to validate.*/
    private final String fileFullPathName;

    /**Name of the fits file to validate.*/
    private String fileName;

    /**The status and messages of this validation will be stored here.*/
    private final ValidationReport fitsReport;

    /**
     * If this attribute is set to true, the @{link #isValid()} method will
     * return always false. It is used to allow validation of files which are
     * already in error for some other reason.
     */
    private final boolean forceInvalid;

    /**
     * Meta keyword of the OCAFile where the log of OCA validation is stored.
     */
    private final String logMetaKeyword = Consts.LOGMSG_METAKEYWORD; 
    
    /**
     * Meta keyword of the OCAFile where the result of validation is stored.
     * It assumes the values "true" or "false".
     */
    private final String resultMetaKeyword = Consts.RESULT_METAKEYWORD; 
    
    /**The context of this validator.*/
    private final ValidatorSetup setup;

    /**
     * Has this validation step been successfully and fully executed (fits 
     * verify, OCA rules, additional checks)?
     */
    private boolean validationResult;
    
    private boolean internal;

    @Override public String toString()
    {
        return "VS of fits " + fileName;
    }
    
    /**Create a FitsValidator for the input file.
     * @param setup the context of this application.
     * @param fileFullPathName the pathname of the file to validate.
     * @param fileCategory the category of the file to validate.
     * @param forceInvalid if this FitsValidator is forced to return a 
     * validation error as result.
     */
    public FitsValidator(final ValidatorSetup setup,
            final String fileFullPathName, final String fileCategory,
            final boolean forceInvalid, final boolean internal)
    {
        if (fileFullPathName == null)
        {
            logger.error("Null input argument: fileFullPathName");
            throw new IllegalArgumentException(
                    "Null input argument: fileFullPathName");
        }

        this.forceInvalid = forceInvalid;
        this.fitsReport = new ValidationReport("FitsValidator on fits "
                + fileFullPathName + " [catg:" + fileCategory + "]", null);
        this.fileFullPathName = fileFullPathName;
        this.fileCategory = fileCategory;
        this.validationResult = false;
        this.setup = setup;
        this.internal = internal;
        fileName = null;
        try
        {
            // Avoid creating an extra File object just to get the name:
            fileName = fileFullPathName.substring(1 + 
                    fileFullPathName.lastIndexOf(File.separator));
        }
        catch( final java.lang.IndexOutOfBoundsException e )
        {
            fileName = fileFullPathName;
        }
    }

    
    /* (non-Javadoc)
     * @see org.eso.phase3.validator.ValidationStep#isValid()
     */
    @Override
    public boolean isValid()
    {
        final String methodLogName = "FitsValidator::isValid";
        logger.trace(methodLogName);
        if (forceInvalid)
        {
            logger.debug("validation forced to error for " + fileName);
            return false;
        }
        else
        {
            logger.debug("Validation success for " + fileName + ": " + validationResult);
            return validationResult;
        }
    }

    
    /* (non-Javadoc)
     * @see org.eso.phase3.validator.ValidationStep#runValidation()
     */
    @Override
    public void runValidation()
    {
        logger.trace("");
        logger.debug("Validating file: "+ fileFullPathName);
        validationResult = true; // will be set to false when anything goes wrong.
        try
        {
            runFitsVerify(setup.getValidatorStat());
        }
        catch( final ValidatorException e )
        {
            setup.getValidatorStat().add(StatType.ERROR_FITS_VALIDATION);
            validationResult = false;
            logger.error(e.toString());
            fitsReport.attemptStatus(STATUS.ERROR, e.toString());
            return;
        }

        final ValidationReport metaDataReport = new ValidationReport(
                "Metadata validation of fits file " + fileFullPathName,
                null);
        fitsReport.addsubreport(metaDataReport);
        final boolean pre = preOcaValidation(metaDataReport);

        
        // Next steps make sense only if the categoryMap is set:
        if (fileCategory == null)
        {
            validationResult = false;
            String msg = fileName + ": category is not set. Cannot validate file.";
            fitsReport.attemptStatus(STATUS.ERROR, msg);
            logger.error(msg);
            return;
        }
        else if (!setup.getHttpConf().isValid(fileCategory))
        {
            validationResult = false;
            String origin = setup.getReleaseParser().getCategoryLocationMap().get(fileName);
            if (origin == null) {origin="(original file no longer available)";}
            String msg = origin + ": invalid category " + fileCategory 
                    + " defined for file " + fileName;
            fitsReport.attemptStatus(STATUS.ERROR, msg);
            logger.error(msg);
            return;
        }
        else
        {
            logger.debug(fileCategory + " is a valid category");
        }

        boolean oca = ocaValidation(metaDataReport);
        if (pre && oca)
        {
            String msg = fileName + " is valid.";
            fitsReport.addInfo(msg);
            logger.debug(msg);
        }
        else
        {
            // If any extension is wrong, the whole fits is wrong:
            validationResult = false;
        }

        if (validationResult)
        {
            fitsReport.attemptStatus(STATUS.VALID);
            logger.info("Validation correctly performed for: "
                    + fileFullPathName);
        }
        else
        {
            fitsReport.attemptStatus(STATUS.ERROR);
            logger.warn("Validation returned with errors for: " + fileFullPathName);
        }
        return;
    }

    
    /* (non-Javadoc)
     * @see org.eso.phase3.validator.ValidationStep#validationReport()
     */
    @Override
    public synchronized ValidationReport validationReport()
    {
        final String methodLogName = "FitsValidator::validationReport";
        logger.trace(methodLogName);
        return fitsReport;
    }
    
    /**
     * Perform the meta data validation according to the OCA rules for the fits 
     * file of this FitsValidator.
     * 
     * @param idxMainHeader
     *            the extension (0..HDUNumber-1) of the fits file being validated.
     * @parameter extCategory the category of this extension.
     * @param ocaReport
     *            the {@link #ValidationReport} object where to write the
     *            validation result.
     *            
     * @return true if validation is OK and false otherwise.
     */
    private boolean ocaValidation(final ValidationReport ocaReport)
    {
        String selectRulesFile = setup.getHttpConf().getSelectionRulesFileName(fileCategory);
        if (selectRulesFile == null)
        {
            String msg = fileName + " No metadata rules validation needed for category "
            + fileCategory;
            ocaReport.attemptStatus(STATUS.VALID, msg);
            logger.debug(msg);
            return true;
        }
        
        boolean ocaResult = true;
        final int idxMainHeader = setup.getReleaseParser().getIndexParsedHeader(fileName);
        int numberOfHdus = 0;
        
        DataTransportFormatHandler fh = null;
        try {
        	fh = ValidationUtil.allocateDTFH(fileFullPathName);
        	numberOfHdus = fh.getNumberOfHDUs();
        } catch (final Exception e ) {
            String msg = e.toString();
            logger.error("Error while classifing "+ fileName + ": " + msg);
            ocaReport.attemptStatus(STATUS.ERROR, fileName + ": "
                    + e.getMessage());
            setup.getValidatorStat().add(StatType.ERROR_METADATA);
            return false;
        }
        
        for (int currentIndex = idxMainHeader; currentIndex < numberOfHdus; currentIndex++)
        {
            ValidatorOcaParser vop;
            try
            {
                vop = setup.createValidatorOcaParser(fileFullPathName, fileCategory, fh, currentIndex, currentIndex-idxMainHeader);
            }
            catch( final Exception e )
            {
                logger.error(fileName + ": " + e.toString());
                ocaReport.attemptStatus(STATUS.ERROR, 
                        fileName + ": " + e.getMessage());
                setup.getValidatorStat().add(StatType.ERROR_METADATA);
                return false;
            }
    
            if (!vop.isValid())
            {
            	logger.error(fileName + ": has invalid OCA rules");
                ocaReport.attemptStatus(STATUS.ERROR, 
                        fileName + ": " + vop.getErrorMsg());
                setup.getValidatorStat().add(StatType.ERROR_METADATA);
                ocaResult = false;
                continue;
            }
            
            if (!vop.hasRulesFile())
            {
                logger.debug("no OCA rules file for " +fileFullPathName + ", idx="+ currentIndex);
                continue;
            }
    
            String msg = "";
            Map<String, String> classification = null;
            try
            {
                classification = vop.getClassification(internal);
                msg = classification.get(logMetaKeyword);
    
                msg = org.apache.commons.lang.StringEscapeUtils.unescapeJava(msg);
    
                final String isValid = classification.get(resultMetaKeyword);
                logger.debug("From OCA parser: " + resultMetaKeyword + "="
                        + isValid);
                ocaResult = isValid.equals("true");
                if (isValid.equals("true"))
                {
                    ocaReport.attemptStatus(STATUS.VALID, msg);
                }
                else
                {
                    logger.error("Metadata validation failed"
                            + Arrays.toString(msg.split("\n")));
                    ocaReport.attemptStatus(STATUS.ERROR, fileName
                            + ": Metadata validation errors using rules for category " 
                            + fileCategory + " (" + vop.getRulesFilename() + "):" + msg);
                    setup.getValidatorStat().add(StatType.ERROR_METADATA);
                    ocaResult = false;
                }
            }
            catch( final Exception e )
            {
                msg = e.toString();
                logger.error("Error while classifing "+ fileName + ": " + msg);
                ocaReport.attemptStatus(STATUS.ERROR, fileName + ": "
                        + e.getMessage());
                setup.getValidatorStat().add(StatType.ERROR_METADATA);
            } finally {
            	if (fh != null) 
            		fh.dispose();
            }
    
        }
        return ocaResult;
    }

    
    /**
     * Perform the validation steps before invoking the OCA rules.
     * Currently these steps are:<ul>
     * <li> Check of md5sum if the optional md5sum keyword is present for this file.</li>
     * <li> Existence of the mandatory CHECKSUM keyword for this file.</ul>
     * Note that the value of the CHECKSUM is not checked here (this is a task 
     * for the fits verify utility).
     * @param r
     *      the {@link #ValidationReport} object where to write the result.
     * @return true if validation succeeded, false otherwise.
     */
    private boolean preOcaValidation(final ValidationReport r)
    {
        logger.trace("");
        boolean resultOk;
        // DFS09899 - precondition are not restricted to the CHECKSUM keyword. 
        List<FailedPrecondition> failedPreconditions = 
                setup.getReleaseParser().failedPreconditions(fileName);

        if (failedPreconditions.isEmpty())
        {
            resultOk = true;
            String msg = fileName + ": OK - no failed precondition found.";
            logger.debug(msg);
            r.attemptStatus(STATUS.VALID, msg);
        }
        else 
        {
            resultOk = false;
            for (FailedPrecondition f : failedPreconditions)
            { 
                String msg = fileName + ": " + f.toString(); 
                boolean ret = f.updateStat(setup.getValidatorStat());
                if (!ret) 
                {
                    msg += "(statistics were not updated for this failure) ."; 
                }
                else 
                {
                    msg += " ."; 
                }
                logger.error(msg);
                r.attemptStatus(STATUS.ERROR, msg);
            }
        }
        
        String expected = setup.getReleaseParser().getMd5Sum(fileName); 
        if (expected == null)
        {
            // Currently the md5sum is not mandatory - nothing else to do.
            return resultOk;
        }
        try 
        {
            String computed = ValidationUtil.md5sum(fileFullPathName);
            if (computed.toLowerCase().equals(expected.toLowerCase()))
            {
                String msg = fileName 
                        + " . Computed md5sum matches the expected md5sum ("
                        + expected + ")";
                logger.debug(msg);
                r.attemptStatus(STATUS.VALID, msg);
            }
            else 
            {
                resultOk = false;
                String msg = fileName + ": expected md5sum=" + expected 
                        + " is different from the computed md5sum="+computed;
                logger.error(msg);
                r.attemptStatus(STATUS.ERROR, msg);
                setup.getValidatorStat().add(StatType.ERROR_MD5);
            }
        }
        catch (Exception e)
        {
            String msg = fileName + " - Exception computing md5sum: ";
            logger.error(msg+ e.toString());
            resultOk = false;
            r.attemptStatus(STATUS.ERROR, msg+e.getMessage());
            setup.getValidatorStat().add(StatType.ERROR_MD5);
        }
        return resultOk;
    }

    
    /**
     * Run the fits verification utility, if its available, on this fits file.
     * If the utility can be run and the run fails, updated the statistics.
     * 
     * @throws ValidatorException
     * @return number of entries added to the stat object.
     */
    private int runFitsVerify(final ValidatorStat stat) throws ValidatorException
    {
        int addedStat = 0;
        // File-wide validation with fits validator external utility:
        FitsVerifyExecutor fve = setup.createFitsVerifyExecutor(fileFullPathName);
        if (fve == null) {
        	fitsReport.attemptStatus(STATUS.VALID, "FITS verification skipped");
        }
        else if (fve.isAvailable())
        {
            final int exitCode = fve.getExitStatus();
            if (exitCode != 0)
            {
                stat.add(StatType.ERROR_FITS_VALIDATION);
                addedStat++;
                validationResult = false;
                String msg = fileName
                        + ": fits verification error. Details in the log file.";
                fitsReport.attemptStatus(STATUS.ERROR, msg);
                msg = "Error from " + fve.getFitsVerifyPathName();
                if (fve.getError() != null)
                {
                    fitsReport.addInfo(msg);
                    fitsReport.addInfo(fve.getError());
                    fitsReport.addInfo(fve.getMessage());
                    logger.error(msg + " (exit status =" + fve.getExitStatus() + ") : "+ fve.getError());
                }
                else 
                {
                    logger.error("Empty error message from received"
                            + fve.getFitsVerifyPathName() + " (exit status =" 
                            + fve.getExitStatus() + ")" );
                }
            }
            else
            {
                fitsReport.attemptStatus(STATUS.VALID, "No errors reported, exit status=0 (" + fve.getShortName() + ")");
            }
        }
        else
        {
            // If fits verify is not found consider the fits file invalid (but 
            // do not add an error message). Still go on with the rest of the validation.
            fitsReport.addInfo(fve.getMessage());
            fitsReport.attemptStatus(STATUS.ERROR);
            validationResult = false;
            logger.debug("No fitsverify: file is considered invalid, but validation goes on.");
        }
        return addedStat;
    }
}
