package org.eso.phase3.validator;
import java.io.File;

import org.apache.log4j.Logger;
import org.eso.util.filesystem.FileSystemOperations;

/**
 * Wrapper around the external utility fits verify.
 * For the default fits verify see: 
 * http://heasarc.gsfc.nasa.gov/docs/software/ftools/fitsverify/
 *
 * @author dsforna
 * 
 */
public class FitsVerifyExecutorImp implements FitsVerifyExecutor
{
    /**
     * Fits verify utility will be searched once per execution.
     */
    private static volatile boolean attemptedLocation = false;

    /** This will change to the real name (configured value), if the
     * executable is found. */
    private static volatile String fitsVerifyName = Consts.FITSVERIFY_DEFAULT_NAME;

    /**
     * This will be set to the pathname of the found executable, it will remain
     * null if no executable can be located. 
     */
    private static volatile String fitsVerifyPathName = null;

    /** Apache Log4J logger for this class namespace. */
    private static final Logger logger = Logger.getLogger(FitsVerifyExecutorImp.class);

    private final ValidatorConfiguration config;

    /**Location of the fits file in input to fits verify utility.*/
    private final String fitsFileFullPathName;

    /**Exit status returned by fits verify.*/
    private int exitStatus;
    
    /**
     * When fitsverify's exit code is not 0 this is standard out and standard err 
     * of fits verify. When fitsverify's exit code is 0 this is empty. 
     */
    private String error = null;
    
    /**
     *  When fits verify is available this is the standard out from its run. 
     *  Otherwise is a message stating that fits verify is not available. 
     */
    private String message;
    
    /**
     * Locate fits verify utility (one-off operation) and then run it. If
     * fitsFileFullPathName is not null, run fits verify on the file, if it has
     * been located. Otherwise just try to locate it. A null input filename will
     * force the attempt to locate fits verify, even if the attempt was already
     * performed in this run.
     * 
     * @param config
     *            the configuration for this run.
     * @param fitsFileFullPathName
     *            the fits file to verify, or null if there is no file to verify
     *            in this invocation.
     * @throws ValidatorException
     */
    public FitsVerifyExecutorImp(final ValidatorConfiguration config,
           final String fitsFileFullPathName) throws ValidatorException
    {
        this.config = config;
        this.fitsFileFullPathName = fitsFileFullPathName;
        // until this fits verify is not executed:
        this.message = null;
        this.error = null;
        this.exitStatus = -1;

        if (fitsFileFullPathName == null)
        {
            locateFitsVerify(); // even if there was a previous attempt.
            return;
        }

        if (!attemptedLocation)
        {
            locateFitsVerify();
        }

        final File f = new File(fitsFileFullPathName);
        if (!f.exists())
        {
            throw new ValidatorException("Fits File " + fitsFileFullPathName
                    + " does not exist.");
        }
        if (!f.canRead())
        {
            throw new ValidatorException("Fits File " + fitsFileFullPathName
                    + " is not readable.");
        }

        if (fitsVerifyPathName != null)
        {
            // If fits verify is available, execute it:
            execute();
        }
        else
        {
            exitStatus = 0;
            error = "";
            message = fitsVerifyName
                    + " not available. Skipped verification on "
                    + fitsFileFullPathName;
            logger.debug(message);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.FitsVerifyExecutor#getError()
     */
    public String getError()
    {
        return error;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.FitsVerifyExecutor#getExitStatus()
     */
    @Override
    public int getExitStatus()
    {
        return exitStatus;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.FitsVerifyExecutor#getFitsVerifyPathName()
     */
    @Override
    public String getFitsVerifyPathName()
    {
        return fitsVerifyPathName;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.FitsVerifyExecutor#getMessage()
     */
    @Override
    public String getMessage()
    {
        return message;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.FitsVerifyExecutor#getShortName()
     */
    @Override
    public String getShortName()
    {
        final String methodName = "FitsVerifyExecutor::getShortName";
        logger.trace(methodName);
        return fitsVerifyName;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.FitsVerifyExecutor#isAvailable()
     */
    @Override
    public boolean isAvailable()
    {
        return (fitsVerifyPathName != null);
    }

    /**
     * Execute fits verify on this fits file.
     */
    private void execute()
    {
        long start = System.currentTimeMillis();
        long setupTime = 0; 
        long execTime = 0;
        final SystemCommandFromArray s = new SystemCommandFromArray();
        s.setCommandLine(new String[] { fitsVerifyPathName, fitsFileFullPathName });
        logger.debug("Prepared fitsverify system command to run on: " + fitsFileFullPathName);
        try {
            setupTime = System.currentTimeMillis()-start;
            exitStatus = s.execute();
            execTime = System.currentTimeMillis()-start - setupTime;
            logger.debug("Sytem command: setup time[ms]=" + setupTime + " , execution time[ms]=" + execTime);
            message = s.getStandardOutputString();
            if (exitStatus != 0) {
                // something was wrong, there is an error/warning on stdout:
                error = s.getStandardErrorString() + message;
            }
            if ((error != null) && (!error.equals(""))) {
                logger.warn("fits verify reported errors - exit status=" + exitStatus);
            }
            logger.debug(fitsVerifyName + " exit status=" + exitStatus + " (run on " + fitsFileFullPathName + ")");
        } catch (final Exception e) {
            exitStatus = -1;
            message = "";
            error = e.toString();
            logger.error(error);
        }
    }

    /**
     * Try to locate the fits verify executable on the file system. 1. First
     * check the configured path and name, if available. 2. Then scan the
     * classpath and search in all directories and all parent directory of jar
     * files. 3. Then scan the PATH.
     */
    private synchronized void locateFitsVerify()
    {
        class CheckPath
        {
            private final String path;
            public CheckPath(final String path)
            {
                this.path = path;
            }
            /**
             * Check that this path is an executable file and returns its  
             * absolute path. 
             * @return absolute path of the fits verify executable, or null if 
             * the executable is not found.
             */
            public String getExecutable()
            {
                String locatedPath = null;
                final File fvExe = new File(path);
                if (fvExe.exists() && fvExe.isFile())
                {
                    if (fvExe.canExecute())
                    {
                        logger.debug(fvExe.getAbsolutePath()
                                + " is a valid executable. Using it as fits verify.");
                        locatedPath = fvExe.getAbsolutePath();
                    }
                    else
                    {
                        logger.error(fvExe.getAbsolutePath()
                                + " is not executable. Cannot run it.");
                    }
                }
                return locatedPath;
            }
            /**
             * Return the last component of path, i.e. the substring after the
             * last File.separator . If path ends with File.separator return an
             * empty string
             * 
             * @param path
             * @return last component of path
             */
            public String getName()
            {
                String name = "";
                if ((path.lastIndexOf(File.separator) > 0)
                        && (path.lastIndexOf(File.separator) < path.length()))
                {
                    name = path.substring(path.lastIndexOf(File.separator) + 1);
                }
                else
                {
                    name = path;
                }
                logger.debug("Extracted name from " + path + " [" + name + "]");
                return name;
            }
        } // End inner class.

        final String methodName = "FitsVerifyExecutorImp::locateFitsVerify";
        logger.trace(methodName);

        if (attemptedLocation) 
        { 
            // Maybe another thread executed this method while this thread was 
            // waiting to acquire the lock. 
            return; 
        } 
        attemptedLocation = true;
        fitsVerifyPathName = null; // will be not null if an executable is found.

        final String fromConf = config.getOptionValue(ValidatorConfiguration.FITSVERIFY.name);
        
        if (fromConf.equals(ValidatorConfiguration.FITSVERIFY.defaultValue))
        	return;

        if (!fromConf.equals(""))
        {
            logger.info("Utility to use as fits verify configured as: " + fromConf);
            final CheckPath checkPath = new CheckPath(fromConf);
            fitsVerifyName = checkPath.getName();
            fitsVerifyPathName = checkPath.getExecutable();
            if (fitsVerifyPathName != null)
            {
                // Get the real name, not a symlink:
                fitsVerifyName = (new CheckPath(fitsVerifyPathName)).getName();
                logger.info("Found executable declared in configuration. Path on disk:"
                        + fitsVerifyPathName);
                return;
            }
            else
            {
                logger.debug("Conf value is not a full path on disk. ["
                        + fitsVerifyName + "]");
            }
        }

        logger.info("Trying to locate fits verify (name: " + fitsVerifyName
                + " ) on the classpath.");
        final String classPath = System.getProperties().getProperty("java.class.path");
        final String[] pathel = classPath.split(File.pathSeparator);

        // Check if fits verify is located in one of the classpath elements.
        // Note: the first valid executable found will be used.
        for (final String el : pathel)
        {
            final File f = new File(el);
            File fd = null;
            if (FileSystemOperations.isFile(f))
            {
                fd = f.getParentFile();
                if (fd == null)
                {
                    // This case happens when the classpath entry refers 
                    // implicitly to the current directory (when the validator  
                    // is started from command line with java -jar validator.jar)
                    fd = new File(System.getProperty("user.dir"));
                }
            }
            else
            {
                fd = f;
            }
            logger.debug("Looking for fits verify in directory:" + fd.getAbsolutePath());
            fitsVerifyPathName = (new CheckPath(fd.getAbsolutePath() + File.separator + fitsVerifyName)).getExecutable();
            // Return if the executable was found:
            if (fitsVerifyPathName != null) {
                logger.info("Found executable on classpath. Path on disk:" + fitsVerifyPathName);
                return;
            }
        }

        // Last attempt: see if fitsVerifyName is on the PATH:
        logger.info("fits verify is not on the classpath. Searching the PATH for executables.");
        final SystemCommandFromArray sc = new SystemCommandFromArray();
        sc.setCommandLine(new String[] { fitsVerifyName });
        try {
            if (sc.execute() == 0) {
                // fitsVerifyName is on the executable path and therefore can be
                // executed even if its location on disk is not given:
                fitsVerifyPathName = fitsVerifyName;
                logger.info("Found executable on PATH for executables. Name:" + fitsVerifyPathName);
                return;
            }
        } catch (final Exception e) {
            error = "Failed attempted execution of " + fitsVerifyName + ": " + e.getMessage();
            logger.warn(error);
        }

        logger.warn("A candidate fits verify was not located.");
    }

}
