package org.eso.phase3.validator;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.log4j.Logger;

/**
 * HttpConf Multiple Headers: replacement for HttpConfImp in order to support  
 * OCA rules for extension headers. 
 * This class stores in a local buffer the content of the remote OCA rules files
 * and provides an <code>InputStrem</code> to read the buffer of each files. It
 * also stores the information retrieved from the category list file. The
 * category list file is in the format: category [associated-rules-file]
 * Examples: ancillary.spectrum.1d science.spectrum.1d science.spectrum.1d.oca
 * 1. A category is valid only if it has an entry in the category list file. 2.
 * If a category has an associated OCA rules file, the file must be declared in
 * the entry for the category. Note that the syntax is on purpose strict: no
 * comments allowed, only blanks to separate category and file, either 1 or 2
 * element per line. Anything else will raise an error.
 * 
 * @author dsforna
 * 
 */
public class HttpConfMH implements HttpConfMHI
{
    /** Apache Log4J logger for this class namespace. */
    private static final Logger logger = Logger.getLogger(HttpConfMHI.class);

    /**File HttpConf.CATG_LIST and OCA rule files must be found at this URL.*/
    private final String ocaBaseUrl;

    /**Each OCA rules file is read once through HTTP and its content is stored here.*/
    private final Map<String, byte[]> cachedOcaRules = 
            new HashMap<String, byte[]>();

    /**
     * Map with (category, OCA selectionRule filename) entries. 
     * Applying the selectionRule file to a fits file, the name of the 
     * OCA rules file to use for classification is retrieved. 
     */
    private final Map<String, String> categorySelectionRuleFileMap = 
            new HashMap<String, String>();

    public HttpConfMH(final ValidatorConfiguration config) throws IOException
    {
        this.ocaBaseUrl = config.getOptionValue(ValidatorConfiguration.URL.name);
        parseDeclaredCategories();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.HttpConf#getList()
     */
    @Override
    public Map<String, String> getDeclaredCategories()
    {
        return categorySelectionRuleFileMap;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.HttpConf#getFileStream(java.lang.String)
     */
    public InputStream getFileStream(final OcaRulesKey ocaRulesKey) throws IOException
    {
        
        if (ocaRulesKey == null)
        {
            logger.error("Null input argument: ocaRulesKey");
            throw new IllegalArgumentException("Null input argument: ocaRulesKey");
        }

        if (!hasRules(ocaRulesKey))
        {
            throw new IOException("No rules for: " + ocaRulesKey.getAssociateOcaRuleFile());
        }

        if (!cachedOcaRules.containsKey(ocaRulesKey.getAssociateOcaRuleFile()))
        {
            logger.debug("Reading and caching OCA rules for " + ocaRulesKey.getAssociateOcaRuleFile());
            cachedOcaRules.put(ocaRulesKey.getAssociateOcaRuleFile(), readFromUrl(ocaRulesKey.getAssociateOcaRuleFile()));
        }
        final InputStream is = new ByteArrayInputStream(
                cachedOcaRules.get(ocaRulesKey));
        return is;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.HttpConf#hasRules(java.lang.String)
     */
    @Override
    public boolean hasRules(final OcaRulesKey ocaRulesKey)
    {
        if (ocaRulesKey == null)
        {
            logger.error("Null input argument: ocaRulesKey");
            throw new IllegalArgumentException("Null input argument: ocaRulesKey");
        }

        if (ocaRulesKey.getAssociateOcaRuleFile() != null)
        {
            return true;
        }
        else
        {
            logger.debug("No rules defined [" + ocaRulesKey.toString() + "]");
            return false;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.HttpConf#isValid(java.lang.String)
     */
    public boolean isValid(final String catg)
    {
        if (catg != null)
        {
            return categorySelectionRuleFileMap.keySet().contains(catg);
        }
        else
        {
            return false;
        }
    }

    /**
     * Retrieve the URLspecification for the input category mapping the category
     * to its corresponding OCA rules file. If the input category does not have
     * an OCA rules file associated, this method throws an exception.
     * 
     * @param catg
     * @return the specification of the URL for the category OCA rules file.
     * @throws MalformedURLException
     */
    private String ocaFileUrlAsString(String ruleFileName)
            throws MalformedURLException
    {
        if (ruleFileName!= null)
        {
            // URL use always forward slash, so File.separator is wrong here:
            final String fileSpec = ocaBaseUrl + "/"
                    + ruleFileName;
            logger.debug("Returning for " + ruleFileName 
                    + " the String URL " + fileSpec);
            return fileSpec;
        }
        else
        {
            throw new MalformedURLException(ruleFileName
                    + "  does not have an associated oca rules file.");
        }
    }

    /**
     * Parse and store the category list file. Executed once, at construction.
     * 
     * @throws IOException
     *             if the category list file cannot be retrieved, or if the file
     *             is not in the expected format.
     */
    private void parseDeclaredCategories() throws IOException
    {

        String urlString = ocaBaseUrl + "/" + HttpConfMHI.CATG_LIST_FILE;
        final URL u = new URL(urlString);
        
        InputStreamReader is;
        try 
        {
            is = new InputStreamReader(u.openStream());
        }
        catch ( java.net.UnknownHostException e)
        {
            throw new IOException(
                    "Cannot determine the IP address of the host in the configured URL: " 
                    + urlString);
        }
        catch (Exception e)
        {
            throw new IOException("Cannot read from the configured URL: "+urlString);
        }
        
        if (is == null)
        {
            throw new IOException("Cannot find file " + HttpConfMHI.CATG_LIST_FILE
                    + " in " + ocaBaseUrl);
        }
        final BufferedReader br = new BufferedReader(is);
        if (br == null)
        {
            throw new IOException("Cannot read " + HttpConfMHI.CATG_LIST_FILE
                    + " from " + ocaBaseUrl);
        }

        // NOTE: If the syntax can be relaxed maybe it's better to use
        // Properties (i.e. p.load(is) ).
        int lineNumber = 0;
        while( true )
        {
            String l = br.readLine();
            lineNumber++;
            if (l == null)
            {
                break;
            }
            l = l.trim();
            if (!l.equals(""))
            {
                final String[] c = l.split("\\s+");
                OcaRulesKey key = new OcaRulesKey(
                        new String[]{Consts.CATG_KW}, new String[]{c[0].trim()}, 0);
                if (c.length == 1)
                {
                    categorySelectionRuleFileMap.put(c[0].trim(), null);
                }
                else if (c.length == 2)
                {
                    categorySelectionRuleFileMap.put(c[0].trim(), c[1].trim());
                }
                else
                {
                    throw new IOException("Invalid line in "
                            + HttpConfMHI.CATG_LIST_FILE + " . Line number "
                            + lineNumber + " [" + l + "]");
                }
            }
        }
    }

    /**
     * Read a stream of byte from the URL_CLI corresponding to the input
     * category.
     * 
     * @param catg
     * @return Array of read bytes.
     * @throws IOException
     */
    private byte[] readFromUrl(String ruleFileName) throws IOException
    {
        logger.trace("");
        final URL fileUrl = new URL(ocaFileUrlAsString(ruleFileName));
        if (fileUrl == null)
        {
            logger.error("Cannot read OCA rules for " +ruleFileName);
            throw new IOException("Cannot read OCA rules for " + ruleFileName);
        }
        final InputStream is = fileUrl.openStream();

        final ArrayList<Byte> ba = new ArrayList<Byte>(is.available());
        int nextByte = is.read();
        while( nextByte > -1 )
        {
            ba.add((byte) nextByte);
            nextByte = is.read();
        }
        is.close();
        final byte[] storedBytes = new byte[ba.size()];
        for (int i = 0; i < storedBytes.length; i++)
        {
            storedBytes[i] = ba.get(i);
        }
        return storedBytes;
    }

    /* (non-Javadoc)
     * @see org.eso.phase3.validator.HttpConfMHII#getFileStream(java.lang.String)
     */
    @Override
    public InputStream getFileStream(String ocaRulesFileName)
            throws IOException
    {
        if (ocaRulesFileName == null)
        {
            logger.error("Null input argument: ocaRulesFileName");
            throw new IllegalArgumentException("Null input argument: ocaRulesFileName");
        }


        if (!cachedOcaRules.containsKey(ocaRulesFileName))
        {
            logger.debug("Reading and caching OCA rules for " + ocaRulesFileName);
            cachedOcaRules.put(ocaRulesFileName, readFromUrl(ocaRulesFileName));
        }
        final InputStream is = new ByteArrayInputStream(
                cachedOcaRules.get(ocaRulesFileName));
        return is;
    }

    /* (non-Javadoc)
     * @see org.eso.phase3.validator.HttpConfMHII#getClassificationFileName(java.lang.String)
     */
    @Override
    public String getSelectionRulesFileName(String catg)
    {
        if (categorySelectionRuleFileMap.containsKey(catg)) 
        {
            return categorySelectionRuleFileMap.get(catg);
        }
        else
        {
            return null;
        }
    }
}
