/**
 * 
 */
package org.eso.phase3.validator;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import org.apache.log4j.Logger;
import org.eso.phase3.validator.ValidationReport.STATUS;

/**
 * This class takes care of running validation steps using a thread executor. 
 * A ValidationStepRunnerManager accepts in input ValidationStep which need to 
 * be run, and return in output the ValidationStep once they have been run.
 * @author dsforna
 *
 */
public class MultiThreadValidationStepRunnerManager implements
        ValidationStepRunnerManager
{
 
    /**
     * Helper inner class which takes care of the actual run of ValidationStep.
     * @author dsforna
     */
    class ValidationStepRunner implements Runnable
    {
        /**The validation step to run, as retrieved from the outer class' queue.*/
        private final ValidationStep vs;
        public ValidationStepRunner() throws InterruptedException 
        {
            String threadName = Thread.currentThread().getName();
            vs = inputBq.take();
            long start = System.currentTimeMillis();
            logger.debug(threadName + " created the ValidationStepRunner in " 
                        +(System.currentTimeMillis()-start) + " ms for " + vs.toString());
        }
        
        /* (non-Javadoc)
         * @see java.lang.Runnable#run()
         */
        @Override
        public void run()
        {
            logger.trace("");
            String threadName = Thread.currentThread().getName();
            logger.debug(threadName + " - Starting validation for  " 
                    + vs.toString());
            long start = System.currentTimeMillis();
            vs.runValidation();
            logger.debug(threadName + " - Finished validation in " 
                    +(System.currentTimeMillis()-start) +" ms for " + vs.toString());
            incProgressBar();
            try
            {
                start = System.currentTimeMillis();
                outputBq.put(vs);
                logger.debug(threadName + " Time spent to put in output queue: "
                        +(System.currentTimeMillis()-start) + " ms");
                
            }
            catch( InterruptedException e )
            {
                vs.validationReport().attemptStatus(STATUS.ERROR, e.getMessage());
                logger.error(threadName + " - " + e.toString());
            }
        }
    }

    /** Apache Log4J logger for this class namespace. */
    private static final Logger logger = Logger.getLogger(MultiThreadValidationStepRunnerManager.class);
    /**ValidationStep in input will be enqueued here while waiting to be run.*/
    private final BlockingQueue<ValidationStep> inputBq = new LinkedBlockingQueue<ValidationStep>();
    /**Once a ValidationStep has completed, it is put in this output queue.*/
    private final BlockingQueue<ValidationStep> outputBq = new LinkedBlockingQueue<ValidationStep>();
    
    /**How many ValidationStep have been received in input.*/
    private int acceptedSteps = 0;

    /**How many ValidationStep have been received in output.*/
    private int returnedSteps = 0;
    
    /**If this ValidationStepRunnerManager has started executing the ValidationStep.*/
    private boolean started = false;
    
    /**If this ValidationStepRunnerManager has stopped accepting ValidationStep in input.*/
    private boolean stopped = false;
    
    /**Progress bar's increment (in percent) for each executed ValidationStep.*/
    private final double incPerStep;  
    
    /**Progress bar to increment after executing a ValidationStep.*/
    private final ProgressBar progressBar;

    private final ExecutorService threadExecutor;

    public MultiThreadValidationStepRunnerManager(double incPerStep, ProgressBar progressBar, int howManyThreads)
    {
        this.incPerStep=incPerStep;  
        this.progressBar = progressBar;
        threadExecutor = Executors.newFixedThreadPool(howManyThreads);
    }

    private synchronized void incProgressBar() 
    {
        if (progressBar== null) 
        {
            return;
        }
        progressBar.increment(incPerStep);
    }

    
    /* (non-Javadoc)
     * @see org.eso.phase3.validator.ValidationStepRunnerManager#dequeue()
     */
    @Override
    public ValidationStep dequeue() throws InterruptedException
    {
        logger.trace("");
        final ValidationStep vs = outputBq.take(); 
        returnedSteps++;
        return vs;
    }

    /* (non-Javadoc)
     * @see org.eso.phase3.validator.ValidationStepRunnerManager#enqueue(org.eso.phase3.validator.ValidationStep)
     */
    @Override
    public void enqueue(ValidationStep vs)
    {
        logger.trace("");
        if (vs == null)
        {
            logger.error("Null input argument: vs");
            throw new IllegalArgumentException("Null input argument: vs");
        }
        
        if (stopped)
        {
            String msg = "ValidationStepRunnerManager was already stopped.";
            logger.error(msg);
            throw new IllegalArgumentException(msg);
        }
        
        try
        {
            inputBq.put(vs);
            acceptedSteps++;
            if (started)
            {
                while (! inputBq.isEmpty()) 
                {
                    threadExecutor.execute(new ValidationStepRunner());
                }

            }
        }
        catch( Throwable e )
        {
            vs.validationReport().attemptStatus(STATUS.ERROR, e.getMessage());
            logger.error(e.toString());
        }
        
    }

    /* (non-Javadoc)
     * @see org.eso.phase3.validator.ValidationStepRunnerManager#queueIsEmpty()
     */
    @Override
    public boolean queueIsEmpty()
    {
        logger.trace("");
        
        if (! started) { return inputBq.isEmpty(); }
        
        if (! stopped) 
        { 
            // Only after closing the input queue will check if there are still tasks enqueued.
            return false; 
        }
        return (returnedSteps == acceptedSteps);
    }

    /* (non-Javadoc)
     * @see org.eso.phase3.validator.ValidationStepRunnerManager#start()
     */
    @Override
    public void start()
    {
        logger.trace("");
        if (started)
        {
            String msg = "ValidationStepRunnerManager was already started.";
            logger.error(msg);
            throw new IllegalArgumentException(msg);
        }
        started = true;
        while (! inputBq.isEmpty()) 
        {
            try
            {
                threadExecutor.execute(new ValidationStepRunner());
            }
            catch( InterruptedException e )
            {
                logger.error(e.toString());
            }
        }
    }

    /* (non-Javadoc)
     * @see org.eso.phase3.validator.ValidationStepRunnerManager#stop()
     */
    @Override
    public void stop()
    {
        logger.trace("");
        stopped = true;
        while (! inputBq.isEmpty()) 
        {
            try
            {
                threadExecutor.execute(new ValidationStepRunner());
            }
            catch( InterruptedException e )
            {
                logger.error(e.toString());
            }
        }
        int pending = acceptedSteps -outputBq.size();

        logger.info("The pending " + pending
                + " ValidationStep will be executed, no additional ones will be enqueued.");
        threadExecutor.shutdown(); // no new threads started, but running threads will go on.
    }
}
