/**
 * 
 */
package org.eso.phase3.validator;

import java.io.File;

import org.apache.log4j.Logger;
import org.eso.phase3.validator.ValidationReport.STATUS;
import org.eso.phase3.validator.ValidatorStat.StatType;

/**
 * @author dsforna
 *
 */
public class NonFitsValidator implements ValidationStep
{
    /** Apache Log4J logger for this class namespace. */
    private static final Logger logger = Logger.getLogger(NonFitsValidator.class);

    
    /**Category of the fits file, as declared in the release's structure.*/
    private final String fileCategory;

    /**Path and name on disk of the fits file to validate.*/
    private final String fileFullPathName;

    /**Name of the fits file to validate.*/
    private String fileName;

    /**The status and messages of this validation will be stored here.*/
    private final ValidationReport fileReport;

    /**
     * If this attribute is set to true, the @{link #isValid()} method will
     * return always false. It is used to allow validation of files which are
     * already in error for some other reason.
     */
    private final boolean forceInvalid;

    /**The context of this validator.*/
    private final ValidatorSetup setup;

    /**
     * Has this validation step been successfully and fully executed (fits 
     * verify, OCA rules, additional checks)?
     */
    private boolean validationResult;

    /** Create a NonFitsValidator which is not forced in error before the validation.
     * Equivalent to {@link#NonFitsValidator(ValidatorSetup, String, String, boolean)}
     * with boolean set to false.  
     * 
     */
    public NonFitsValidator(final ValidatorSetup setup,
            final String fileFullPathName, final String fileCategory)
    {
        // forceInvalid flag is false by default:
        this(setup, fileFullPathName, fileCategory, false);
    }

    @Override public String toString()
    {
        return "VS of nonfits " + fileName;
    }

    
    /**Create a NonFitsValidator for the input file.
     * @param setup the context of this application.
     * @param fileFullPathName the pathname of the file to validate.
     * @param fileCategory the category of the file to validate.
     * @param forceInvalid if this NonFitsValidator is forced to return a 
     * validation error as result.
     */
    public NonFitsValidator(final ValidatorSetup setup,
            final String fileFullPathName, final String fileCategory,
            final boolean forceInvalid)
    {
        if (fileFullPathName == null)
        {
            logger.error("Null input argument: fileFullPathName");
            throw new IllegalArgumentException(
                    "Null input argument: fileFullPathName");
        }

        this.forceInvalid = forceInvalid;
        this.fileReport = new ValidationReport("FitsValidator on fits "
                + fileFullPathName + " [catg:" + fileCategory + "]", null);
        this.fileFullPathName = fileFullPathName;
        this.fileCategory = fileCategory;
        this.validationResult = false;
        this.setup = setup;

        fileName = null;
        try
        {
            fileName = fileFullPathName.substring(1 + 
                    fileFullPathName.lastIndexOf(File.separator));
        }
        catch( final java.lang.IndexOutOfBoundsException e )
        {
            fileName = fileFullPathName;
        }
    }

    
    /* (non-Javadoc)
     * @see org.eso.phase3.validator.ValidationStep#isValid()
     */
    @Override
    public boolean isValid()
    {
        logger.trace("");
        if (forceInvalid)
        {
            logger.debug("validation forced to error for " + fileName);
            return false;
        }
        else
        {
            logger.debug("Validation success for " + fileName + ": " + validationResult);
            return validationResult;
        }
    }

    
    /* (non-Javadoc)
     * @see org.eso.phase3.validator.ValidationStep#runValidation()
     */
    @Override
    public void runValidation()
    {
        logger.trace("");
        logger.debug("Validating file: "+ fileFullPathName);
        
        validationResult = true; // will be set to false when anything goes wrong.
        runMd5Sum();

        if (fileCategory == null)
        {
            validationResult = false;
            String msg = fileName + ": category is not set. Cannot validate file.";
            fileReport.attemptStatus(STATUS.ERROR, msg);
            logger.error(msg);
            return;
        }
        else if (!setup.getHttpConf().isValid(fileCategory))
        {
            validationResult = false;
            String origin = setup.getReleaseParser().getCategoryLocationMap().get(fileName);
            if (origin == null) {origin="(original file no longer available)";}
            String msg = origin + ": invalid category " + fileCategory 
                    + " defined for file " + fileName;
            fileReport.attemptStatus(STATUS.ERROR, msg);
            logger.error(msg);
            return;
        }
        else
        {
            logger.debug(fileCategory + " is a valid category");
        }


        if (validationResult)
        {
            fileReport.attemptStatus(STATUS.VALID);
            logger.info("Validation correctly performed for: "
                    + fileFullPathName);
        }
        return;
    }

    /**
     * Run the md5sum on this file. Update the statistic and this ValidationReport
     * if the computed value does not match with the value in the release's structure.
     * @param expected the value of the md5sum (hex string) declared in the in 
     * the release's structure for this file. 
     */
    private void runMd5Sum()
    {
        String expected = setup.getReleaseParser().getMd5Sum(fileName); 
        if (expected == null)
        {
           // Currently the md5sum is not mandatory - nothing to do.
           return;
        }
        try 
        {
            String computed = ValidationUtil.md5sum(fileFullPathName);
            if (computed.toLowerCase().equals(expected.toLowerCase()))
            {
                logger.debug(fileName 
                        + " . Computed md5sum matches the expected md5sum ("
                        + expected + ")");
            }
            else 
            {
                String msg = fileName + ": expected md5sum=" + expected 
                        + " is different from the computed md5sum="+computed;
                logger.error(msg);
                validationResult = false;
                fileReport.attemptStatus(STATUS.ERROR, msg);
                setup.getValidatorStat().add(StatType.ERROR_MD5);
            }
        }
        catch (Exception e)
        {
            String msg = fileName + " - Exception computing md5sum: ";
            logger.error(msg+ e.toString());
            validationResult = false;
            fileReport.attemptStatus(STATUS.ERROR, msg+e.getMessage());
            setup.getValidatorStat().add(StatType.ERROR_MD5);
        }
    }

    /* (non-Javadoc)
     * @see org.eso.phase3.validator.ValidationStep#validationReport()
     */
    @Override
    public synchronized ValidationReport validationReport()
    {
        logger.trace("");
        return this.fileReport;
    }
}
