package org.eso.phase3.validator;

import java.io.FileReader;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.apache.log4j.Logger;

/**
 * This client does not really connect to an external server but instead it
 * returns {@link ArchivedFilesClient.ARCHIVED} as status (by default) for any
 * input file. A different list of statuses can be specified in an optional
 * configuration file which can be passed in the system property
 * FORCE_ARCHIVED_STATUS (i.e. by adding
 * "-Dforce.archived.status=config_file_name" on command line): in this case
 * return as status what is declared in this configuration file. To instantiate
 * this object in the validator:
 * <ul>
 * <li>Set in the validator configuration the property: <br>
 * validator.mockup.file = null_archived_files.properties</li>
 * <li>Put on the classpath a file called null_archived_files.properties with
 * the single line:<br>
 * org.eso.phase3.validator.ArchivedFilesClientImp =
 * org.eso.phase3.validator.NullArchivedFilesClient</li>
 * </ul>
 * 
 * @author dsforna
 */
public class NullArchivedFilesClient implements ArchivedFilesClient
{
    /**
     * Optional property for the filename which holds a list of file id with
     * forced archived status in the format of a Properties object.
     */
    public static String FORCE_ARCHIVED_STATUS = "force.archived.status";

    /** Apache Log4J logger for this class namespace. */
    private static final Logger logger = Logger.getLogger(NullArchivedFilesClient.class);

    private boolean connectedStatus = true;

    private boolean errorStatus = false;

    private Properties forcedFiles = null;

    /**
     * Create the object and load the list of forced status if the filename with
     * the list is given in the System Properties.
     * 
     * @param config
     *            not used.
     */
    public NullArchivedFilesClient(final ValidatorConfiguration config)
    {
        logger.debug("Creating NULL object");
        final String forcedListFile = System.getProperty(FORCE_ARCHIVED_STATUS);
        if (forcedListFile != null)
        {
            logger.info("Loading list of forced files/status : "
                    + FORCE_ARCHIVED_STATUS);
            try
            {
                forcedFiles = new Properties();
                forcedFiles.load(new FileReader(forcedListFile));
            }
            catch( final Exception e )
            {
                logger.error("Error loading  list of forced files/status ("
                        + FORCE_ARCHIVED_STATUS + "). " + e.toString());
                forcedFiles = null;
            }
        }

        if (forcedFiles == null)
        {
            // (Null object pattern.)
            forcedFiles = new Properties();
        }
    }

    /**
     * Return
     * 
     * @see org.eso.phase3.validator.ArchivedFilesClient#getArchived(java.util.Set)
     */
    @Override
    public Set<String> getArchived(final Set<String> fileIds)
            throws ValidatorException
    {

        final Map<String, String> retMap = status(fileIds);
        final HashSet<String> ret = new HashSet<String>();
        for (final String key : retMap.keySet())
        {
            if (retMap.get(key).equals(ARCHIVED))
            {
                ret.add(key);
            }
        }
        return ret;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.ArchivedFilesClient#hasError()
     */
    @Override
    public boolean hasError()
    {
        return errorStatus;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.ArchivedFilesClient#isConnected()
     */
    @Override
    public boolean isAvailable()
    {
        logger.trace("");
        return connectedStatus;

    }

    /**
     * Force the HTTP connection status of this object.
     * 
     * @param status
     *            the forced status.
     */
    public void setConnected(final boolean status)
    {
        logger.info("The HTTP connection status for this boject will be set to "
                + status);
        connectedStatus = status;
    }

    /**
     * Force the Error status of this object.
     * 
     * @param status
     *            the forced status.
     */
    public void setError(final boolean status)
    {
        logger.info("The Error status for this boject will be set to " + status);
        errorStatus = status;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.ArchivedFilesClient#status(java.util.Set)
     */
    @Override
    public Map<String, String> status(final Set<String> fileIds)
            throws ValidatorException
    {
        if (fileIds == null)
        {
            logger.error("Null input argument: fileIds");
            throw new IllegalArgumentException("Null input argument: fileIds");
        }
        if (!connectedStatus)
        {
            throw new ValidatorException("Not connected.");
        }
        final Map<String, String> retMap = new HashMap<String, String>();
        for (final String key : fileIds)
        {
            String value = ARCHIVED;
            if (forcedFiles.containsKey(key))
            {
                value = forcedFiles.getProperty(key);
                logger.debug("Found a forced status for " + key + ": " + value);
            }

            if (!ALLOWED_RETURN_VALUES.contains(value))
            {
                logger.error("Status " + value + "will be forced to "
                        + UNKNOWN_RETURNED_VALUE_ERROR
                        + " because it is not in the list of allowed values ("
                        + Arrays.toString(ALLOWED_RETURN_VALUES.toArray())
                        + ")");
                value = UNKNOWN_RETURNED_VALUE_ERROR;
            }
            retMap.put(key, value);
        }
        return retMap;
    }
}
