package org.eso.phase3.validator;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;
import java.util.TreeSet;
import java.util.Vector;

import org.apache.log4j.Logger;
import org.eso.oca.fits.DataTransportFormatHandler;
import org.eso.oca.fits.OCAFile;
import org.eso.oca.fits.OCAFileFactory;
import org.eso.oca.fits.OCAFileFilter;
import org.eso.oca.fits.TypedHeaderCard;
import org.eso.oca.fits.TypedHeaderCardException;

/**
 * This class adapts the {@link OCAFileFactory} to extract info from a single
 * file instead of from all the files in a directory. The result is a list, but
 * always of one element (OCAFile object) or zero elements in case of error.
 * 
 * @author dsforna
 */

public class OCASingleFileFactory extends OCAFileFactory
{

    /** Apache Log4J logger for this class namespace. */
    private static final Logger logger = Logger.getLogger(OCASingleFileFactory.class);

    /**
     * Splits a full pathname in {directory, filename} elements that can then be
     * passed to this constructor.
     * 
     * @param filePathName
     *            full pathname on disk of a file
     * @return 2-elements array with directory and filename.
     */
    public static String[] splitFullPath(final String filePathName)
    {
        final String methodName = "OCASingleFileFactory::splitFullPath";
        logger.trace(methodName);
        if (filePathName == null)
        {
            logger.warn(methodName + " - Null pathname in input.");
            return null;
        }
        final int lastSeparator = filePathName.lastIndexOf(File.separator);
        final String[] ret = { filePathName.substring(0, lastSeparator),
            filePathName.substring(lastSeparator + 1) };
        return ret;
    }

    private final String fileName;
    private final DataTransportFormatHandler dtfh;
    
    public OCASingleFileFactory(final String directory, final String fileName, final DataTransportFormatHandler fh)
    {
        super(directory);
        this.fileName = fileName;
        this.dtfh = fh;
    }

    
    @Override
    public void createOCAFiles(final int headerIndex, final String[] keywords,
            final OCAFileFilter filter, final ArrayList<OCAFile> readFiles,
            final ArrayList<File> skippedFiles)
    {
        Map<String, String>empty = Collections.emptyMap();
        createOCAFilesWithExtraKeywords(empty, 
                headerIndex, keywords, filter, readFiles, skippedFiles);
    }
    
    /**
     * Extracts the specified keywords from the given FITS extension of the
     * single HDR/FITS/TFITS file specified at construction. The resulting
     * OCAFile is appended to readFiles. If the file were skipped is appended to
     * skippedFiles.
     * 
     * @throws NullPointerException
     *             if keywords, readFiles or skippedFiles is null.
     * 
     * @param extraKeywordsAndValues
     *            the keywords (and values) to be added to keywords 
     * @param headerIndex
     *            the FITS extension from which keywords should be read.
     * @param keywords
     *            the list of keywords to be extracted.
     * @param filter
     *            . Not used.
     * @param readFiles
     *            the read file will be appended to this list.
     * @param skippedFiles
     *            the skipped file will be appended to this list.
     */
    public void createOCAFilesWithExtraKeywords(
            Map<String, String> extraKeywordsAndValues,
            final int headerIndex, final String[] keywords,
            final OCAFileFilter filter, final ArrayList<OCAFile> readFiles,
            final ArrayList<File> skippedFiles)
    {
        final String methodName = "OCASingleFileFactory::createOCAFiles(int, String[], "
                + "OCAFileFilter, ArrayList<OCAFile>, ArrayList<File>)";

        logger.trace(methodName);

        // The keyword array and file lists must not be null.
        if ((keywords == null) || (readFiles == null) || (skippedFiles == null))
        {
            final String message = methodName + " - null parameter(s):"
                    + (keywords == null ? " keywords" : "")
                    + (readFiles == null ? " readFiles" : "")
                    + (skippedFiles == null ? " skippedFiles" : "") + ".";

            logger.fatal(message);
            throw new NullPointerException(message);
        }

        // Supplement the requested keywords with the standard set that
        // virtually all applications require. Kept empty for now.
        final TreeSet tmpData = new TreeSet(Arrays.asList(keywords));
        final String[] requestedData = (String[]) tmpData.toArray(new String[0]);

        // Get the list of files in the current directory.
        final Vector<String> files = new Vector<String>();
        files.add(super.getRootDir() + File.separator + fileName);

        // Loop over the list of files.
        for (final Iterator<String> it = files.iterator(); it.hasNext();)
        {
            // Get the current filename in canonical form.
            final String linkname = it.next();
            final String filename = OCAFile.getCanonicalPath(linkname);
            if (filename == null)
            {
                // This filename does not have a canonical form - skip it.
                logger.error("Skipping invalid pathname: " + linkname);
                skippedFiles.add(new File(linkname));
                continue;
            }
            logger.debug("Processing pathname: " + linkname);
            // Get the requested keywords from the FITS header.
            TypedHeaderCard[] cards = null;
            try
            {
                cards = dtfh.getFITSCards(headerIndex, requestedData);
            }
            catch( final Exception e )
            {
            	String msg = "Unable to retrieve FITS cards for file: " + linkname + " [" + e.toString() + "]";
            	if (e.getCause() != null)
            		msg += ". Cause: [" + e.getCause().toString() + "]";
                logger.error(msg);
                skippedFiles.add(new File(linkname));
                continue;
            }

            // Convert the list of FITS cards into a Hashtable
            // (to be stored in an OCAFile).
            Hashtable keywordValues;

            try
            {
                keywordValues = convertArrayOfCardsToHashtable(cards);
            }
            catch( final TypedHeaderCardException e )
            {
                logger.error(methodName + " - unable to convert FITS cards "
                        + "to hashtable for file [" + linkname + "] ["
                        + e.toString() + "].");

                skippedFiles.add(new File(linkname));
                continue;
            }

            // Add the meta-keywords FILENAME and LINKNAME.
            keywordValues.put(OCAFile.FILENAME, filename);
            keywordValues.put(OCAFile.LINKNAME, linkname);


            // Add any extra keywords needed for classifications:
            for (String keyword: extraKeywordsAndValues.keySet())
            {
                // Seems that EXTNUM must be a java.math.BigInteger, but I let 
                // the OCA TypedHeaderCard object set it:
                if (keyword.equals(Consts.EXTNUM))
                {
                    //keywordValues.put(keyword, java.math.BigInteger.valueOf(Long.decode(extraKeywordsAndValues.get(keyword)));
                    try
                    {
                        TypedHeaderCard c = new TypedHeaderCard(keyword, 
                                Integer.decode(extraKeywordsAndValues.get(keyword)), 
                                "no comment");
                        keywordValues.put(keyword, c.getValueObject());
                    }
                    catch(Exception e )
                    {
                        logger.error(e.toString());
                    }
                }
                else
                {
                    keywordValues.put(keyword, extraKeywordsAndValues.get(keyword));
                }
            }
            
            
            // Create a new OCAFile from the card Hashtable.
            final OCAFile ocaFile = new OCAFile(keywordValues);

            // Check the new OCAFile against the file filter.
            if (filter != null)
            {
                if (filter.filter(ocaFile) == true)
                {
                    logger.debug(methodName + " - file filter accepted file ["
                            + linkname + "].");
                    readFiles.add(ocaFile);
                }
                else
                {
                    logger.debug(methodName + " - file filter rejected file ["
                            + linkname + "].");
                    skippedFiles.add(new File(linkname));
                }
            }
            else
            {
                logger.debug("No filter. Accepted file [" + linkname + "].");
                readFiles.add(ocaFile);
            }
        } // End Loop over the list of files.
    }
}
