package org.eso.phase3.validator;

import java.io.PrintStream;

/**
 * API of the validator main class. An application can call these methods
 * instead of calling the validator as separated command line process.
 * 
 * @author dsforna
 */
public interface Phase3Validator
{
    /**
     * Enable a progress bar on the input PrintStream. If the input is null,
     * System.out is used.
     * 
     * @param out
     *            PrintStream where to send the progress bar
     */
    public void enableProgressBar(PrintStream out);

    /**
     * Pass as argument the location of the package to validate. All other
     * parameters should be already configured with default values because do
     * not depend on the package location.
     * 
     * @param releaseDirectory
     *            the string path of the Release directory.
     * @return the result of validation (true/false).
     * @throws ValidatorException
     *             if it arises during the validation.
     */
    public boolean validate(String releaseDirectory) throws ValidatorException;

    /**
     * Pass as first argument the file and path of a configuration files and as
     * second argument the location of the package to validate. The
     * configuration file should hold the part of the configuration which do not
     * depends on the package to check. Note that the validator provides default
     * configuration values: this configuration file is needed only if the
     * default values must be overridden.
     * 
     * @param confFilePath
     *            path and name of the configuration file.
     * @param releaseDirectory
     *            the string path of the Release directory.
     * @return the result of validation (true/false).
     * @throws ValidatorException
     *             if it arises during the validation.
     */
    public boolean validate(String confFilePath, String releaseDirectory)
            throws ValidatorException;

    /**
     * The input String[] <code>args</code> must be in the same format of
     * command line arguments. For instance, in order to pass only the directory
     * of the release to validate, <code>args</code> must be a String[2], with
     * the first element equal to either "-r" or "--reldir".
     * 
     * @param args
     *            the input arguments as would be received by the main().
     * @return the result of validation (true/false).
     * @throws ValidatorException
     *             if it arises during the validation.
     */
    public boolean validate(String[] args) throws ValidatorException;

    /**
     * @return a String holding all the errors on performed validation. If there
     *         are no error the String is empty and not null.
     * @throws ValidatorException
     *             if the validation was not performed yet.
     */
    public String validationErrors() throws ValidatorException;

    /**
     * @return a String holding statistic on performed validation.
     * @throws ValidatorException
     *             if the validation was not performed yet.
     */
    public String validationStatistics() throws ValidatorException;

    /**
     * After validation an application can call this method to obtain a detailed
     * description of the validation's execution. If this method is called
     * before the validation, it raises a ValidatorException.
     * 
     * @return the validation description as String.
     * @throws ValidatorException
     *             if the validation was not performed yet.
     */
    public String verboseReport() throws ValidatorException;
    
    public ValidatorSetup getSetup();

}