package org.eso.phase3.validator;
import java.io.File;
import java.io.FileWriter;
import java.io.PrintStream;
import java.util.Enumeration;

import org.apache.log4j.Appender;
import org.apache.log4j.FileAppender;
import org.apache.log4j.Logger;
import org.eso.phase3.validator.catalog.IndexedKeywordValidator;

/**
 * The phase 3 validator main class: It has a main so that it can used as
 * command line application, and an API so that an external client can
 * programmatically access it.
 * 
 * @author dsforna
 */
public class Phase3ValidatorImp implements Phase3Validator
{
    /** Apache Log4J logger for this class namespace. */
    private static final Logger logger = Logger.getLogger(Phase3ValidatorImp.class);

    /**
     * Main run, executed when the validator is called from command line.
     * 
     * @param args
     *            the configuration input parameters.
     * @throws ValidatorException
     *             in case of error during the validation. Note that exception
     *             at startup are trapped so the usage help can be displayed.
     */
    public static void main(final String[] args) throws ValidatorException
    {
    	long start = System.currentTimeMillis();    	

        Phase3ValidatorImp validator = null;
        System.out.println("Starting up validation tool.");
        ValidationUtil.configureLog4j();
        try
        {
            if (args.length == 0)
            {
                // print help:
                new ValidatorConfigurationCli(
                        new String[]{"--"+ValidatorConfiguration.HELP.longCli}); 
            }
            
            validator = new Phase3ValidatorImp(args);
        }
        catch( final ValidatorException e )
        {
            logger.error(e.toString());
            System.out.println("");
            System.out.println("Starting up failed because of the following error.");
            Throwable t = e.getCause();
            // Print only the original cause and not the chain of exception:
            String errMsg = e.getMessage();
            while (t != null)
            {
                if (t.getMessage() != null)
                {
                    errMsg = t.getMessage();
                }
                
                t = t.getCause();
            }
            System.out.println(errMsg);
            //System.out.println(e.getMessage());
            //System.out.println(e.toString());            
            System.out.println("");
            System.out.println("use -h option to get this usage help.");
            new ValidatorConfigurationCli(
                    new String[]{"--"+ValidatorConfiguration.HELP.longCli}); 
            return;
        }
        System.out.println("Performing validation. Please wait...");

        validator.enableProgressBar(System.out);

        final boolean ret = validator.doValidation();
        String msg = null;
        if (ret)
        {
            msg = Consts.MSG_OK;
        }
        else
        {
            msg = Consts.MSG_ERROR;
        }

        boolean tocFileOk = true;
        try
        {
            final File tocFile = new File(System.getProperty("user.dir")
                    + File.separator + "validator.toc");
            if (tocFile.exists())
            {
                if (! tocFile.delete()) 
                {
                    logger.error("Cannot delete pre-existing file: " 
                            + tocFile.getAbsolutePath());
                }

            }
            final FileWriter fw = new FileWriter(tocFile);
            fw.write(Consts.STATUS_HEADER + "\n");
            fw.write(msg + "\n");
            fw.write(Consts.SUMMARY_HEADER + "\n");
            fw.write(validator.validationStatistics());
            fw.write(Consts.TOC_HEADER + "\n");
            fw.write(validator.verboseReport());
            fw.close();
            validator.setup.getProgressBar().displayPercent(100);

            System.out.println("");
            System.out.println("Release summary and content written on file: "
                    + tocFile.getAbsolutePath());
        }
        catch( final Exception e )
        {
            logger.error(e.toString());
            System.out.println(e.getMessage());
            tocFileOk = false;
        }

        boolean errorFileOk = true;
        try
        {
            final File errorFile = new File(System.getProperty("user.dir")
                    + File.separator + "validator.error");
            if (errorFile.exists())
            {
                if (! errorFile.delete()) 
                {
                    logger.error("Cannot delete pre-existing file: " 
                            + errorFile.getAbsolutePath());
                }
            }
            System.out.println("Release error(s) are being written on file: "
                    + errorFile.getAbsolutePath());
            final FileWriter fw = new FileWriter(errorFile);
            fw.write(validator.validationErrors());
            fw.close();
        }
        catch( final Exception e )
        {
            logger.error(e.toString());
            System.out.println(e.getMessage());
            errorFileOk = false;
        }

        if (!tocFileOk)
        {
            System.out.println(validator.verboseReport());
        }

        if (!errorFileOk)
        {
            System.out.println(validator.validationErrors());
        }

        final Enumeration<Appender> ea = Logger.getRootLogger().getAllAppenders();
        while( ea.hasMoreElements() )
        {
            final Appender a = ea.nextElement();
            if (a instanceof FileAppender)
            {
                // WARN: here it is assuming that there is a single file
                // appender
                // which is the then log file.
                System.out.println("Log file: " + ((FileAppender) a).getFile());
            }
        }
        System.out.println("");
        System.out.println(validator.validationStatistics());
        System.out.println(msg);
        System.out.println("Execution time: " + String.valueOf(0.001*(System.currentTimeMillis()-start)));
    }

    /**
     * At each validation all the sub-objects are re-instantiated, but this
     * object needs to keep track of the ValidatorManager because it is used to
     * provide the validation reports.
     */
    private ValidatorManager mngr = null;

    private boolean mngrConstructed = false;

    /** The context of the validator application. */
    private ValidatorSetup setup = null;
    
    /* true for internal (IDP) releases */
    private boolean internal = false;

    /**
     * Instantiate an object but delay the actual construction on a later
     * moment, so that the configuration input parameters do not need to be
     * specified here.
     */
    public Phase3ValidatorImp()
    {
    	/* DFS11237 reset catalog validator from previous executions */
    	IndexedKeywordValidator.reset();
    }
    
    public Phase3ValidatorImp(boolean internal)
    {
    	this.internal = internal;
    	/* DFS11237 reset catalog validator from previous executions */
    	IndexedKeywordValidator.reset();
    }

    /**
     * Instantiate an object using the configuration input parameters. The
     * actual work is delegated to the method {@link construct}.
     * 
     * @param args
     *            command line arguments used as configuration values.
     * @throws ValidatorException
     */
    public Phase3ValidatorImp(final String[] args) throws ValidatorException
    {
        construct(args);
    }

    /**
     * Perform the validation operations.
     * 
     * @return the result of the validation as boolean.
     * @throws ValidatorException
     *             When the validation operations raise an exception.
     */
    public boolean doValidation() throws ValidatorException
    {
        mngr = setup.getValidatorManager();
        mngrConstructed = true;
        final boolean res = mngr.validateRelease(internal);
        return res;
    }

    /*
     * (non-Javadoc)
     * 
     * @seeorg.eso.phase3.validator.Phase3Validator#enableProgressBar(java.io.
     * PrintStream)
     */
    @Override
    public void enableProgressBar(PrintStream out)
    {
        logger.trace("");
        if (out == null)
        {
            out = System.out;
        }

        setup.getProgressBar().setPrintStream(out);
        setup.getProgressBar().start(Consts.PROGRESS_BAR_LENGTH, 0,
                Consts.PROGRESS_BAR_MIN_DELTA_MSEC);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.Phase3Validator#validate(java.lang.String)
     */
    public boolean validate(final String releaseDirectory)
            throws ValidatorException
    {
        logger.trace("");
        if (releaseDirectory == null)
        {
            logger.error("Null input argument: releaseDirectory");
            throw new IllegalArgumentException(
                    "Null input argument: releaseDirectory");
        }
        final boolean ret = validate(new String[] {
            "--" + ValidatorConfiguration.RELEASE_DIR.longCli, releaseDirectory });
        return ret;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.Phase3Validator#validate(java.lang.String,
     * java.lang.String)
     */
    @Override
    public boolean validate(final String confFilePath,
            final String releaseDirectory) throws ValidatorException
    {
        logger.trace("");
        if (releaseDirectory == null)
        {
            logger.error("Null input argument: releaseDirectory");
            throw new IllegalArgumentException(
                    "Null input argument: releaseDirectory");
        }
        boolean ret;
        if (confFilePath == null)
        {
            logger.warn("Ignoring null input file path.");
            ret = validate(releaseDirectory);
        }
        else
        {
            ret = validate(new String[] {
                "--" + ValidatorConfiguration.RELEASE_DIR.longCli,
                releaseDirectory,
                "--" + ValidatorConfiguration.CONF_FILE_PATH.longCli,
                confFilePath });
        }
        return ret;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.eso.phase3.validator.Phase3Validator#validate(java.lang.String[])
     */
    @Override
    public boolean validate(final String[] args) throws ValidatorException
    {
        logger.trace("");
        construct(args);
        return doValidation();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.Phase3Validator#validationErrors()
     */
    @Override
    public String validationErrors() throws ValidatorException
    {
        logger.trace("");
        if (!mngrConstructed)
        {
            throw new ValidatorException("No validation performed yet.");
        }
        return mngr.getErrors();

    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.Phase3Validator#validationStatistics()
     */
    @Override
    public String validationStatistics()
    {
        return mngr.statistic();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.Phase3Validator#reportOfValidation()
     */
    public String verboseReport() throws ValidatorException
    {
        final String methodLogName = "Phase3ValidatorImp::reportOfValidation";
        logger.trace(methodLogName);
        if (!mngrConstructed)
        {
            throw new ValidatorException(
                    "Validation manager was not constructed yet.");
        }
        return mngr.reportTable();
    }

    /**
     * This method actually performs the construction, instantiating the
     * sub-objects of this validator, with the exception of the
     * {@link ValidatorManager}, which is re-instantiated for each validation.
     * This must be called before starting the first validation.
     * 
     * @param args
     *            String[] of input arguments. If it is null an
     *            IllegalArgumentException is thrown.
     * @throws ValidatorException
     */
    private void construct(final String[] args) throws ValidatorException
    {
        logger.trace("");
        if (args == null)
        {
            logger.error("Null input argument: args");
            throw new IllegalArgumentException("Null input argument: args");
        }
        // Re-init this object:
        setup = new ValidatorSetupImp(args);
        mngr = null;
        mngrConstructed = false;
    }

	public ValidatorSetup getSetup() {
		return setup;
	}
}
