package org.eso.phase3.validator;

import java.io.PrintStream;

import org.apache.log4j.Logger;

/**
 * Display a simple command line progress bar. In case this simple bar is not
 * adequate, a better implementation could be achieved using the JCurses
 * library. See http://sourceforge.net/projects/javacurses/
 * 
 * @author dsforna
 */
public class ProgressBar
{
    public static final int BLINK_PERIOD_MSEC = 40;

    /** blink every this number of carry overs*/ 
    public static final int CARRY_OVERS_PER_BLINK = 80;

    public static final String DOING_CHAR = "*";

    public static final String DONE_CHAR = ".";

    public static final String FRAME_CHAR = "|";

    /** Apache Log4J logger for this class namespace. */
    private static final Logger logger = Logger.getLogger(ProgressBar.class);

    /**
     * Part of the total length of a progress bar is used for the open/close
     * frame chars.
     */
    private volatile int effectiveLength = 0;
    
    /**
     * If false this progress bar will not blink the last char when updated
     * without a change in displayed length.
     */
    private volatile boolean canBlink = true;

    private volatile double carryOver = 0.0;

    private volatile int carryOversFromLastBlink=0;

    /** Percent of completion in the previous updated event. */
    private volatile int lastPercent = 0;

    /** When the last update took place. */
    private volatile long lastTime = 0;

    /**
     * An update event which arrives earlier than this quantity after the
     * previous update will be ignored.
     */
    private volatile long minDeltaUpdatemsec = 0;

    /** Where to print this progress bar. */
    private volatile PrintStream out = null;

    /** Whether this progress bar is started for updating. */
    private volatile boolean started = false;

    public ProgressBar()
    {
    }

    /**
     * Blink the last char to show some activity even if the length is not
     * increased.
     */
    public void blink()
    {
        displayPercent(getLastPercent());
    }

    /**
     * @return the current percent of this progress bar.
     */
    public int getLastPercent()
    {
        return lastPercent;
    }

    /**
     * Accumulate fraction of percent increments and display them together when 
     * the sum reaches an integer value. Note that a rapid accumulation can 
     * bypass the minimum delta time between two updates, but anyway there can 
     * be no more than 100 of such updates during the progress bar life cycle.
     * 
     * @param inc
     */
    public synchronized void increment(final double inc)
    {
        if (!started)
        {
            return;
        }
        
        //logger.debug("Progress bar: incrementing of " + inc + " last%=" 
        // + lastPercent + ", carryOver="+carryOver);
        carryOversFromLastBlink++;
        carryOver += inc;
        final int delta = (int) carryOver;
        if (barCharLen(delta) >= 1)
        {
            final int newPercent = lastPercent + delta;
            doDisplayPercent(newPercent);
            if (newPercent == lastPercent) 
            {
                // The update was actually performed.
                carryOversFromLastBlink = 0;
                carryOver = carryOver - delta;
            }
        }
        
        if (carryOversFromLastBlink > CARRY_OVERS_PER_BLINK) 
        {
            blink();
        }

    }

    /**
     * The blinking is possible only for PreintStream where backspaces are
     * allowed (example: from Eclipse IDE they are not, from terminal they are).
     * 
     * @param canBlink
     *            the blink status
     */
    public void setBlink(final boolean canBlink)
    {
        this.canBlink = canBlink;
    }

    /**
     * Before being used a ProgressBar must be first told where to send the
     * output.
     * 
     * @param out
     */
    public void setPrintStream(final PrintStream out)
    {
        if (out != null)
        {
            this.out = out;
            logger.debug("Output PrintStream set.");
        }
        else
        {
            logger.warn("Null input PrintStream: ignored.");
            if (this.out == null)
            {
                logger.info("This progress bar will stay disabled.");
            }
        }
    }

    /**
     * Start this progress bar with an initial percent of completion. The
     * initial percent can be also 0. A progress bar can be updated only after
     * it has been started.
     * 
     * @param barLength
     *            the total length of this progress bar in characters
     * @param percent
     *            the initial percent of completion (0..100)
     * @param minDeltaUpdatemsec
     *            the minimum interval of time between 2 updates, in
     *            milliseconds.
     */
    public void start(final int barLength, final int percent,
            final int minDeltaUpdatemsec)
    {
        if (out == null)
        {
            logger.warn("Output PrintStream is still null, set it first.");
            return;
        }
        started = true;
        this.minDeltaUpdatemsec = minDeltaUpdatemsec;
        lastPercent = percent;
        lastTime = now();
        effectiveLength = barLength - 2 * FRAME_CHAR.length();

        // ( +2 to center the approximation error of dividing an integer by 4).
        final int pos25 = (effectiveLength + 2) / 4;

        // Note that the total length of the header is: 4 + 4*pos25
        // (the added 4 is the length in characters of "100%").
        // // final String header25 = String.format("%-" + pos25 + "s" + "%-" + pos25
        // //         + "s" + "%-" + pos25 + "s" + "%-" + pos25 + "s" + "%s", "INIT",
        // //         "PARSE HEADERS", "CHECK CONSISTENCY", "VERIFY FITS", "DONE");

        // Even if not using the real percent in Consts, this is more similar to
        // a real run:
        final String header = String.format("%-" + (pos25-10) + "s" + "%-" + (pos25-5)
                + "s" + "%-" + (pos25) + "s" + "%-" + (pos25+15) + "s" + "%s", 
                "INIT","PARSE HEADERS", "CHECK CONSISTENCY", "VERIFY FITS", "DONE");
        
        out.println("");
        out.println(header);
        out.print(FRAME_CHAR);
        out.print(DONE_CHAR);
        for (int i = 1; i <= barCharLen(percent); i++)
        {
            out.print(DONE_CHAR);
        }
        doBlink();
    }

    /**
     * Update the displayed bar if enough time is elapsed from last update.
     * Updating means to set the length in accordance with the input percent of
     * completion. If the minimum waiting time between 2 updates is set, this
     * method might actually skip the update of the progress bar.
     * 
     * @param percent
     *            the percent of completion (0..100) to be shown. Note that it
     *            is an absolute value, not an increment on the previous value.
     */
    public synchronized void displayPercent(final int percent)
    {
        if (!started)
        {
            return;
        }
        carryOversFromLastBlink = 0;
        carryOver = 0;
        doDisplayPercent(percent);
    }

    /**
     * @param percent
     *            the percent (0..100) of completion for the task associated to
     *            this progress bar.
     * @return the corresponding length in character of the displayed progress
     *         bar.
     */
    private int barCharLen(final int percent)
    {
        final int charLen = (percent * effectiveLength) / 100;
        return charLen;
    }

    /**
     * Temporary replace the DONE_CHAR at the current cursor position with a a
     * DOING_CHAR. Then put back the DONE_CHAR. It is a way to signal that some
     * activity is going on, even if the displayed length of the progress bar is
     * not increased in this update event.
     */
    private void doBlink()
    {

        final int halfBlinkmsec = BLINK_PERIOD_MSEC / 2;
        out.flush();
        if (!canBlink)
        {
            return;
        }
        carryOversFromLastBlink = 0;
        
        out.print("\b");
        out.print(DOING_CHAR);
        out.flush();
        try
        {
            Thread.sleep(halfBlinkmsec);
        }
        catch( final InterruptedException e )
        {
        }
        out.print("\b");
        out.print(DONE_CHAR);
        out.flush();
        try
        {
            Thread.sleep(halfBlinkmsec);
        }
        catch( final InterruptedException e )
        {
        }
    }

    /**
     * Actually performing the updating of this progress bar.
     * 
     * @param percent
     */
    private void doDisplayPercent(int percent)
    {

        if (percent > 100)
        {
            percent = 100;
        }
        if (percent < 0)
        {
            percent = 0;
        }
        final int nchar = barCharLen(percent) - barCharLen(lastPercent);
        // logger.debug("Progress Bar at "+percent+"% (nchar="+barCharLen(percent)+")");
        if (nchar > 0)
        {
            for (int i = 1; i <= nchar; i++)
            {
                out.print(DONE_CHAR);
                out.flush();
            }
            logger.debug("Updated the Progress bar. Now displaying percent="+percent);
        }
        else 
        {
            doBlink();
        }
        
        if (percent >= lastPercent)
        {
            lastPercent = percent;
        }
        else 
        {
            logger.warn("New percent value ("+percent
                    +") would be lower than current value ("+lastPercent +")");
        }
        lastTime = now();

        if (percent == 100)
        {
            out.print(DONE_CHAR);
            out.println(FRAME_CHAR);
            started = false;
            logger.debug("Progress bar has been completed.");
        }
    }

    private long now()
    {
        return System.currentTimeMillis();
    }


    /**
     * Simple command line test of a ProgressBar. Use 4 different lengths for
     * the same bar.
     * 
     * @param args
     */
    public static void main(final String[] args) throws Exception
    {
        Thread.sleep(2000);
        final ProgressBar b = new ProgressBar();
        b.setPrintStream(System.out);
        // try the increment method:
        double smallInc = 0.1;
        System.out.println("\nProgress bar with small increments = " + smallInc);
        b.start(80, 0, 10);
        
        while( b.getLastPercent() < 100 )
        {
            b.increment(smallInc);
        }
        smallInc = 0.00001;
        System.out.println("\nProgress bar with small increments = " + smallInc);
        b.start(80, 0, 10);
        while( b.getLastPercent() < 100 )
        {
            b.increment(smallInc);
        }
        // Try different lengths:
        for (int L = 78; L < 81; L++)
        {
            System.out.println("\nProgress bar (blinking a bit at ~ 66%) of length="
                    + L);
            b.start(L, 2, 10);
            Thread.sleep(10);
            b.displayPercent(10);
            Thread.sleep(10);
            for (int i = 11; i <= 100; i++)
            {
                Thread.sleep(10);
                b.displayPercent(i);
                if (i == 66)
                {
                    for (int j = 0; j < 100; j++)
                    {
                        // blink a bit:
                        b.displayPercent(i);
                        Thread.sleep(5);
                    }
                }
            }
        }
    }

}
