package org.eso.phase3.validator;

import java.io.File;
import java.io.IOException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.Vector;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.log4j.Logger;
import org.eso.oca.fits.DataTransportFormatHandler;
import org.eso.oca.fits.TypedHeaderCard;
import org.eso.phase3.catalog.domain.CatalogConstants;
import org.eso.phase3.domain.Constants;
import org.eso.phase3.validator.ValidatorStat.StatType;

/**
 * Parse the fits files' headers of this release to obtain the definitions of
 * category of files, provenance of products and datasets. Duplicated
 * definitions across the files are detected and reported. <b>WARNING:</b> if
 * two fits files with the same name are found, the second one is ignored. If a
 * fits file cannot be parsed, it is marked to be in error but the parsing goes
 * on. The fields are assigned using both the constructor's input and the
 * information extracted from the fits headers.
 * <ul>
 * </ul>
 * The following attributes are assigned from the release directory content: <li>
 * <code>fitsOnDisk</code>: list of fits file in this Release.</li> <li>
 * <code>fullPath</code>: mapping the file name with the File object for all the
 * files of the release directory found on disk.</li>
 * <ul>
 * </ul>
 * The following attributes are assigned from fits headers extracted information: <li>
 * <code>categoryMap</code>: category of each file in this release.</li> <li>
 * <code>provenanceMap</code>: originating files of each provenance definition.
 * (filename where the definition is found: list of ids).</li> <li>
 * <code>datasetMap</code>: associated files of each dataset definition.
 * (filename where the definition is found: list of filenames).</li>
 * 
 * @author dsforna
 */
public class ReleaseParserImp extends AbstractReleaseParserWithExtraInfo
{
    private static final Logger logger = Logger.getLogger(ReleaseParserImp.class);

    /**base directory of the release to parse.*/
    private final String releaseDir;

    /**(filename, category) as parsed from the fits headers.*/
    private final Map<String, String> categoryMap;

    /**(filename, md5sum) as parsed from the fits headers.*/
    private final Map<String, String> md5SumMap;

    /** fits filename and its failed preconditions (empty list if all the preconditions are true).*/ 
    private final Map<String, List<FailedPrecondition>> failedPreconditionMap;
    
    /**(filename, dataset components) as parsed from the fits headers.*/
    private final Map<String, Set<String>> datasetMap;

    /**(filename, provenance components) as parsed from the fits headers.*/
    private final Map<String, Set<String>> provenanceMap;
    
    /**Full paths of disk files with the same name.*/
    private final Map<String, List<String>> duplicatedFiles;

    /**(filename, error message) for the fits files that cannot be parsed.*/
    private final Map<String, String> fitsInError;

    /**(filename, file path) for fits files only.*/
    private final Map<String, String> fitsOnDisk;

    /**(filename,file path) of all the files.*/
    private final Map<String, String> fullPath;

    /**(filename, header-index) of the parsed headers (index is 0 for fits, 
     * 0 or 1 for compressed fits.) */
    private final Map<String, Integer> indexParsedHeader;
    
    /**
     * Entries (ancillary file name, science file names) defining in which science 
     * file(s) the category of an ancillary file is defined. If the category is 
     * defined in multiple science files, their names are returned comma 
     * separated in a single String.  
     */
    private final Map<String, String> categoryLocationMap; // for DFS09735

    /**Entries (filename, value of the ROWCOUNT keyword) for catalog files. 
     * ROWCOUNT should be found in the primary header of the file. */
    private final Map<String, Long> rowcountMap;
    
    /**Entries (filename, value of the NAXIS2 keyword) for catalog files.
     * NAXIS2 should be found in the (first and only) Extension header of the 
     * file.*/
    private final Map<String, Long> naxis2Map;
    
    /** Which percent shows the progress bar at the end of the parsing. */
    private int endParsingPercent;
    
    /**Progress bar shown on command line (if enabled).*/
    private ProgressBar progressBar;
    
    /**Statistic about parsing are accumulated here.*/
    private ValidatorStat stat;
    
    private boolean checkErrorFiles = false;

    /**
     * Store the input list of fits file to parse and check that they are
     * actually unique. Duplicated files are not added twice to the list of
     * files on disk (so the second copy will not be parsed) but are added to
     * the list of duplicated files: if this list is not empty the parsing will
     * end with an error even if all the parsed files are correct.
     * 
     * @param releaseDir
     *            the base of the directory tree from where to retrieve the list
     *            of fits files.
     * @throws IOException
     *            raised by a disk operation.
     */
    public ReleaseParserImp(final String releaseDir) throws IOException
    {
        if (releaseDir == null)
        {
            logger.error("Null input argument: releaseDir");
            throw new IllegalArgumentException(
                    "Null input argument: releaseDir");
        }

        final File test = new File(releaseDir);
        if (!test.isDirectory())
        {
            throw new IOException("Not a directory: " + releaseDir);
        }

        if (!test.canRead())
        {
            throw new IOException("Not a readeable directory: "
                    + releaseDir);
        }
        
        if (!test.canExecute())
        {
            throw new IOException("Cannot examine (no executable permission) directory: "
                    + releaseDir);
        }
        
        this.releaseDir = releaseDir;
        fullPath = new HashMap<String, String>();
        fitsOnDisk = new HashMap<String, String>();
        fitsInError = new HashMap<String, String>();
        duplicatedFiles = new HashMap<String, List<String>>();
        categoryMap = new HashMap<String, String>();
        provenanceMap = new HashMap<String, Set<String>>();
        datasetMap = new HashMap<String, Set<String>>();
        indexParsedHeader = new HashMap<String, Integer>();
        md5SumMap = new HashMap<String, String>();
        failedPreconditionMap = new HashMap<String, List<FailedPrecondition>>();
        categoryLocationMap = new HashMap<String, String>(); 
        rowcountMap = new ConcurrentHashMap<String, Long>();
        naxis2Map = new ConcurrentHashMap<String, Long>();
        
        // dummy progress bar and stat objects so they are never null:
        setProgressBar(new ProgressBar(), 0);
        stat = new ValidatorStat();
    }

    /* (non-Javadoc)
     * @see org.eso.phase3.util.ReleaseParser#getCategoryLocationMap()
     */
    public Map<String, String> getCategoryLocationMap()
    {
        return categoryLocationMap;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#datesetOf(java.lang.String)
     */
    @Override
    public Set<String> datesetOf(final String filename)
    {
        logger.trace("");
        if (datasetMap.containsKey(filename))
        {
            return datasetMap.get(filename);
        }
        else
        {
            return Collections.emptySet();
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#getCategory(java.lang.String)
     */
    @Override
    public String getCategory(final String fileName)
    {
        return categoryMap.get(fileName);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#getCategoryMap()
     */
    @Override
    public Map<String, String> getCategoryMap()
    {
        return categoryMap;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#geDatasetMap()
     */
    @Override
    public Map<String, Set<String>> getDatasetMap()
    {
        logger.trace("");
        return datasetMap;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#getDatasets()
     */
    @Override
    public Set<Set<String>> getDatasets()
    {
        return new HashSet<Set<String>>(datasetMap.values());
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#getFitsInErrorMap()
     */
    public Map<String, String> getFitsInErrorMap()
    {
        return fitsInError;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#getProvenanceMap()
     */
    @Override
    public Map<String, Set<String>> getProvenanceMap()
    {
        return provenanceMap;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#getRemoteFiles()
     */
    @Override
    public Set<String> getRemoteFiles()
    {
        logger.trace("");
        return Collections.emptySet();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#getRemovedFiles()
     */
    @Override
    public Set<String> getRemovedFiles()
    {
        logger.trace("");
        return Collections.emptySet();
    }

    /**
     * @param filename not used, this parser does not have any remote file. 
     * @see
     * org.eso.phase3.validator.ReleaseParser#isRemoteFile(java.lang.String)
     */
    @Override
    public boolean isRemoteFile(final String filename)
    {
        logger.trace("");
        return false;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#getLocalFilesMap()
     */
    @Override
    public Map<String, String> getLocalFilesMap()
    {
        return fullPath;
    }
    
    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#parse()
     */
    @Override
    public void parse() throws ParseException
    {
        logger.trace("");

        final double progressBarDelta = endParsingPercent - progressBar.getLastPercent();
        final double quantum = progressBarDelta/12.0;
        final double retrieveFilesInc = 4 * quantum;
        final double structParseInc = 7 * quantum;
        final double fitsInErrorInc = 1 * quantum;
        boolean duplicationFound = false;

        /* catalog files */
        int simpleCatFiles = 0;
        int mainCatFiles = 0;
        int dataCatFiles = 0;
        
        final List<String> errors = new ArrayList<String>();
        
        try {
        	retrieveReleaseDirFiles(retrieveFilesInc);
        } catch (final ParseException e) {
        	errors.add(e.toString());
        	for (int i = 0; i < e.getErrorOffset(); i++)
        		stat.add(StatType.ERROR_FITS_VALIDATION);
        } catch (final Exception e) {
        	errors.add(e.toString());
        	stat.add(StatType.ERROR_OTHER);
        }

        // These data structure are needed to log the origin in case of duplication:
        Map<String, String> categoryMapOrigin = new HashMap<String, String>();
        Map<Set<String>, String> datasetMapOrigin = new HashMap<Set<String>, String>();
        
        for (final String dup: duplicatedFiles.keySet()) {
            errors.add(dup + ": " + "duplicated on disk " + Arrays.toString(duplicatedFiles.get(dup).toArray()));
            stat.add(StatType.ERROR_DUPLICATION);
        }

        logger.info("Parsing release structure from the fits headers.");
        int parsedFiles = 0;
        Iterator<String> names = fitsOnDisk.keySet().iterator();
        while (names.hasNext()) {
            parsedFiles++;
            progressBar.increment(structParseInc/fitsOnDisk.keySet().size());
            String fileName = names.next();
            
            if (fileName.length() > Constants.MAX_FILENAME_LENGTH) {
                errors.add("Name longer than " + Constants.MAX_FILENAME_LENGTH + " [" + fileName + "]");
                stat.add(StatType.ERROR_OTHER);
            }
            
            try {
                logger.debug(
                        "Parsing the release structure held in the header of fits: "
                        + fileName);
                final FitsReleaseStructureParserImp fp = new FitsReleaseStructureParserImp(
                        new File(fitsOnDisk.get(fileName)));
               
                int parsedIndex = fp.parse();
                indexParsedHeader.put(fileName, parsedIndex); 
                failedPreconditionMap.put(fileName, fp.getFailedPreconditions());
                if (!fp.getCategoryMap().isEmpty()) {
                    for (String fname : fp.getCategoryMap().keySet()) {
                        if (categoryMapOrigin.containsKey(fname)) {
                            String previousLoc = categoryLocationMap.get(fname);
                            if (previousLoc == null)  {
                                // Even if it should never happen.
                                previousLoc = ""; 
                                logger.warn("Previous location is null for existing category.");
                            }
                            categoryLocationMap.put(fname, previousLoc + ", " + fp.getFilename());
                            String catgAlready = categoryMap.get(fname);
                            String catgInThisFits=fp.getCategoryMap().get(fname);
                            if (catgAlready.equals(catgInThisFits)) {
                                String msg= fname + ": category is defined (with the same value) "
                                		+ " in " + categoryMapOrigin.get(fname) 
                                        + " and " + fp.getFilename();
                                logger.debug(msg);
                            } else {
                                String msg= "Category of file " + fname 
                                        + " found in " + categoryMapOrigin.get(fname) 
                                        + " and " + fp.getFilename();
                                logger.error(msg);
                                errors.add(msg);
                                stat.add(StatType.ERROR_DUPLICATION);
                                duplicationFound=true;
                            }
                        } else {
                            categoryMapOrigin.put(fname, fp.getFilename());
                            String catg = fp.getCategoryMap().get(fname);
                            categoryMap.put(fname, catg);
                        	if (Util.isSimpleCatalog(catg))
                        		simpleCatFiles++;
                        	else if (Util.isBigCatalogMain(catg))
                        		mainCatFiles++;
                        	else if (Util.isBigCatalogData(catg))
                        		dataCatFiles++;
                            categoryLocationMap.put(fname, fp.getFilename());
                        }
                    }
                }

                if (!fp.getDataset().isEmpty()) {
                    // Note that if the origin of the duplication is not needed, 
                    // datasetMap can be directly used here (method containsValue)
                    datasetMap.put(fp.getFilename(), fp.getDataset());
                    if (datasetMapOrigin.containsKey(fp.getDataset())) {
                        String originalDefinitionFile = datasetMapOrigin.get(fp.getDataset());
                        String msg = "Duplicated definition of the same dataset found in "
                                + originalDefinitionFile + " and " + fp.getFilename() 
                                + ": " + Arrays.toString(fp.getDataset().toArray());
                        logger.error(msg);
                        errors.add(msg);
                        stat.add(StatType.ERROR_DUPLICATION);
                        duplicationFound=true;
                    } else {    
                        datasetMapOrigin.put(fp.getDataset(), fp.getFilename()); 
                    }
                }
                if (!fp.getProvenance().isEmpty()) {
                    provenanceMap.put(fp.getFilename(), fp.getProvenance());
                }
                
                if (!fp.getMd5sumMap().isEmpty()) {
                    // NOTE that if an md5sum is defined twice, the latter value 
                    // overrides the former. No warning or error: it can be 
                    // implemented (as for categories) but it would slow down 
                    // the parsing. 
                    md5SumMap.putAll(fp.getMd5sumMap());
                }
            } catch (final Exception e) {
                logger.error("Exception for file " + fileName+ " , " 
                        + parsedFiles + " of " + fitsOnDisk.keySet().size() 
                        + ": " + e.toString());
                fitsInError.put(fileName, "Ignoring header. " + e.getMessage());
            }
        }

        /* check catalog files consistency */
        if (simpleCatFiles > 1) {
        	String msg = "This release contains more than one file of category " + CatalogConstants.CATALOG_CATG + ", this is not allowed";
        	logger.error(msg);
        	errors.add(msg);
            stat.add(StatType.ERROR_CATALOG_VALIDATION);
        }
        
        if (mainCatFiles > 1) {
        	String msg = "This release contains more than one file of category " + CatalogConstants.BCATALOG_MAIN_CATG + ", this is not allowed";
        	logger.error(msg);
        	errors.add(msg);
        	stat.add(StatType.ERROR_CATALOG_VALIDATION);
        }
        
        if ((simpleCatFiles > 0) && (mainCatFiles > 0)) {
        	String msg = "This release contains files of category " + CatalogConstants.CATALOG_CATG + " and " + CatalogConstants.BCATALOG_MAIN_CATG + ", this is not allowed";
        	logger.error(msg);
        	errors.add(msg);
        	stat.add(StatType.ERROR_CATALOG_VALIDATION);
        }
        
        if ((simpleCatFiles > 0) && (dataCatFiles > 0)) {
        	String msg = "This release contains files of category " + CatalogConstants.CATALOG_CATG + " and " + CatalogConstants.BCATALOG_DATA_CATG + ", this is not allowed"; 
        	logger.error(msg);
        	errors.add(msg);
        	stat.add(StatType.ERROR_CATALOG_VALIDATION);
        }
        
        /* DFS11641 */
        if ((mainCatFiles == 0) && (dataCatFiles > 0)) {
        	String msg = "This release contains files of category " + CatalogConstants.BCATALOG_DATA_CATG + " but no files of category "
        		+ CatalogConstants.BCATALOG_MAIN_CATG + ", this is not allowed";
        	logger.error(msg);
        	errors.add(msg);
        	stat.add(StatType.ERROR_CATALOG_VALIDATION);
        }
        
        /* DFS11640 */
        if ((mainCatFiles == 1) && (dataCatFiles == 0)) {
        	String msg = "This release contains files of category " + CatalogConstants.BCATALOG_MAIN_CATG + " but no files of category " 
        		+ CatalogConstants.BCATALOG_DATA_CATG + ", this is not allowed";
        	logger.error(msg);
        	errors.add(msg);
        	stat.add(StatType.ERROR_CATALOG_VALIDATION);
        }
      
       	Integer tfields = null;
        
       	/* iterate over all the datasets */
      	for (Entry<String, Set<String>> entry: datasetMap.entrySet()) {
      		String mainFile = entry.getKey();
       		String category = categoryMap.get(mainFile);
       		if (category == null) 
       			continue;

       		/* check TFIELDS on catalogs */
       		if (Util.isCatalog(category)) {
       			String filePath = fitsOnDisk.get(mainFile);
       			/* the missing file error will be raised later */
       			if (filePath == null) {
       				break;
       			}
       			try {
       				DataTransportFormatHandler dtfh = ValidationUtil.allocateDTFH(filePath);
       				int catHeader = Util.findCatalogHeader(dtfh);
       				TypedHeaderCard[] cards = dtfh.getFITSCards(catHeader, new String[] { "TFIELDS" });
       				if (cards[0].isDefined()) {
       					int currentTfields = Integer.valueOf(cards[0].getValue());
       					if (tfields == null)
       						tfields = currentTfields;
       					else if (tfields != currentTfields)
       						errors.add("The value of TFIELDS is not the same in all catalog parts");
       				}
       			} catch (Exception e) {
       				/* this is ignored, because the error (missing extension) will be raised by the CatalogValidator */
       			}
       		}
       	}
      	
      	/* check for mixed ARCFILE and ORIGFILE provenance */
//      	for (String product: provenanceMap.keySet()) {
//      		boolean hasInternalReference = false;
//      		boolean hasExternalReference = false;
//			/* retrieve associated filenames */
//			Set<String> sources = provenanceMap.get(product);
//			for (String source: sources) {
//				String localSource = getLocalFilesMap().get(source);
//				boolean remoteSource = getRemoteFiles().contains(source);
//				if ((localSource == null) && !remoteSource) {					
//					/* it's an external dependency */
//					hasExternalReference = true;
//				} else {
//					/* it's an internal dependency */
//					hasInternalReference = true;
//				}
//			}
//			if (hasInternalReference && hasExternalReference) {
//				String msg = "Mixed ARCFILE and ORIGFILE provenance entries found in file [" + product + "]";
//				logger.error(msg);
//	        	errors.add(msg);
//	        	stat.add(StatType.ERROR_PROVENANCE);
//			}
//		}      	
      	
      	errors.addAll(checkForRecursiveDatasets(datasetMap, stat));
        
        if (!fitsInError.keySet().isEmpty()) {
            logger.warn(fitsInError.keySet().size() + " fits file header produced an error while being parsed for obtaining the release's structure.");
            for (final String failedFile : fitsInError.keySet()) {
                // Add the fits to the release structure, even if the parsing has 
                // failed so that they will be checked by the fits verify utility:
                progressBar.increment(fitsInErrorInc / fitsInError.keySet().size());
                errors.add(failedFile + ": " + fitsInError.get(failedFile));
                if (!categoryMap.containsKey(failedFile)) {
                    categoryMap.put(failedFile, null);
                    logger.warn("Added a null category for fits file in error:"
                            + failedFile);
                }
            }
        }
        progressBar.displayPercent(endParsingPercent);
        if (duplicationFound) {
            logger.error("Duplicated definition found while parsing the release's structure.");
        } else {
            logger.debug("No duplicated definitions in the release's structure.");
        }
        
        // Report any errors to the caller:
        if (errors.size() > 0) {
            throw new ParseException(ValidationUtil.joinListString(errors, Consts.NEWLINE), 0);
        }
        
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#provenanceOf(java.lang.String)
     */
    @Override
    public Set<String> provenanceOf(final String filename)
    {
        logger.trace("");
        if (provenanceMap.containsKey(filename)) {
            return provenanceMap.get(filename);
        } else {
            return Collections.emptySet();
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.eso.phase3.validator.ReleaseParserWithExtraInfo#setProgressBar(org
     * .eso.phase3.validator.ProgressBar, int)
     */
    @Override
    public void setProgressBar(final ProgressBar progressBar, final int endParsingPercent) {
        if (progressBar != null) {
            this.progressBar = progressBar;
        }

        if (endParsingPercent > 100) {
            this.endParsingPercent = 100;
        } else if (endParsingPercent < 0) {
            this.endParsingPercent = 0;
        } else {
            this.endParsingPercent = endParsingPercent;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.eso.phase3.validator.ReleaseParserWithExtraInfo#setStat(org.eso.phase3
     * .validator.ValidatorStat)
     */
    @Override
    public void setStat(final ValidatorStat stat)
    {
        logger.trace("");
        if (stat == null) {
            logger.error("Null input argument: stat");
            throw new IllegalArgumentException("Null input argument: stat");
        }
        this.stat = stat;
    }

    /**
     * Retrieve the list of files contained in the directory tree routed at 
     * {@link#releaseDir} and find duplicated file names. Duplicated names are 
     * listed in {@link#duplicatedFiles} 
     * @param progessBarIncrement increment of the progress bar for this operation.
     * @throws IOException
     */
    private void retrieveReleaseDirFiles(final double progessBarIncrement) throws Exception
    {
        logger.debug("Retrieving release files from directory: " + releaseDir);
        final int parsingPercentNow = progressBar.getLastPercent();
        final double quantum = progessBarIncrement / 10.0;
        progressBar.increment(quantum);
        Vector<String> names = ValidationUtil.allFilePaths(releaseDir);
        logger.info("Retrieved a total of " + names.size() 
                + " files from directory: " + releaseDir 
                + ". (Now filtering non-fits and marking duplicated files).");
        Iterator<String> it = names.iterator();
        progressBar.blink();
        List<String> invalidFiles = new ArrayList<String>();
        while (it.hasNext()) {
            progressBar.increment((8*quantum)/names.size());
            final String name = it.next();
            logger.debug("Found file on disk: " + name);
            
            final File f = new File(name);
            
            /* identify error files */
            Matcher m = Consts.errorFilePattern.matcher(f.getName());
            if (m.matches()) {
            	if (names.contains(f.getParent() + "/" + m.group(1)))
            		invalidFiles.add(m.group(1));
            	/* we have to ignore this file in the rest of the process */
            	it.remove();
            	continue;
            }
            
            // Use canonical file to make sure that there are no multiple
            // references to the same file through symbolic links. (Should 
            // multiple references of the same file be allowed, change it 
            // to getAbsoluteFile() ).
            String cName = f.getCanonicalFile().getName();

            if (fullPath.containsKey(cName)) {
                logger.debug("Found duplication for " + cName + ": "
                        + fullPath.containsKey(cName) + ", "
                        + f.getCanonicalFile().getAbsolutePath());
                stat.add(StatType.ERROR_DUPLICATION);
                List<String> paths = null;
                if (duplicatedFiles.containsKey(cName)) {
                    paths = duplicatedFiles.get(cName);
                } else {
                    paths = new ArrayList<String>();
                    paths.add(fullPath.get(cName));
                }
                paths.add(f.getCanonicalFile().getAbsolutePath());
                duplicatedFiles.put(cName, paths);
            } else {
                fullPath.put(cName, f.getCanonicalFile().getAbsolutePath());
                if (ValidationUtil.isConsideredFits(name)) {
                    fitsOnDisk.put(cName, f.getCanonicalFile().getAbsolutePath());
                }
            }
        }
        
        if (checkErrorFiles && invalidFiles.size() > 0) {
        	throw new ParseException("The following files failed FITS verification: " + Arrays.toString(invalidFiles.toArray()), invalidFiles.size());
        }
        
        for (final String cName : duplicatedFiles.keySet())
        {
            progressBar.increment(quantum/duplicatedFiles.keySet().size());
            logger.error("Duplicated file paths will be ignored for the file named "
                    + cName + " . All paths on disk:"
                    + Arrays.toString(duplicatedFiles.get(cName).toArray()));
        }
        progressBar.displayPercent(parsingPercentNow + (int)progessBarIncrement);
    }

	/* (non-Javadoc)
     * @see org.eso.phase3.validator.ReleaseParserWithExtraInfo#indexParsedHeader(java.lang.String)
     */
    @Override
    public int getIndexParsedHeader(String filename)
    {
        if (indexParsedHeader.containsKey(filename)) {
            return indexParsedHeader.get(filename);
        } else {
            return 0;
        }
    }
    

    /* (non-Javadoc)
     * @see org.eso.phase3.util.ReleaseParser#getMd5Sum(java.lang.String)
     */
    @Override
    public String getMd5Sum(String filename)
    {
        return md5SumMap.get(filename);
    }

    /* (non-Javadoc)
     * @see org.eso.phase3.util.ReleaseParser#hasChecksum(java.lang.String)
     */
    @Override
    public List<FailedPrecondition> failedPreconditions(String filename)
    {
        if (failedPreconditionMap.containsKey(filename)) {
            return failedPreconditionMap.get(filename);
        } else {
            logger.warn("List of missing checksum not found for file "+ filename);
            return Collections.emptyList();
        }
    }

	public boolean isCheckErrorFiles() {
		return checkErrorFiles;
	}

	public void setCheckErrorFiles(boolean checkErrorFiles) {
		this.checkErrorFiles = checkErrorFiles;
	}
}
