package org.eso.phase3.validator;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import org.apache.log4j.Logger;
import org.eso.phase3.validator.Util;
import org.eso.phase3.validator.ValidatorStat.StatType;

/**
 * Validate an UPDATE release. An UPDATE release is a release which takes the
 * structure of an archived release and applies changes to it. The possible
 * changes are replace/delete existing files and add new files. Only the new
 * files are available on disk. REPLACE and DELETE actions need to be explicitly
 * specified in the change file while ADD is implicit: all the disk files in the
 * release directory are considered the be ADDed to the structure.
 * 
 * @author dsforna
 * 
 */
public class ReleaseUpdateParser extends AbstractReleaseParserWithExtraInfo
{
    /** Apache Log4J logger for this class namespace. */
    private static final Logger logger = Logger.getLogger(ReleaseUpdateParser.class);

    private ValidatorStat stat;
    private ProgressBar progressBar;
    private int endParsingPercent;

    private Map<String, String> categoryMap;
    private Map<String, Set<String>> datasetMap;
    private Map<String, Set<String>> provenanceMap;

    private final String releaseDir;
    private final ReleaseParserWithExtraInfo localParser;

    private Set<String> remoteFiles;

    private final Set<String> removedFiles;

    /**
     * Store the input list of fits file to parse and check that they are
     * actually unique. Duplicated files are not added twice to the list of
     * files on disk (so the second copy will not be parsed) but are added to
     * the list of duplicated files: if this list is not empty the parsing will
     * end with an error even if all the parsed files are correct.
     * 
     * @param releaseDir
     *            the base of the directory tree from where to retrieve the list
     *            of fits files.
     * @throws IOException
     *             raised by a disk operation.
     */
    public ReleaseUpdateParser(final String releaseDir) throws IOException
    {
        localParser = new ReleaseParserImp(releaseDir);
        this.releaseDir = releaseDir;

        categoryMap = new HashMap<String, String>();
        datasetMap = new HashMap<String, Set<String>>();
        provenanceMap = new HashMap<String, Set<String>>();
        remoteFiles = new HashSet<String>();
        removedFiles = new HashSet<String>();

        // dummy progress bar and stat objects so they are never null:
        setProgressBar(new ProgressBar(), 0);
        stat = new ValidatorStat();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#datesetOf(java.lang.String)
     */
    @Override
    public Set<String> datesetOf(final String filename)
    {
        logger.trace("");
        if (datasetMap.containsKey(filename))
        {
            return datasetMap.get(filename);
        }
        else
        {
            return Collections.emptySet();
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#getCategory(java.lang.String)
     */
    @Override
    public String getCategory(final String fileName)
    {
        logger.trace("");
        return categoryMap.get(fileName);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#getFiles()
     */
    @Override
    public Map<String, String> getCategoryMap()
    {
        logger.trace("");
        return categoryMap;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#geDatasetMap()
     */
    @Override
    public Map<String, Set<String>> getDatasetMap()
    {
        logger.trace("");
        return datasetMap;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#getDatasets()
     */
    @Override
    public Set<Set<String>> getDatasets()
    {
        logger.trace("");
        return new HashSet<Set<String>>(datasetMap.values());
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#getFitsInErrorMap()
     */
    @Override
    public Map<String, String> getFitsInErrorMap()
    {
        logger.trace("");
        return localParser.getFitsInErrorMap();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#getProvenance()
     */
    @Override
    public Map<String, Set<String>> getProvenanceMap()
    {
        logger.trace("");
        return provenanceMap;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#getRemoteFiles()
     */
    @Override
    public Set<String> getRemoteFiles()
    {
        logger.trace("");
        return remoteFiles;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#getRemovedFiles()
     */
    @Override
    public Set<String> getRemovedFiles()
    {
        logger.trace("");
        return removedFiles;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.eso.phase3.util.ReleaseParser#fromOriginalRelease(java.lang.String)
     */
    @Override
    public boolean isRemoteFile(final String filename)
    {
        logger.trace("");
        return remoteFiles.contains(filename);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#getFullPaths()
     */
    @Override
    public Map<String, String> getLocalFilesMap()
    {
        logger.trace("");
        return localParser.getLocalFilesMap();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#parse()
     */
    @Override
    public void parse() throws ParseException
    {
        logger.trace("");
        int availablePercentInc = endParsingPercent-progressBar.getLastPercent();

        // Assume that the update release is locally small, so 1/3 of the time 
        // is for the local parsing and 2/3 for the rest (remote/change/merge): 
        int localPercentInc = availablePercentInc/3;
        localParser.setProgressBar(progressBar, localPercentInc);
        localParser.setStat(stat);
        logger.debug("Parse local content");
        try 
        {
            localParser.parse();
        }
        catch (ParseException e)
        {
            logger.error("Local Parser error:" + e.toString());
            throw e;
        }
        finally 
        {
            // So if the parsing of the local structure, or later on of the remote 
            // structure, raises an exception the parsed data will not be empty.
            categoryMap = localParser.getCategoryMap();
            datasetMap = localParser.getDatasetMap();
            provenanceMap = localParser.getProvenanceMap();
        }

        logger.debug("Merge local content with remote content.");
        List<String> errors = mergeContent();
        progressBar.displayPercent(endParsingPercent);
        if (!errors.isEmpty())
        {
            throw new ParseException(ValidationUtil.joinListString(errors,
                    Consts.NEWLINE), 0);
        }
        
        /* NOTE: this is executed twice: once in ReleaseParserImp and once here */
        errors.addAll(checkForRecursiveDatasets(datasetMap, stat));;
        
        logger.debug("Release content has been parsed.");
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#provenanceOf(java.lang.String)
     */
    @Override
    public Set<String> provenanceOf(final String filename)
    {
        logger.trace("");
        if (provenanceMap.containsKey(filename))
        {
            return provenanceMap.get(filename);
        }
        else
        {
            return Collections.emptySet();
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.validator.ReleaseParserWithExtraInfo#setProgressBar(
     * org.eso.phase3.validator.ProgressBar, int)
     */
    @Override
    public void setProgressBar(final ProgressBar progressBar,
            final int endParsingPercent)
    {
        logger.trace("");
        if (progressBar != null)
        {
            this.progressBar = progressBar;
        }

        if (endParsingPercent > 100)
        {
            this.endParsingPercent = 100;
        }
        else if (endParsingPercent < 0)
        {
            this.endParsingPercent = 0;
        }
        else
        {
            this.endParsingPercent = endParsingPercent;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.eso.phase3.validator.ReleaseParserWithExtraInfo#setStat(org.eso.phase3
     * .validator.ValidatorStat)
     */
    @Override
    public void setStat(final ValidatorStat stat)
    {
        logger.trace("");
        if (stat == null)
        {
            logger.error("Null input argument: stat");
            throw new IllegalArgumentException("Null input argument: stat");
        }
        this.stat = stat;
    }

    /**Apply the changes specified in {@link Consts#CHANGES_USER_FILENAME} file 
     * to the remote content of the release.
     * @param changes map of actions (filename, action) to apply to the remote 
     * content of the release.
     * @return list of errors. Empty list if no errors.
     */
    private List<String> applyChanges(final Map<String, Consts.ChangeAction> changes)
    {
        logger.trace("");
        final List<String> errors = new ArrayList<String>();
        for (final Entry<String, Consts.ChangeAction> entry : changes.entrySet())
        {
            final String filename = entry.getKey();
            final Consts.ChangeAction action = entry.getValue();
            if ((!action.equals(Consts.ChangeAction.DELETE))
                    && (!action.equals(Consts.ChangeAction.REPLACE)))
            {
                throw new IllegalArgumentException("Unknown action: "
                        + action.toString());
            }

            if (!remoteFiles.contains(filename))
            {
                final String errorMsg = action.toString()
                        + " action specified for file not present in the original release structure: "
                        + filename;
                logger.error(errorMsg);
                errors.add(errorMsg);
                stat.add(StatType.ERROR_INCONSISTENCY);
                continue;
            }

            logger.debug("Removing from original release " + filename);
            remoteFiles.remove(filename);
            removedFiles.add(filename);

            String remoteCategory = categoryMap.get(filename);
            if (remoteCategory == null) 
            {
                logger.error("Found a NULL remote category for file "+ filename);
                continue;
            }
            
            if (Util.isScience(remoteCategory) || action.equals(Consts.ChangeAction.DELETE))
            {
                logger.debug("Deleting remote category of file " + filename 
                        +" [" + remoteCategory + "]");
                categoryMap.remove(filename);
            }
            else
            {
                logger.debug("Remote category of replaced non-scientific file " 
                        + filename + " [" + remoteCategory 
                        + "] will be kept (unless redefined locally).");
            }

            if (provenanceMap.containsKey(filename))
            {
                logger.debug("Deleting existing provenance ["
                        + provenanceMap.get(filename) + "] for " + filename);
                provenanceMap.remove(filename);

            }
            if (datasetMap.containsKey(filename))
            {
                logger.debug("Deleting existing dataset ["
                        + datasetMap.get(filename) + "] for " + filename);
                datasetMap.remove(filename);
            }

            // In case of DELETE the local directory must not contain a new
            // version of the file, while in case of REPLACE the local directory 
            // must contain a new version of the file.
            if (localParser.getLocalFilesMap().containsKey(filename)
                    && action.equals(Consts.ChangeAction.DELETE))
            {
                final String errorMsg = localParser.getLocalFilesMap().get(filename)
                        + ": "
                        + Consts.ChangeAction.DELETE.toString()
                        + " action specified but the file is present on disk.";
                logger.error(errorMsg);
                errors.add(errorMsg);
                stat.add(StatType.ERROR_INCONSISTENCY);
            }
            if (!localParser.getLocalFilesMap().containsKey(filename)
                    && action.equals(Consts.ChangeAction.REPLACE))
            {
                final String errorMsg = filename
                        + ": "
                        + Consts.ChangeAction.REPLACE.toString()
                        + " action specified but the file is not present on disk. ";
                logger.error(errorMsg);
                errors.add(errorMsg);
                stat.add(StatType.ERROR_INCONSISTENCY);
            }
        }
        return errors;
    }

    /**
     * Merge the modified structure of the original release with the structure
     * parsed from disk files. The structure of the original release comes from
     * the file {@link Consts#CONTENT_ESO_FILENAME} and the changes to the
     * original structure come from the file
     * {@link Consts#CHANGES_USER_FILENAME}.
     * 
     * @return list of found errors (empty list if no error).
     * @throws ParseException
     */
    private List<String> mergeContent() throws ParseException
    {
        logger.trace("");
        double quantum = ((double)(endParsingPercent-progressBar.getLastPercent()))/ 7.0;
        
        progressBar.increment(quantum);
        progressBar.blink();
        final ReleaseParser remoteContentFileParser = new RemoteStructureParser(releaseDir);
        try {
            remoteContentFileParser.parse();
        } catch(ParseException e) {
            logger.error("Remote parser error: " + e.toString());
            // Remote parser does not know about statistics, so the update is here: 
            stat.add(StatType.ERROR_INCONSISTENCY);
            throw e;
        }
        
        List<String> errors = null;
        categoryMap = remoteContentFileParser.getCategoryMap();
        datasetMap = remoteContentFileParser.getDatasetMap();
        provenanceMap = remoteContentFileParser.getProvenanceMap();
        remoteFiles = new HashSet<String>(
                remoteContentFileParser.getCategoryMap().keySet());

        logger.debug("Remote content of the release was parsed from file "
                + Consts.CONTENT_ESO_FILENAME);
        try
        {
            progressBar.increment(quantum);
            progressBar.blink();
            final Map<String, Consts.ChangeAction> changes = parseChangesFile();
            progressBar.increment(quantum);
            progressBar.blink();
            logger.debug("User's changes were correctly parsed from "
                    + Consts.CHANGES_USER_FILENAME);
            errors = applyChanges(changes);
            logger.debug("User's changes have been applied.");
        }
        catch( final IOException e )
        {
            throw new ParseException(e.getMessage(), 0);
        }

        final Iterator<String> filenames = localParser.getCategoryMap().keySet().iterator();
        final Map<String, String> localCategoryMap = localParser.getCategoryMap();
        
        progressBar.blink();
        double quantum4 = 4*quantum;
        while( filenames.hasNext() )
        {
            progressBar.increment(quantum4/localParser.getCategoryMap().keySet().size());
            final String filename = filenames.next();
            logger.debug("Consistency check on merged content for file "+ filename);
            if (categoryMap.containsKey(filename))
            {
                String localCategory = localCategoryMap.get(filename);
                String remoteCategory= categoryMap.get(filename);
                if (removedFiles.contains(filename))
                {
                    logger.debug(filename + ": category is redefined (from:  " 
                            + remoteCategory +" to: "  + localCategory 
                            + "). It is not an error because the remote file is to be removed.");
                    categoryMap.put(filename, localCategory);
                
                }
                else if (remoteCategory.equalsIgnoreCase(localCategory))
                {
                    // Note that a remote category is never null:
                    logger.debug(filename 
                            + ": category is defined (with the same value) both remotely and locally.");
                }
                else 
                {
                    String msg = filename 
                            + ": category defined both remotely (category=" 
                            + remoteCategory + ") and locally (category=" 
                            + localCategory + ").";
                    logger.error(msg);
                    errors.add(msg);
                    stat.add(StatType.ERROR_DUPLICATION);
                }
            }
            else 
            {
                categoryMap.put(filename, localCategoryMap.get(filename));
            }

            // Errors for duplicated keys of datasetMap and provenanceMap imply 
            // duplicated files (local/remote) errors, which are handled at a 
            // higher level, during file validation. 
            final Set<String> dataset = localParser.datesetOf(filename);
            if (!dataset.isEmpty())
            {
                if (! datasetMap.containsValue(dataset))
                { 
                    datasetMap.put(filename, dataset);
                }
                else 
                {
                    String msg = filename 
                            + ": duplicated dataset. A remote file already defined this dataset: " 
                            + Arrays.toString(dataset.toArray());
                    logger.error(msg);
                    errors.add(msg);
                    stat.add(StatType.ERROR_DUPLICATION);
                }
            }
            
            final Set<String> provenance = localParser.provenanceOf(filename);
            if (!provenance.isEmpty())
            {
                provenanceMap.put(filename, provenance);
            }
        }
        return errors;
    }

    /**
     * Parse the user's changes file of an UPDATE release. Each line of the file
     * must be either an empty string or in the form: {REPLACE|DELETE} filename
     * Blanks are ignored (a line of only blank characters is considered empty).
     * In case of a parsing error this method throws an exception instead of 
     * going on with the parsing, in other words the output map of changes will 
     * be returned (and therefore applied) only if the parsed file is fully correct. 
     * 
     * @return Map<String, ChangeAction> the parsed changes in the format
     *         (filename, action)
     * @throws IOException
     *             in case of error in reading from the change file.
     * @throws ParseException
     *             in case of error in the read line.
     */
    private Map<String, Consts.ChangeAction> parseChangesFile()
            throws IOException, ParseException
    {
        logger.trace("");
        final String changeFilePathName = releaseDir + File.separator
                + Consts.CHANGES_USER_FILENAME;
        final File changesFile = new File(changeFilePathName);

        if (!changesFile.exists())
        {
            logger.info("There is not a text file with user defined changes ["
                    + changeFilePathName + "]");
            return Collections.emptyMap();
        }

        if (!changesFile.canRead())
        {
            throw new IOException("Cannot read file: " + changeFilePathName);
        }

        final InputStreamReader is = new InputStreamReader(new FileInputStream(
                changesFile));
        if (is == null)
        {
            throw new IOException("Cannot open input stream from file "
                    + changeFilePathName);
        }
        final BufferedReader br = new BufferedReader(is);
        if (br == null)
        {
            throw new IOException("Cannot read file: " + changeFilePathName);
        }

        final Map<String, Consts.ChangeAction> content = new HashMap<String, 
                Consts.ChangeAction>();
        int lineNumber = 0;
        while (true)
        {
            String l = br.readLine();
            lineNumber++;
            if (l == null)
            {
                break;
            }
            l = l.trim();
            if (l.equals(""))
            {
                continue;
            }
            final String[] c = l.split("\\s+");
            if (c.length != 2)
            {
                throw new ParseException("Invalid line in " + changeFilePathName 
                        +" - Line "+ lineNumber +": " + l, lineNumber);
            }

            final String changeString = c[0].trim();
            final String filename = c[1].trim();
            Consts.ChangeAction change = null;
            try
            {
                change = Consts.ChangeAction.valueOf(changeString);
            }
            catch( final IllegalArgumentException e )
            {
                stat.add(StatType.ERROR_INCONSISTENCY);
                throw new ParseException("Parsed an invalid action ["
                        + changeString + "] in " + changeFilePathName 
                        +" - Line "+ lineNumber +": " + l, lineNumber);
            }

            if (content.containsKey(filename))
            {
                if (content.get(filename).equals(change))
                {
                    logger.warn("Change action " + change.toString()
                            + " defined twice for " + filename);
                }
                else
                {
                    stat.add(StatType.ERROR_INCONSISTENCY);
                    throw new ParseException("Redefined action [" + change
                            + "] for " + filename
                            + " . Previous defined action: "
                            + content.get(filename) + " (line "+ lineNumber +")", 
                            lineNumber);
                }
            }
            content.put(filename, change);
        }
        return content;
    }

    /* (non-Javadoc)
     * @see org.eso.phase3.validator.ReleaseParserWithExtraInfo#indexParsedHeader(java.lang.String)
     */
    @Override
    public int getIndexParsedHeader(String filename)
    {
        logger.trace("");
        return localParser.getIndexParsedHeader(filename);
    }
    
    /* (non-Javadoc)
     * @see org.eso.phase3.util.ReleaseParser#getMd5Sum(java.lang.String)
     */
    @Override
    public String getMd5Sum(String filename)
    {
        logger.trace("");
        return localParser.getMd5Sum(filename);
    }

    /* (non-Javadoc)
     * @see org.eso.phase3.util.ReleaseParser#hasChecksum(java.lang.String)
     */
    @Override
    public List<FailedPrecondition> failedPreconditions(String filename)
    {
        logger.trace("");
        return localParser.failedPreconditions(filename);
        
    }

    /* (non-Javadoc)
     * @see org.eso.phase3.util.ReleaseParser#getCategoryLocationMap()
     */
    @Override
    public Map<String, String> getCategoryLocationMap()
    {
        logger.trace("");
        return localParser.getCategoryLocationMap();
        
    }
}
