package org.eso.phase3.validator;

import java.io.File;
import java.io.IOException;
import java.text.ParseException;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.log4j.Logger;
import org.json.JSONException;
import org.json.JSONObject;

public class RemoteStructureParser implements ReleaseParser
{
    /** Apache Log4J logger for this class namespace. */
    private static final Logger logger = Logger.getLogger(RemoteStructureParser.class);

    private Map<String, String> categoryMap;

    private Map<String, Set<String>> datasetMap;

    private Map<String, Set<String>> provenanceMap;

    private final String releaseDir;

    public RemoteStructureParser(final String releaseDir)
    {
        if (releaseDir == null)
        {
            logger.error("Null input argument: releaseDir");
            throw new IllegalArgumentException(
                    "Null input argument: releaseDir");
        }
        this.releaseDir = releaseDir;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#datesetOf(java.lang.String)
     */
    @Override
    public Set<String> datesetOf(final String filename)
    {
        logger.trace("");
        if (filename == null)
        {
            logger.error("Null input argument: filename");
            throw new IllegalArgumentException("Null input argument: filename");
        }
        if (datasetMap.containsKey(filename))
        {
            return datasetMap.get(filename);
        }
        return Collections.emptySet();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#getCategory(java.lang.String)
     */
    @Override
    public String getCategory(final String fileName)
    {
        logger.trace("");
        if (fileName == null)
        {
            logger.error("Null input argument: fileName");
            throw new IllegalArgumentException("Null input argument: fileName");
        }
        return categoryMap.get(fileName);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#getFiles()
     */
    @Override
    public Map<String, String> getCategoryMap()
    {
        logger.trace("");
        return categoryMap;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#getDatasetMap()
     */
    @Override
    public Map<String, Set<String>> getDatasetMap()
    {
        logger.trace("");
        return datasetMap;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#getDatasets()
     */
    @Override
    public Set<Set<String>> getDatasets()
    {
        logger.trace("");
        return new HashSet<Set<String>>(datasetMap.values());
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#getFitsInErrorMap()
     */
    @Override
    public Map<String, String> getFitsInErrorMap()
    {
        return Collections.emptyMap();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#getProvenance()
     */
    @Override
    public Map<String, Set<String>> getProvenanceMap()
    {
        return provenanceMap;
    }

    /**
     * For a {@link RemoteStructureParser} the remote files are all the files
     * found in the structure.
     * 
     * @see org.eso.phase3.validator.ReleaseParser#getRemoteFiles()
     */
    @Override
    public Set<String> getRemoteFiles()
    {
        logger.trace("");
        return categoryMap.keySet();
    }

    /**
     * For a {@link RemoteStructureParser} there are never removed files.
     * 
     * @see org.eso.phase3.validator.ReleaseParser#getRemovedFiles()
     */
    @Override
    public Set<String> getRemovedFiles()
    {
        logger.trace("");
        return Collections.emptySet();
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.eso.phase3.util.ReleaseParser#fromOriginalRelease(java.lang.String)
     */
    @Override
    public boolean isRemoteFile(final String filename)
    {
        logger.trace("");
        if (filename == null)
        {
            logger.error("Null input argument: filename");
            throw new IllegalArgumentException("Null input argument: filename");
        }
        return categoryMap.containsKey(filename);

    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#getFullPaths()
     */
    @Override
    public Map<String, String> getLocalFilesMap()
    {
        logger.trace("");
        throw new UnsupportedOperationException(
                "Full path is undefined for remote files.");
    }

    /* (non-Javadoc)
     * @see org.eso.phase3.util.ReleaseParser#getIndexParsedHeader(java.lang.String)
     */
    @Override
    public int getIndexParsedHeader(String filename)
    {
        logger.trace("");
        throw new UnsupportedOperationException(
                "Index of parsed header is undefined for remote files.");
    }

    /* (non-Javadoc)
     * @see org.eso.phase3.util.ReleaseParser#parse()
     */
    @Override
    public void parse() throws ParseException
    {
        try
        {
            parseContentFile();
        }
        catch( final IOException e )
        {
            throw new ParseException(e.getMessage(), 0);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eso.phase3.util.ReleaseParser#provenanceOf(java.lang.String)
     */
    @Override
    public Set<String> provenanceOf(final String filename)
    {
        logger.trace("");
        if (filename == null)
        {
            logger.error("Null input argument: filename");
            throw new IllegalArgumentException("Null input argument: filename");
        }
        if (provenanceMap.containsKey(filename))
        {
            return provenanceMap.get(filename);
        }
        return Collections.emptySet();
    }

    /**
     * Parse the remote structure from <code>CONTENT_ESO_FILENAME</code> file.
     * 
     * @throws IOException
     */
    private void parseContentFile() throws IOException
    {
        final String remoteStructureFile = releaseDir + File.separator
                + Consts.CONTENT_ESO_FILENAME;
        final File f = new File(remoteStructureFile);
        if (!f.exists())
        {
            final String msg = "Missing "+Consts.CONTENT_ESO_FILENAME 
                + " file. For a release with UPDATE modification type the file " 
                + Consts.CONTENT_ESO_FILENAME 
                + " must be present in the release base directory ("+releaseDir+")";
            logger.error(msg);
            throw new IOException(msg);
        }
        if (!f.isFile())
        {
            final String msg = remoteStructureFile + " is not a reguar file.";
            logger.error(msg);
            throw new IOException(msg);
        }
        if (!f.canRead())
        {
            final String msg = remoteStructureFile + " is not readable.";
            logger.error(msg);
            throw new IOException(msg);
        }
        logger.info("Parsing Release content @ESO from " + remoteStructureFile);

        final String allStructure = ValidationUtil.readFile(f);
        try
        {
            final JSONObject jo = new JSONObject(allStructure);
            logger.debug(Consts.CONTENT_ESO_FILENAME 
                    + " read as JSON object. Now parsing the subobjects.");
            final JSONObject joCat = (JSONObject)jo.get(
                    Consts.CONTENT_ESO_CATEGORIES_KEY);
            final JSONObject joDs = (JSONObject)jo.get(
                    Consts.CONTENT_ESO_DATASETS_KEY);
            final JSONObject joProv = (JSONObject)jo.get(
                    Consts.CONTENT_ESO_PROVENANCE_KEY);
            categoryMap = ValidationUtil.getMapStringString(joCat);
            datasetMap = ValidationUtil.getMapStringSet(joDs);
            provenanceMap = ValidationUtil.getMapStringSet(joProv);
            logger.debug("Remotely there are defined " 
                    +  categoryMap.keySet().size() + " files. " 
                    + datasetMap.keySet().size() + " datasets. "
                    + provenanceMap.keySet().size() + " provenances.");
            
        }
        catch( final JSONException e )
        {
            String msg = "Failed to parse the remote structure file "
                + remoteStructureFile + " . Error: " + e.getMessage();
            logger.error(msg);
            throw new IOException(msg);
        }
        //logger.debug("Content of category map:" + Arrays.toString(
        // categoryMap.entrySet().toArray()));
    }

    /** 
     * No need to compute md5sum for remote files, so this method always return null.
     * @see org.eso.phase3.validator.ReleaseParser#getMd5Sum(java.lang.String)
     * @param filename not used.
     */
    @Override
    public String getMd5Sum(String filename)
    {
        logger.trace("");
        return null;
        
    }

    /**
     * Preconditions are checked only on local files. 
     * @see org.eso.phase3.validator.ReleaseParser#failedPreconditions(java.lang.String)
     * @param filename not used.
     */
    @Override
    public List<FailedPrecondition> failedPreconditions(String filename)
    {
        logger.trace("");
        return Collections.emptyList();
        
    }

    /* (non-Javadoc)
     * @see org.eso.phase3.util.ReleaseParser#getCategoryLocationMap()
     */
    public Map<String, String> getCategoryLocationMap()
    {
        return Collections.emptyMap();
    }

}
