/**
 * 
 */
package org.eso.phase3.validator;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;

/**
 * Run the validation steps in single-thread mode. For backward compatibility.
 * @author dsforna
 *
 */
public class SingleThreadValidationStepRunnerManager implements
        ValidationStepRunnerManager
{
    /** Apache Log4J logger for this class namespace. */
    private static final Logger logger = Logger.getLogger(SingleThreadValidationStepRunnerManager.class);

    private boolean started = false; 
    private boolean stopped = false;

    private final List<ValidationStep> vsListIn = new ArrayList<ValidationStep>(); 
    private final List<ValidationStep> vsListOut = new ArrayList<ValidationStep>(); 
    
    private final ProgressBar progressBar;
    private final double incPerStep;
    
    public SingleThreadValidationStepRunnerManager(double incPerStep, ProgressBar progressBar)
    {
        this.incPerStep = incPerStep;
        if (progressBar == null) 
        {
            progressBar = new ProgressBar();
        }
        this.progressBar = progressBar;
    }
    
    /**
     * In a multi-threading implementation this would need to be synchronized.
     */
    private void incProgressBar() 
    {
        progressBar.increment(incPerStep);
    }
    
    private void runVs(ValidationStep vs)
    {
        vs.runValidation();
        vsListOut.add(vs);
        incProgressBar();
    }
    
    /* (non-Javadoc)
     * @see org.eso.phase3.validator.ValidationStepRunnerManager#add(org.eso.phase3.validator.ValidationStep)
     */
    @Override
    public void enqueue(ValidationStep vs)
    {
        String methodName = "SingleThreadValidationStepRunnerManager::add";
        logger.trace("");
        if (stopped){
            throw new IllegalThreadStateException(methodName 
                    + " is stopped, cannot accept new tasks");
        }
        if (started)
        {
            runVs(vs);
        }
        else 
        {
            vsListIn.add(vs);
        }
    }


    /* (non-Javadoc)
     * @see org.eso.phase3.validator.ValidationStepRunnerManager#executedSteps()
     */
    @Override
    public ValidationStep dequeue()
    {
        logger.trace("");
        if (vsListOut.size() > 0)
        {
            return vsListOut.remove(0);
        }
        else { return null; } 
    }

    /* (non-Javadoc)
     * @see org.eso.phase3.validator.ValidationStepRunnerManager#start()
     */
    @Override
    public void start()
    {
        logger.trace("");
        started = true; 

    }

    /* (non-Javadoc)
     * @see org.eso.phase3.validator.ValidationStepRunnerManager#stop()
     */
    @Override
    public void stop()
    {
        logger.trace("");
        stopped = true;
        while(!vsListIn.isEmpty())
        {
            runVs(vsListIn.remove(0));
        }
    }


    /* (non-Javadoc)
     * @see org.eso.phase3.validator.ValidationStepRunnerManager#allExecuted()
     */
    @Override
    public boolean queueIsEmpty()
    {
        logger.trace("");
        return (stopped && vsListIn.isEmpty() && vsListOut.isEmpty()); 
    }
}
